"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
import logging

import pytz
import datetime
import os

from sandbox.projects.browser.autotests.allure_parser import AllureReport
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserAutotestRun import BrowserAutotestRun, CONFIG_NAMES
from sandbox.projects.browser.autotests_qa_tools.common import (
    build_number_tuple, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsAllureReport
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.teamcity import run_teamcity_build
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import common
from sandbox import sdk2
import sandbox.common.types.client as ctc

BROWSER_ARTIFACT_URL = TEAMCITY_URL + '/app/rest/builds/id:{}/artifacts/content/brands/yandex/Yandex.{}?guest=1'
ISOLATES_URL = TEAMCITY_URL + '/app/rest/builds/id:{}/artifacts/content/mds_keys.json/?guest=1'
TIMEZONE = pytz.timezone('Europe/Moscow')


class BrowserAutotestNightlyRun(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        environments = [
            PipEnvironment('teamcity-client==4.8.2'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        browser_version = sdk2.parameters.String('Browser version', required=True)
        distribution_type = sdk2.parameters.String('Distribution type', default='beta')
        framework_branch = sdk2.parameters.String('Framework branch', required=True)
        launch_autotests_at = sdk2.parameters.Integer(
            'When trigger autotests builds', default=None,
            description='Format: unix timestamp, if empty builds will be triggered immediately after browser builds finish'
        )
        config_file = sdk2.parameters.String(
            'Predefined autotests launching config', choices=[(_, _) for _ in CONFIG_NAMES])
        check_tests_status = sdk2.parameters.Bool('Check autotests status', default=False)
        with sdk2.parameters.Group('Teamcity browser builds') as teamcity_build_group:
            browser_branch = sdk2.parameters.String('Branch to build browser', required=True)
            browser_build_type_id = sdk2.parameters.String('Browser build type id', required=True)
            isolates_build_type_id = sdk2.parameters.String('Tests build type id', required=True)
            browser_build_params = sdk2.parameters.Dict('Parameters', default={'partner-names': '@vips-beta'})
            teamcity_builds_comment = sdk2.parameters.String('Comment', default='Nightly branded beta')
            teamcity_builds_tags = sdk2.parameters.List('Tags', default=['n-beta'])
            without_fake = sdk2.parameters.Bool('Do not build fake and tests', default=False)
            sleep_time = sdk2.parameters.Integer('How often check if teamcity build finished. In minutes', default=15)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity and bb',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    class Context(sdk2.Context):
        browser_builds = []
        tests_build = None
        build_problems = []

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(self.Parameters.oauth_vault)
        )

    @property
    @common.utils.singleton
    def bitbucket_client(self):
        return bitbucket.BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                                   sdk2.Vault.data(self.Parameters.oauth_vault))

    @decorators.retries(5, delay=2, backoff=2)
    def get_latest_commit(self):
        return self.bitbucket_client.get_latest_commit(
            'stardust', 'browser', self.Parameters.browser_branch)

    # The delay is big because Teamcity checks new changes once in a while
    @decorators.retries(5, delay=120, backoff=1.2)
    def get_change(self, commit):
        change = self.teamcity_client.Change(buildType__id=self.Parameters.browser_build_type_id, version=commit)
        change.id  # force lazy client to request change
        return change

    def launch_teamcity_builds(self):
        change = self.get_change(self.get_latest_commit())
        build_browser_params = {
            'branch': self.Parameters.browser_branch,
            'build_type': self.Parameters.browser_build_type_id,
            'change': change,
            'comment': "{}\nTriggered by sandbox-{}".format(
                self.Parameters.teamcity_builds_comment,
                self.id,
            ),
            'tags': self.Parameters.teamcity_builds_tags,
            'parameters': self.Parameters.browser_build_params,
            'make_unique': True,
            'make_deps_unique': True,
        }
        self.Context.browser_builds.append(
            run_teamcity_build(self.teamcity_client, **build_browser_params).id
        )
        self.Context.tests_build = run_teamcity_build(
            self.teamcity_client,
            branch=self.Parameters.browser_branch,
            build_type=self.Parameters.isolates_build_type_id,
            change=change,
            comment=self.Parameters.teamcity_builds_comment,
            tags=self.Parameters.teamcity_builds_tags,
            make_unique=True,
            make_deps_unique=True,
        ).id

        if not self.Parameters.without_fake:
            self.Context.browser_builds.append(
                run_teamcity_build(self.teamcity_client, **build_browser_params).id
            )

        raise sdk2.WaitTask(
            BrowserWaitTeamcityBuilds(
                self,
                description='Wait browser builds for autotests in {} branch'.format(self.Parameters.browser_branch),
                notifications=self.Parameters.notifications,
                mode='WAIT_GIVEN',
                builds=' '.join(
                    str(build_id) for build_id in self.Context.browser_builds + [self.Context.tests_build]),
                oauth_vault=ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
                sleep_time=self.Parameters.sleep_time,
            ).enqueue(),
            list(Status.Group.FINISH + Status.Group.BREAK),
            True,
        )

    def check_teamcity_build(self, build_id):
        build = self.teamcity_client.builds[build_id]
        if build.status != 'SUCCESS':
            self.Context.build_problems.append("{}#{} <a href={}>build</a> failed".format(
                build.build_type.name, build_id, build.web_url
            ))
            return False
        return True

    def get_successful_builds(self, build_ids):
        res = []
        for build_id in build_ids:
            if self.check_teamcity_build(build_id):
                res.append(build_id)
        return res

    def fail_task(self, failure_reason):
        self.set_info('\n'.join(self.Context.build_problems), do_escape=False)
        raise TaskFailure(failure_reason)

    def trigger_autotests(self):
        if not self.Parameters.config_file:
            return
        browser_builds = sorted(
            [self.teamcity_client.builds[build_id] for build_id in
             self.get_successful_builds(self.Context.browser_builds)],
            key=build_number_tuple
        )
        if not browser_builds or not self.check_teamcity_build(self.Context.tests_build):
            self.fail_task("Too few teamcity builds to run autotests")
        if len(browser_builds) == 1 and not self.Parameters.without_fake:
            self.Context.build_problems.append('Launched autotests without fake browser build')
        raise sdk2.WaitTask(
            BrowserAutotestRun(
                self,
                framework_branch=self.Parameters.framework_branch,
                description=self.Parameters.description,
                config_file=self.Parameters.config_file,
                build_id=browser_builds[0].id,
                fake_build_id=browser_builds[1].id if len(browser_builds) > 1 else None,
                browser_tests_build_id=self.Context.tests_build,
                browser_version=self.Parameters.browser_version,
                tags=self.Parameters.tags
            ).enqueue(),
            list(Status.Group.FINISH + Status.Group.BREAK),
            True
        )

    def on_execute(self):
        if not self.Context.browser_builds:
            self.launch_teamcity_builds()

        with self.memoize_stage.do_not_run_to_early:
            if self.Parameters.launch_autotests_at:
                time_to_launch = datetime.datetime.fromtimestamp(self.Parameters.launch_autotests_at, TIMEZONE)
                now = datetime.datetime.now(TIMEZONE)
                if time_to_launch < now:
                    self.Context.build_problems.append('Waited too long for browser builds')
                    self.fail_task('Waited too long for browser builds, will not launch autotests')
                else:
                    raise sdk2.WaitTime((time_to_launch - now).total_seconds())

        with self.memoize_stage.trigger_autotests:
            logging.info('Launching autotests in teamcity')
            self.trigger_autotests()

        autotest_task = self.find(task_type=BrowserAutotestRun).first()

        if self.Parameters.config_file:
            if not autotest_task or autotest_task.status != Status.SUCCESS:
                self.Context.build_problems.append(
                    'Autotest <a href="{}">task</a> is in status {}'.format(
                        get_task_link(autotest_task.id),
                        "<span class='status status_{}'>{}</span>".format(
                            autotest_task.status.lower(), autotest_task.status)
                    )
                )
            if self.Parameters.check_tests_status:
                allure_report_resource = AutotestsAllureReport.find(task=autotest_task).first()
                allure_data = os.path.join(str(sdk2.ResourceData(allure_report_resource).path), 'data')
                if AllureReport(allure_data, None).failures:
                    self.Context.build_problems.append('Some tests failed')
        if self.Context.build_problems:
            self.fail_task("There were build problems")

    @sdk2.header(title='Launched builds')
    def header(self):
        res = ""
        if self.Context.browser_builds:
            res += "<p>Browser <a href={}/viewLog.html?buildId={}>build</a></p>".format(
                TEAMCITY_URL, self.Context.browser_builds[0])
            if len(self.Context.browser_builds) > 1:
                res += "<p>Fake browser <a href={}/viewLog.html?buildId={}>build</a></p>".format(
                    TEAMCITY_URL, self.Context.browser_builds[1])
        if self.Context.tests_build:
            res += "<p>Tests <a href={}/viewLog.html?buildId={}>build</a></p>".format(
                TEAMCITY_URL, self.Context.tests_build)
        if self.Context.jenkins_build_url:
            res += "<p>Autotests <a href={}>job</a></p>".format(self.Context.jenkins_build_url)
        if not res:
            res = "No builds launched"
        return res
