# -*- coding: utf-8 -*-
"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
import json
import itertools

from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserAutotestsTask import (
    BrowserAutotestsTask, TEAMCITY_AUTOTESTS_POOLS, BINARY_TEAMCITY_AUTOTESTS_POOLS, TEAMCITY_AUTOTESTS_BUILD,
    MAIN_MAC_FLAVOUR)
from sandbox.projects.browser.autotests_qa_tools.common import TEAMCITY_URL, html_link
from sandbox.projects.browser.common.git import repositories
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


NODE_NAME_MAPPING = {
    'win10-x64-clos': 'win10_x64',
    'win8-clos': 'win8_x64',
    'win7-x64-ru-clos': 'win7_x64',
    'win7-x64-uac-clos': 'win7_x64_uac',
    'win7-x86-clos': 'win7_x86',
    'osx': MAIN_MAC_FLAVOUR,
    'yosemite': 'yosemite',
    'elcapitan': 'elcapitan',
    'sierra': 'sierra',
    'mojave': 'mojave',
}

ELASTIC_TESTING_URL = 'http://browser-elastic-front-test01h.haze.yandex.net:9200'
ELASTIC_URL = 'http://elastic.browser.yandex.net:9200'
CONFIG_NAMES_TEMPLATES = [
    'all_custom_{}.json',
    'all_nightly_beta_{}.json',
    'all_nightly_canary_{}.json',
    'broteam_{}.json',
    'install_update_partners_{}.json',
    'install_update_{}.json',
    'mandatory_{}.json',
    'smoke_brands_beta_custo_{}.json',
    'smoke_brands_{}.json',
    'short_nightly_{}.json',
    'install_update_sanity_{}.json'
]
CONFIG_NAMES_CUSTOM = [
    'screenshots_with_resolutions_win.json',
    'specification_all_tests.json'
]

CONFIG_NAMES = [
    config_name.format(platform)
    for platform, config_name in
    itertools.product(['win', 'mac'], CONFIG_NAMES_TEMPLATES)
] + CONFIG_NAMES_CUSTOM


class BrowserAutotestRun(BrowserAutotestsTask):
    class Requirements(BrowserAutotestsTask.Requirements):
        environments = [
            PipEnvironment('attrdict'),
            PipEnvironment('jsonschema==2.5.1'),
        ] + BrowserAutotestsTask.Requirements.environments.default

    class Parameters(BrowserAutotestsTask.Parameters):
        config_file = sdk2.parameters.String(
            'Predefined autotests launching config', required=True,
            choices=[(_, _) for _ in CONFIG_NAMES + ['custom']])
        with config_file.value['custom']:
            config = sdk2.parameters.JSON('Launching config')

        browser_version = sdk2.parameters.String('Browser version', required=True,
                                                 description='For example "17.6.0"')

        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        fake_build_id = sdk2.parameters.Integer(
            'Fake browser build id',
            description='Id of fake browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        browser_tests_build_id = sdk2.parameters.Integer(
            'Tests build id',
            description='Id of teamcity build with browser tests '
                        '(<a href="{}/project.html?projectId=Browser_Tests_BuildTests">'
                        'Browser_Tests_BuildTests</a>)'.format(TEAMCITY_URL))
        distribution_type = sdk2.parameters.String(
            'Distribution type', default='beta',
            choices=[(_, _) for _ in ['beta', 'canary', 'stable/']]
        )
        number_of_repetitions = sdk2.parameters.Integer(
            'Number of repetitions TASK',
            description=u'Количество повторов таски. Используется только для повторяемых ночных сборок')

    def repo_path(self, *args):
        return str(self.path('browser-test-framework', *args))

    @property
    def config_file_name_for_statistics(self):
        return self.Parameters.config_file

    def get_launch_config_data(self):
        from jsonschema import validate

        repo = repositories.Autotest.browser_test_framework(filter_branches=False)
        # FIXME update_cache is used here, because without it cloning branch fails on branch, which is a suffix of
        # another branch. e.g. cloning feature/19.10 fails if branch bundle/feature/19.10 exists.
        repo.update_cache_repo()
        repo.clone(self.repo_path(), self.real_framework_branch)
        if self.Parameters.config:
            config = self.Parameters.config
            schema_path = self.repo_path('launch_configs', 'schemas', 'config_schema.json')
            with open(schema_path) as json_data:
                schema = json.load(json_data)
                validate(config, schema)
        else:
            with open(self.repo_path('launch_configs', self.Parameters.config_file)) as file:
                config = json.load(file)

        return config

    def get_teamcity_builds_to_launch(self):
        from attrdict import AttrDict

        res = {}
        config = self.get_launch_config_data()
        common_params = {
            'browser_tests_build_id': self.Parameters.browser_tests_build_id,
            'build_id': self.Parameters.build_id,
            'fake_build_id': self.Parameters.fake_build_id,
        }

        for i, pack in enumerate(config['packs']):
            if 'launch_arguments' in pack:
                if 'mergeArray' in pack['launch_arguments'] and pack['launch_arguments']['mergeArray'] is True:
                    pack['launch_arguments'] = AttrDict(config['launch_arguments']) + pack['launch_arguments']
            else:
                pack['launch_arguments'] = config['launch_arguments']
            ya_settings = ' '.join(
                '--ya-settings={}:{}'.format(k, v) for k, v in pack['launch_arguments']['ya-settings'].iteritems()
                if not v.startswith('$'))
            ya_test_data = ' '.join(
                '--ya-test-data={}:{}'.format(k, v) for k, v in pack['launch_arguments']['ya-test-data'].iteritems()
                if not v.startswith('$'))
            teamcity_params = dict(common_params)
            teamcity_params['additional_arguments'] = '{} {} {} {}'.format(
                self.Parameters.default_tests_launch_arguments,
                pack['launch_arguments'].get("tests_launch_arguments", ""),
                ya_settings,
                ya_test_data)
            teamcity_params['tests_folder'] = pack['specify_by']['path']
            if 'mark' in pack['specify_by']:
                teamcity_params['marks_expression'] = pack['specify_by']['mark']
            if 'key' in pack['specify_by']:
                teamcity_params['test_names_expression'] = pack['specify_by']['key']
            teamcity_params.update(
                {k: v for k, v in pack['launch_arguments'].iteritems() if v and not isinstance(v, dict)})

            teamcity_params["autotests_pack_name"] = pack['pack_name']
            agent_pools = (BINARY_TEAMCITY_AUTOTESTS_POOLS
                           if (teamcity_params['tests_folder'] == 'tests_binary'
                               and not self.Parameters.use_test_pools)
                           else TEAMCITY_AUTOTESTS_POOLS)
            if NODE_NAME_MAPPING.get(pack['node_label']):
                res["pack_{}_{}".format(i, pack['pack_name'])] = (
                    TEAMCITY_AUTOTESTS_BUILD["win" if "win" in pack['node_label'] else "mac"],
                    agent_pools[NODE_NAME_MAPPING[pack['node_label']]],
                    teamcity_params)

        return res

    def on_finish(self, prev_status, status):

        if self.Parameters.number_of_repetitions > 0:
            # repeat task
            params = {_p[0]: _p[1] for _p in self.Parameters}
            params['number_of_repetitions'] -= 1

            tags = params.setdefault("tags", [])
            if "REPEAT" not in tags:
                tags.append("REPEAT")

            repeat_task = BrowserAutotestRun(
                self,
                **params
            ).enqueue()
            self.set_info(
                u'Запущен повтор таски: {}'.format(html_link(get_task_link(repeat_task.id))),
                do_escape=False)
