"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
import textwrap

from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.BrowserAutotestsTask import (
    BrowserAutotestsTask, TEAMCITY_AUTOTESTS_POOLS, BINARY_TEAMCITY_AUTOTESTS_POOLS,
    TEAMCITY_AUTOTESTS_BUILD, MAIN_MAC_FLAVOUR)
from sandbox.projects.browser.autotests_qa_tools.common import TEAMCITY_URL
from sandbox import sdk2

ADDITIONAL_PARAMS_DEFAULT = {
    'disable_auto_update': False,
    'skip_binary_tests_from_browser_blacklists': True,
    'enable_blacklists': True
}


class BrowserAutotestRunCustom(BrowserAutotestsTask):
    """
       Instruction is <a href="https://wiki.yandex-team.ru/avtomatizacija-testirovanija-desktopnogo-brauzera/t1/zapusk-proizvolnogo-nabora-testov/">here</a>
    """

    class Parameters(BrowserAutotestsTask.Parameters):
        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        fake_build_id = sdk2.parameters.Integer(
            'Fake browser build id',
            description='Id of fake browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        browser_tests_build_id = sdk2.parameters.Integer(
            'Tests build id',
            description='Id of teamcity build with browser tests '
                        '(<a href="{}/project.html?projectId=Browser_Tests_BuildTests">'
                        'Browser_Tests_BuildTests</a>)'.format(TEAMCITY_URL))
        screenshots_tests_config_url = sdk2.parameters.String(
            'Screenshots tests config url',
            description='Json file url with test launch parameters. (optional parameter for screenshots tests '
                        '(<a href="https://wiki.yandex-team.ru/avtomatizacija-testirovanija-desktopnogo-brauzera/t1/'
                        'zapusk-instrumenta-skrinshotov-tablo-na-raznyx-razreshenijax/">see what is it</a>)',
            default='')

        browser_version = sdk2.parameters.String('Browser version', description='For example "17.6.0"')
        build_type = sdk2.parameters.String('Build type', required=True, choices=[(_, _) for _ in ['beta', 'canary']],
                                            default='beta')
        brands = sdk2.parameters.List('Brands', description='Most tests work only with brand "yandex"',
                                      default=['yandex'], required=True)

        with sdk2.parameters.CheckGroup('Tests folders') as tests_folders:
            tests_folders.values.tests = tests_folders.Value('tests', checked=True)
            tests_folders.values.tests_v20 = tests_folders.Value('tests_v20', checked=True)
            tests_folders.values.tests_binary = 'tests_binary'

        tests_collect_by = sdk2.parameters.String('How to collect tests', description='By packs or by tests\' names',
                                                  choices=[(_, _) for _ in ['test_names', 'packs']],
                                                  required=True, default='packs')
        test_collect_condition = sdk2.parameters.String(
            'Test collect condition',
            description=textwrap.dedent(
                '''
                Pytest-style rules for tests collection divided by comma.<br/><br/>
                For example:<br/>
                By test names:<br/>
                <code>test_mandatory_launch or test_tablo,test_auto_update</code>:<br/>
                Will make 2 packs from corresponding tests
                <br/><br/>
                By packs:<br/>
                <code>updater and part1</code>:<br/>
                Will make one pack, from suitable marks and folders
                '''
            ),
            default=''
        )

        test_cases_filter = sdk2.parameters.String(
            'Test cases filter',
            description="Comma-separated list of test cases",
            default=''
        )

        with sdk2.parameters.CheckGroup('Platforms') as tests_launch_platforms:
            tests_launch_platforms.values.win7_x64 = tests_launch_platforms.Value('win7-x64', checked=True)
            tests_launch_platforms.values.win7_x86 = 'win7-x86'
            tests_launch_platforms.values.win8_x64 = 'win8-x64'
            tests_launch_platforms.values.win10_x64 = 'win10-x64'
            tests_launch_platforms.values.mac = 'mac'
            tests_launch_platforms.values.win7_x64_uac = 'win7-x64-uac (only for uac tests)'

        with sdk2.parameters.CheckGroup('Tests additional params') as tests_additional_params:
            for param, checked in ADDITIONAL_PARAMS_DEFAULT.iteritems():
                setattr(
                    tests_additional_params.values, param,
                    tests_additional_params.Value(param, checked=checked)
                )

    def get_teamcity_builds_to_launch(self):
        res = {}
        launch_on_win = any('win' in platform for platform in self.Parameters.tests_launch_platforms)
        launch_on_mac = any('win' not in platform for platform in self.Parameters.tests_launch_platforms)
        if launch_on_win and launch_on_mac:
            raise TaskFailure("Can't run autotests on both win and mac")
        teamcity_platform = 'win' if launch_on_win else 'mac'
        for flavour in self.Parameters.tests_launch_platforms:
            for test_folder in self.Parameters.tests_folders:
                for brand in self.Parameters.brands:
                    for test_expression in self.Parameters.test_collect_condition.split(','):
                        test_expression = test_expression.strip()
                        run_name = '{}_{}_{}_{}'.format(flavour, brand, test_folder, test_expression.replace(' ', '_'))
                        params = {
                            'browser_tests_build_id': self.Parameters.browser_tests_build_id,
                            'fake_build_id': self.Parameters.fake_build_id,
                            'build_id': self.Parameters.build_id,
                            'build_type': self.Parameters.build_type,
                            'tests_folder': test_folder,
                            'additional_arguments': "{} {}".format(
                                self.Parameters.default_tests_launch_arguments,
                                ' '.join(['--ya-settings=disable_failure_recording:No' +
                                          (' --ya-settings=use_20_framework:Yes' if test_folder == 'tests_v20' else ''),
                                          '--ya-test-data=mail_extension_version:1.0.35',
                                          '--ya-settings=brand_name:{}'.format(brand)] +
                                         ['--ya-settings={}:{}'.format(setting,
                                                                       'Yes' if setting in self.Parameters.tests_additional_params else 'No')
                                          for setting in ADDITIONAL_PARAMS_DEFAULT.keys()])
                                )
                        }
                        if self.Parameters.tests_collect_by == 'packs':
                            params['marks_expression'] = test_expression
                        else:
                            params['test_names_expression'] = test_expression
                        if self.Parameters.browser_version:
                            params['browser_version'] = self.Parameters.browser_version
                        if self.Parameters.screenshots_tests_config_url:
                            params['additional_arguments'] += " --ya-urls=screenshots_tests_config:{}".format(
                                self.Parameters.screenshots_tests_config_url)
                        test_cases_filter = self.Parameters.test_cases_filter.replace(" ", "")
                        if test_cases_filter:
                            params['additional_arguments'] += " --ya-testcase-filter={}".format(
                                test_cases_filter)
                        params['autotests_pack_name'] = "custom_{}".format(run_name)
                        agent_pools = (BINARY_TEAMCITY_AUTOTESTS_POOLS
                                       if (params['tests_folder'] == 'tests_binary'
                                           and not self.Parameters.use_test_pools)
                                       else TEAMCITY_AUTOTESTS_POOLS)
                        res[run_name] = (
                            TEAMCITY_AUTOTESTS_BUILD[teamcity_platform],
                            agent_pools[flavour if flavour != 'mac' else MAIN_MAC_FLAVOUR],
                            params)
        return res
