"""
Author: Nikolay v Isaev <nik-isaev@yandex-team.ru>
"""
import logging
import re
import os
import yaml

import sandbox
from sandbox.projects.browser.autotests.classes.autotests_bundle import get_isolate_autotest_bundle
from sandbox.projects.browser.autotests.classes.autotests_result import BtrReport, TestStatuses
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import (
    AutotestsReportResource, AutotestsScreenshotsDiff, AutotestsTestResultsArchives)
from sandbox import common
from sandbox import sdk2
from sandbox.common.types.task import Status
import sandbox.common.types.client as ctc
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.sandboxsdk.environments import PipEnvironment

PROJECT = 'stardust'
REPO = 'browser'
LAUNCHING_PLATFORMS = {
    'win': [
        'win_7_x64',
        # 'win_7_x86',
        # 'win_8_x64',
        'win_10_x64',
    ],
    'mac': ['mac']
}
CONFIGS_PATH = "src/build/yandex/autotests/binary/configs"
BINARY_MANDATORY_TESTCASE_ID = "broinfra_fake-1"


def _get_config_content(git_cli, commit, file_path):
    try:
        content = git_cli.show(':'.join((commit, file_path)))
    except:
        content = "{}"
    return yaml.load(content.decode('utf-8'))


def _get_changed_testcases(old_config, new_config):

    result = set()
    for case_id, new_tests in new_config.get("test_cases", {}).iteritems():
        if new_tests != old_config.get("test_cases", {}).get(case_id):
            result.add(case_id)

    old_blacklist = old_config.get("blacklists", {}) or {}
    old_blacklisted_cases = set(old_blacklist.keys())
    new_blacklist = new_config.get("blacklists", {}) or {}
    new_blacklisted_cases = set(new_blacklist.keys())

    result.update(
        old_blacklisted_cases.symmetric_difference(new_blacklisted_cases))
    return result


def get_changed_binary_testcases(repo_root):

    git_cli = GitCli(repo_root)
    before_changes = git_cli.merge_base('HEAD^1', 'HEAD^2').strip()
    after_changes = git_cli.rev_parse('HEAD^2').strip()

    changed_files = git_cli.diff(
        before_changes, after_changes,
        '--name-only', '--no-renames').splitlines()

    changed_testcases = set()
    files_regexp = r'^{}.+\.yaml$'.format(CONFIGS_PATH)
    for changed_file in changed_files:
        if not re.match(files_regexp, changed_file):
            continue

        _file_path = os.path.relpath(os.path.join(repo_root, changed_file), repo_root)
        changed_testcases.update(
            _get_changed_testcases(
                _get_config_content(git_cli, before_changes, _file_path),
                _get_config_content(git_cli, after_changes, _file_path)))

    return changed_testcases


class BrowserAutotestRunPr(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        teamcity_build_id = sdk2.parameters.Integer('Teamcity build id', required=True)
        branch = sdk2.parameters.String('Pull request branch', required=True)
        platform = sdk2.parameters.String('Platform', choices=[(_, _) for _ in ['win', 'mac']], default='win')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    class Requirements(sdk2.Requirements):

        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [PipEnvironment('teamcity-client==3.0.0'),
                        PipEnvironment('PyYAML', version='3.11'),
                        PipEnvironment('junit-xml', version='1.9')]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    @property
    @common.utils.singleton
    def build_with_isolates(self):
        return self.clients.teamcity.Build(id=self.Parameters.teamcity_build_id).snapshot_dependencies[0]

    @property
    @common.utils.singleton
    def framework_branch(self):
        _version_file_path = "{}/framework_version".format(CONFIGS_PATH)
        with open(self.repo_path(*_version_file_path.split("/")), "r") as _version_file:
            version = _version_file.read()
        return version

    def repo_path(self, *args):
        logging.info('Got path {}'.format(str(self.path('browser', *args))))
        return str(self.path('browser', *args))

    @property
    def child(self):
        return self.find().first()

    def on_execute(self):
        if not self.child:
            pr_id = self.Parameters.branch.split('/')[2]
            description = 'Autotests for <a href={}>PR#{}'.format(
                '{}projects/{}/repos/{}/pull-requests/{}/overview'.format(DEFAULT_BITBUCKET_URL, PROJECT, REPO,
                                                                          pr_id),
                pr_id)
            self.Parameters.description = description

            repositories.Stardust.browser(filter_branches=False).clone(
                self.repo_path(),
                self.Parameters.branch,
                sparse_checkout_paths=[CONFIGS_PATH])

            changed_testcases = get_changed_binary_testcases(self.repo_path())
            changed_testcases.add(BINARY_MANDATORY_TESTCASE_ID)

            bundle = get_isolate_autotest_bundle(self.build_with_isolates.id, self.clients)

            pr = self.clients.bitbucket.get_pr(PROJECT, REPO, pr_id)
            pixel_diff_pr_source_branch = pr.fromRef['id']
            if pixel_diff_pr_source_branch.startswith('refs/heads/'):
                pixel_diff_pr_source_branch = pixel_diff_pr_source_branch.split('refs/heads/', 1)[1]

            run_autotests = RunBrowserAutotests(
                self,
                description=description,
                launch_config={_p: list(changed_testcases) for _p in LAUNCHING_PLATFORMS[self.Parameters.platform]},
                browser_tests_build_id=self.build_with_isolates.id,
                branch=bundle.browser_branch,
                commit=bundle.browser_commit,
                ignore_blacklists=False,
                pixel_diff_pr_source_branch=pixel_diff_pr_source_branch,
            )
            raise sdk2.WaitTask(
                run_autotests.enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True
            )

        with TeamcityArtifactsContext(self.path('artifacts')) as tac:
            self.publish_junit_reports(tac.logger)
            self.publish_pixel_diff_if_exists(tac.logger)
            self.publish_tests_results_if_exits(tac.logger)

    def publish_junit_reports(self, logger):
        from junit_xml import TestSuite

        junits_path = str(self.path('junit_reports'))
        os.mkdir(junits_path)
        resource = AutotestsReportResource.find(task=self.child).first()
        btr_report = BtrReport(str(sdk2.ResourceData(resource).path), self.clients)
        xml_results = btr_2_junit_xml_string(btr_report._data)
        for platform, _ts in xml_results.iteritems():
            report_path = os.path.join(junits_path, '{}_junit.xml'.format(platform))
            with open(report_path, "w") as _f:
                TestSuite.to_file(_f, _ts, prettyprint=False)

            logger.info("##teamcity[importData path='{}' type='junit']".format(report_path))

    def publish_pixel_diff_if_exists(self, logger):
        resource = AutotestsScreenshotsDiff.find(task=self.child).first()
        if not resource:
            return None
        pixel_diff_path = str(sdk2.ResourceData(resource).path)
        logger.info("##teamcity[publishArtifacts '{} => pixel_diff.zip']".format(pixel_diff_path))

    def publish_tests_results_if_exits(self, logger):
        resource = AutotestsTestResultsArchives.find(task=self.child).first()
        if not resource:
            return None

        tests_results_dir = str(sdk2.ResourceData(resource).path)
        for archive in os.listdir(tests_results_dir):
            archive_path = os.path.join(tests_results_dir, archive)
            logger.info("##teamcity[publishArtifacts '{}']".format(archive_path))


def btr_2_junit_xml_string(btr_report_data):
    from junit_xml import TestSuite, TestCase

    result = {}
    for platform, data in btr_report_data.iteritems():
        _tests = []
        for build_id, cases in data.iteritems():
            for case_id, tests in cases.iteritems():
                for test in tests:
                    _case = TestCase(
                        name=u"{}.{}.{}".format(case_id, test['binary'], test['name']),
                        log=test.get("log", "")
                    )
                    if test['status'] not in TestStatuses.PASSED.value:
                        if test['status'] in TestStatuses.SKIPPED.value:
                            _case.add_skipped_info("skipped")
                        else:
                            _case.add_failure_info("Test status: {} not success".format(test['status']))
                    _tests.append(_case)

        result[platform] = [TestSuite("All {} tests".format(platform), _tests)]
    return result
