# -*- coding: utf-8 -*-
"""
Author: Nikolay v Isaev <nik-isaev@yandex-team.ru>
"""
import logging
import os

import sandbox
from sandbox.projects.browser.autotests.classes.autotests_bundle import get_isolate_autotest_bundle
from sandbox.projects.browser.autotests.classes.autotests_result import BtrReport, TestStatuses
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import (
    AutotestsReportResource, AutotestsScreenshotsDiff, AutotestsTestResultsArchives)

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.task import Status
import sandbox.common.types.client as ctc
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.sandboxsdk.environments import PipEnvironment

PROJECT = 'stardust'
REPO = 'browser'
LAUNCHING_PLATFORMS = [
    'win_7_x64',
    'win_10_x64',
    'mac',
]


class BrowserAutotestRunPrBinaryAutotests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        teamcity_build_id = sdk2.parameters.Integer('Teamcity build id', required=True)
        branch = sdk2.parameters.String('Pull request branch', required=True)
        platform = sdk2.parameters.String('Platform', choices=[(_, _) for _ in ['win', 'mac']], default='win')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    class Requirements(sdk2.Requirements):

        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [PipEnvironment('teamcity-client==3.0.0'),
                        PipEnvironment('PyYAML', version='3.11'),
                        PipEnvironment('junit-xml', version='1.9')]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    @property
    @common.utils.singleton
    def build_with_isolates(self):
        return self.clients.teamcity.Build(id=self.Parameters.teamcity_build_id).snapshot_dependencies[0]

    def repo_path(self, *args):
        logging.info('Got path {}'.format(str(self.path('browser', *args))))
        return str(self.path('browser', *args))

    @property
    def child(self):
        return self.find().first()

    def on_execute(self):
        with self.memoize_stage.run_autotests:
            pr_id = self.Parameters.branch.split('/')[2]
            description = 'Autotests for <a href={}>PR#{}'.format(
                '{}projects/{}/repos/{}/pull-requests/{}/overview'.format(DEFAULT_BITBUCKET_URL, PROJECT, REPO,
                                                                          pr_id),
                pr_id)
            self.Parameters.description = description
            bundle = get_isolate_autotest_bundle(self.build_with_isolates.id, self.clients)
            _changed_config = bundle.get_launch_config_changed_cases()
            launch_config = {}
            for _platform in _changed_config:
                if _platform in LAUNCHING_PLATFORMS:
                    launch_config[_platform] = _changed_config[_platform]
            if not launch_config:
                self.set_info(u'Нет измененных тесткейсов. Ничего не запущено')
                return None

            pr = self.clients.bitbucket.get_pr(PROJECT, REPO, pr_id)
            pixel_diff_pr_source_branch = pr.fromRef['id']
            if pixel_diff_pr_source_branch.startswith('refs/heads/'):
                pixel_diff_pr_source_branch = pixel_diff_pr_source_branch.split('refs/heads/', 1)[1]

            run_autotests = RunBrowserAutotests(
                self,
                description=description,
                launch_config=launch_config,
                browser_tests_build_id=self.build_with_isolates.id,
                branch=bundle.browser_branch,
                commit=bundle.browser_commit,
                ignore_blacklists=False,
                pixel_diff_pr_source_branch=pixel_diff_pr_source_branch,
            )
            raise sdk2.WaitTask(
                run_autotests.enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True
            )

        if self.child:
            with TeamcityArtifactsContext(self.path('artifacts')) as tac:
                self.publish_junit_reports(tac.logger)
                self.publish_pixel_diff_if_exists(tac.logger)
                self.publish_tests_results_if_exits(tac.logger)

    def publish_junit_reports(self, logger):
        from junit_xml import TestSuite

        junits_path = str(self.path('junit_reports'))
        os.mkdir(junits_path)
        resource = AutotestsReportResource.find(task=self.child).first()
        btr_report = BtrReport(str(sdk2.ResourceData(resource).path), self.clients)
        xml_results = btr_2_junit_xml_string(btr_report._data)
        for platform, _ts in xml_results.iteritems():
            report_path = os.path.join(junits_path, '{}_junit.xml'.format(platform))
            with open(report_path, "w") as _f:
                TestSuite.to_file(_f, _ts, prettyprint=False)

            logger.info("##teamcity[importData path='{}' type='junit']".format(report_path))

    def publish_pixel_diff_if_exists(self, logger):
        resource = AutotestsScreenshotsDiff.find(task=self.child).first()
        if not resource:
            return None
        pixel_diff_path = str(sdk2.ResourceData(resource).path)
        logger.info("##teamcity[publishArtifacts '{} => pixel_diff.zip']".format(pixel_diff_path))

    def publish_tests_results_if_exits(self, logger):
        resource = AutotestsTestResultsArchives.find(task=self.child).first()
        if not resource:
            return None

        tests_results_dir = str(sdk2.ResourceData(resource).path)
        for archive in os.listdir(tests_results_dir):
            archive_path = os.path.join(tests_results_dir, archive)
            logger.info("##teamcity[publishArtifacts '{}']".format(archive_path))


def btr_2_junit_xml_string(btr_report_data):
    from junit_xml import TestSuite, TestCase

    result = {}
    for platform, data in btr_report_data.iteritems():
        _tests = []
        for build_id, cases in data.iteritems():
            for case_id, tests in cases.iteritems():
                for test in tests:
                    _case = TestCase(
                        name=u"{}.{}.{}".format(case_id, test['binary'], test['name']),
                        log=test.get("log", "")
                    )
                    if test['status'] not in TestStatuses.PASSED.value:
                        if test['status'] in TestStatuses.SKIPPED.value:
                            _case.add_skipped_info("skipped")
                        else:
                            _case.add_failure_info("Test status: {} not success".format(test['status']))
                    _tests.append(_case)

        result[platform] = [TestSuite("All {} tests".format(platform), _tests)]
    return result
