"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.BrowserAutotestRun import CONFIG_NAMES, BrowserAutotestRun
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport.const import TESTSUITES_WIN, TESTSUITES_MAC
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport.Task import BrowserAutotestTestpalmExport
from sandbox.projects.browser.autotests_qa_tools.common import (
    get_browser_version_number, get_framework_branch, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsAllureReport, AutotestsAllureData
from sandbox import sdk2, common
from sandbox.common.types.task import Status
from sandbox.common.types import notification as ctn
from sandbox.common.types import resource as ctr
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment


class BrowserAutotestRunRegression(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        environments = [PipEnvironment('teamcity-client==3.0.0')]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        config_file = sdk2.parameters.String(
            'Predefined autotests launching config', required=True,
            choices=[(_, _) for _ in CONFIG_NAMES])
        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        fake_build_id = sdk2.parameters.Integer(
            'Fake browser build id',
            description='Id of fake browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))
        browser_tests_build_id = sdk2.parameters.Integer(
            'Tests build id',
            description='Id of teamcity build with browser tests '
                        '(<a href="{}/project.html?projectId=Browser_Tests_BuildTests">'
                        'Browser_Tests_BuildTests</a>)'.format(TEAMCITY_URL))
        regression_issue = sdk2.parameters.String('Regression issue',
                                                  description='Leave blank to skip exporting to testpalm')
        group_issues = sdk2.parameters.Dict('Groups issues')
        after_autotests_suites = sdk2.parameters.JSON(
            'After autotests suites',
            default={
                'win': TESTSUITES_WIN,
                'mac': TESTSUITES_MAC,
            })

    def on_save(self):
        if not any([n.recipients == ['browser-autotests-bots'] for n in self.Parameters.notifications]):
            self.Parameters.notifications = list(self.Parameters.notifications) + [
                sdk2.Notification(
                    [Status.FAILURE, Status.Group.BREAK],
                    ['browser-autotests-bots'],
                    ctn.Transport.EMAIL
                ),
                sdk2.Notification(
                    [Status.FAILURE, Status.SUCCESS, Status.NO_RES, Status.EXCEPTION, Status.TIMEOUT, Status.EXPIRED],
                    [self.author],
                    ctn.Transport.EMAIL
                )
            ]

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT),
        )

    def on_execute(self):
        with self.memoize_stage.run_autotests:
            browser_build = self.teamcity_client.builds[self.Parameters.build_id]
            self.Context.browser_version = get_browser_version_number(browser_build)
            self.Context.framework_branch = get_framework_branch(browser_build)
            raise sdk2.WaitTask(
                BrowserAutotestRun(
                    self,
                    description='Release autotests',
                    framework_branch=self.Context.framework_branch,
                    build_id=self.Parameters.build_id,
                    fake_build_id=self.Parameters.fake_build_id,
                    browser_tests_build_id=self.Parameters.browser_tests_build_id,
                    config_file=self.Parameters.config_file,
                    browser_version=self.Context.browser_version,
                ).enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )
        child = self.find().first()
        with self.memoize_stage.publish_allure_report:
            self.set_info('REPORT: <a href="{}">{}</a>'.format(child.Context.report_url, child.Context.report_url),
                          False)
        with self.memoize_stage.export_to_testpalm:
            if self.Parameters.regression_issue:
                allure_report = AutotestsAllureData.find(task=child, state=ctr.State.READY).first()
                if not allure_report:
                    allure_report = AutotestsAllureReport.find(task=child).first()
                raise sdk2.WaitTask(
                    BrowserAutotestTestpalmExport(
                        self,
                        description='Release autotests',
                        allure_report=allure_report,
                        regression_issue=self.Parameters.regression_issue,
                        platform='win' if 'win' in self.Parameters.config_file else 'mac',
                        browser_version=self.Context.browser_version,
                        build_url='{}/app/rest/builds/id:{}/artifacts/content/brands/yandex/Yandex.{}?guest=1'.format(
                            TEAMCITY_URL, self.Parameters.build_id,
                            'exe' if 'win' in self.Parameters.config_file else 'dmg'
                        ),
                        group_issues=self.Parameters.group_issues,
                        after_autotests_suites=self.Parameters.after_autotests_suites
                    ).enqueue(),
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    True,
                )
        for child in self.find():
            if child.status != Status.SUCCESS:
                raise TaskFailure('Child task {} is not successful'.format(child.type))
