"""
Author: Nikolay V. Isaev <nik-isaev@yandex-team.ru>
"""
import json

from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.BrowserAutotestRunCustom import BrowserAutotestRunCustom
from sandbox.projects.browser.autotests_qa_tools.common import (
    get_browser_version_number, get_framework_branch, check_browser_build, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL)
from sandbox import sdk2, common
from sandbox.common.types.task import Status
from sandbox.common.types import notification as ctn
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment


SCREENSHOTS_TEST_CONFIG_SCHEMA = {
    "$schema": "http://json-schema.org/draft-04/schema#",
    'type': 'array',
    'items': {
        'type': 'object',
        'minProperties': 1,
        'additionalProperties': False,
        'properties': {
            "test": {
                'type': 'string',
            },
            "params": {
                'type': 'object',
                'minProperties': 1,
                'additionalProperties': False,
                'properties': {
                    "tiles-count": {
                        'type': 'integer',
                        'optional': True
                    },
                    "resolutions": {
                        'type': 'array',
                        'optional': True,
                        "minItems": 1,
                        "uniqueItems": True,
                        'items': {
                            'type': 'array',
                            "maxItems": 3,
                            "minItems": 3,
                            'items': {
                                'type': 'integer',
                            }
                        }
                    },
                    "enable-features": {
                        'type': 'array',
                        'items': {
                            'type': 'string'
                        }
                    },
                    "disable-features": {
                        'type': 'array',
                        'items': {
                            'type': 'string'
                        }
                    },
                    'force-fieldtrials': {
                        'type': 'object',
                        'additionalProperties': {
                            'type': 'string'
                        }
                    },
                    "force-fieldtrial-params": {
                        'type': 'object',
                        'additionalProperties': {
                            'type': 'string'
                        }
                    }
                }
            }
        }
    }
}


class ScreenshotsTestsConfig(sdk2.Resource):
    share = True


class BrowserAutotestRunScreentoolWithResolutions(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER  # because of teamcity access
        environments = [PipEnvironment('teamcity-client==3.0.0'),
                        PipEnvironment('jsonschema==2.5.1')]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build '
                        '(<a href="{}/project.html?projectId=BrowserBrandedDistributives">'
                        'BrowserBrandedDistributives</a>)'.format(TEAMCITY_URL))

        screenshots_tests_config = sdk2.parameters.JSON(
            'Screenshots tests config',
            description='Json file url with test launch parameters. (optional parameter for screenshots tests '
                        '(<a href="https://wiki.yandex-team.ru/avtomatizacija-testirovanija-desktopnogo-brauzera/t1/'
                        'zapusk-instrumenta-skrinshotov-tablo-na-raznyx-razreshenijax/">see what is it</a>)',
            default=[])

    def on_save(self):
        if not any([n.recipients == ['browser-autotests-bots'] for n in self.Parameters.notifications]):
            self.Parameters.notifications = list(self.Parameters.notifications) + [
                sdk2.Notification(
                    [Status.FAILURE, Status.Group.BREAK],
                    ['browser-autotests-bots'],
                    ctn.Transport.EMAIL
                ),
                sdk2.Notification(
                    [Status.FAILURE, Status.SUCCESS, Status.NO_RES, Status.EXCEPTION, Status.TIMEOUT, Status.EXPIRED],
                    [self.author],
                    ctn.Transport.EMAIL
                )
            ]

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        )

    def validate_params(self, browser_build, screenshots_tests_config):
        build_problems = check_browser_build(browser_build)
        import jsonschema
        try:
            jsonschema.validate(screenshots_tests_config, SCREENSHOTS_TEST_CONFIG_SCHEMA)
        except jsonschema.exceptions.ValidationError as e:
            build_problems.append('Invalid JSON: ' + e.message)

        if build_problems:
            raise RuntimeError("Incorrect params: {}".format(", ".join(build_problems)))

    def on_execute(self):

        with self.memoize_stage.create_config:
            browser_build = self.teamcity_client.Build(id=self.Parameters.build_id)
            self.validate_params(browser_build, self.Parameters.screenshots_tests_config)

            config_path = str(self.path('screenshots_tests_config.json'))
            with open(config_path, 'w') as json_file:
                json.dump(self.Parameters.screenshots_tests_config, json_file, indent=4)

            config_resource = ScreenshotsTestsConfig(self, 'Screenshots tests config', config_path)
            sdk2.ResourceData(config_resource).ready()
            screenshots_tests_config_url = config_resource.http_proxy

        with self.memoize_stage.run_autotests:

            browser_version = get_browser_version_number(browser_build)
            framework_branch = get_framework_branch(browser_build)
            raise sdk2.WaitTask(
                BrowserAutotestRunCustom(
                    self,
                    description=self.Parameters.description,
                    framework_branch=framework_branch,
                    build_id=self.Parameters.build_id,
                    browser_version=browser_version,
                    screenshots_tests_config_url=screenshots_tests_config_url,
                    tests_folders='tests_v20',
                    tests_collect_by="packs",
                    test_collect_condition="screenshots",
                    tests_launch_platforms=['win10_x64'],
                ).enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )
        child = self.find().first()
        self.set_info('REPORT: <a href="{}">{}</a>'.format(child.Context.report_url, child.Context.report_url),
                      False)
        for child in self.find():
            if child.status != Status.SUCCESS:
                raise TaskFailure('Child task {} is not successful'.format(child.type))
