# -*- coding: utf-8 -*-
import json
import logging
import re
from collections import defaultdict, Counter
from os.path import join
import os
from requests import HTTPError

from sandbox.projects.browser.autotests.testpalm_helper import TestpalmClientWrapper
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport.const import (
    is_configuration_main, TESTCASE_STATUSES)
from sandbox.projects.browser.autotests.regression.conf import get_group as get_group_from_config, TestingGroup
from sandbox.projects.browser.autotests_qa_tools.common import split_list
from sandbox.projects.common import decorators

XUNIT_FILE = "xunit.json"
ISSUE_TEXT = "Раны после автотестов: \n{testruns}"
TESTRUN_TITLE = "{group}{browser_version}-After Autotests [{brand}][{os}]{part}"
TESTRUN_TEXT = "{testrun_url} [{platform}] [{brand}] !!(green)Passed: {passed}!!| !!Failed: **{failed}**!!| " \
               "!!Broken: **{broken}**!! | Created: **{created}** |!!(gray)Skipped: {skipped}!!"
CASES_COUNT_UPDATE_LIMIT = 40

ROBOT_USERAGENT = 'Ya.Browser stat collector'
ST_BASE_API_URL = 'https://st-api.yandex-team.ru'
ST_TEST_BASE_API_URL = 'https://st-api.test.yandex-team.ru'
ST_TRACKER_URL = 'https://st.yandex-team.ru'

TESTPALM_API_HOST = "https://testpalm.yandex-team.ru/api/{path}"
TESTPALM_HOST = "https://testpalm.yandex-team.ru/"
TESTPALM_PROJECT = "brocase"
ENVIRONMENT_PARAMETERS = 'Environment parameters'
FINISH_TEST_STEP = "Finish test, save logs, pictures and more"


def find_attachment_by_title(title, testcase_report):
    attachments = testcase_report['attachments']
    steps = testcase_report['steps']
    for step in steps:
        attachments += step['attachments']
    for attachment in attachments:
        if attachment['title'] == title:
            return attachment['source']


def convert_platform_name(platform):
    """
    Удаляем 3 цифру из версии платформы
    :param platform: MacOS 10.12.2 64bit
    :return: MacOS 10.12 64bit
    """
    pattern = r"\d+.\.\d+\.\d+"
    return re.sub(pattern, lambda matchobj: matchobj.group(0)[:-2], platform)


def convert_pytest_status_to_testpalm(status):
    """
    Приводим статусы из pytest к testpalm
    :param status:
    :return:
    """
    if status == "PENDING" or status == TESTCASE_STATUSES.skipped.value:
        return TESTCASE_STATUSES.failed.value
    if status == "CANCELED":
        return TESTCASE_STATUSES.created.value
    else:
        return status


def get_testcases(allure_report_folder):
    """

    :param allure_report_folder:
    :return: {testcase_id: {platform1: {brand1: patf1_brand1_testcase_info, ..}, ..} }
    """
    testcases = {}

    # Parse xml report:
    with open(join(allure_report_folder, XUNIT_FILE), 'r') as f:
        report = json.load(f)

    # Find all test cases in the report:
    test_results = []
    for testsuite in report['testSuites']:
        test_results += testsuite['testCases']

    # Generate the list of results:
    for pytest_result in test_results:
        testcase_info = {}
        testcase_info['uid'] = pytest_result['uid']
        testcase_report = json.load(open(join(allure_report_folder, testcase_info['uid'] + "-testcase.json")))

        testcase_environment_attach_file_name = find_attachment_by_title(ENVIRONMENT_PARAMETERS, testcase_report)
        if testcase_environment_attach_file_name and \
                os.path.exists(join(allure_report_folder, testcase_environment_attach_file_name)):
            testcase_environment = json.load(open(join(allure_report_folder, testcase_environment_attach_file_name)))
            if testcase_report['testId'] is not None:
                testcase_id = testcase_report['testId']['name'].split('/')[-1]
                testcase_info['status'] = convert_pytest_status_to_testpalm(pytest_result['status'])
                platform = convert_platform_name(testcase_environment['platform'])
                brand = testcase_environment['brand_name']

                if testcases.get(testcase_id, {}).get(platform, {}).get(brand) is None:
                    testcases.setdefault(testcase_id, {}).setdefault(platform, {})[brand] = testcase_info
                else:
                    if testcases[testcase_id][platform][brand]['status'] == TESTCASE_STATUSES.passed.value \
                            and testcase_info['status'] != TESTCASE_STATUSES.passed.value:
                        testcases[testcase_id][platform][brand]['status'] = testcase_info['status']

    return testcases


def to_run_testcase(testcase_uuids, testcase_status):
    return [
        {
            "runTestCaseUuid": testcase_uuid,
            "status": testcase_status
        } for testcase_uuid in testcase_uuids
    ]


def is_case_failed_similar(testcase_platforms):
    """
    Падает ли кейс на всех платформах одинаково
    :param testcase_platforms:
    :return: bool
    """
    testcase_status = None
    for platform, brands_testcase_info in testcase_platforms.iteritems():
        for brand, testcase_info in brands_testcase_info.iteritems():
            if testcase_status is None:
                testcase_status = testcase_info['status']
            else:
                if testcase_status != testcase_info['status']:
                    return False
    if testcase_status != TESTCASE_STATUSES.passed.value:
        return True
    else:
        return False


def set_case_skip(testcase_platforms):
    """
    Помечаем case статусом skip везде, кроме основной платформы
    :param testcase_id:
    :param testcases:
    :return:
    """
    for platform, brands in testcase_platforms.iteritems():
        if not is_configuration_main(platform):
            for brand, testcase_info in brands.iteritems():
                testcase_info['status'] = TESTCASE_STATUSES.skipped.value
    return testcase_platforms


def delete_platform_duplicates(testcases):
    """
    Если тест падает одинаково на всех платформах, помечаем его skip везде, кроме основной
    :param testcases:
    :return:
    """
    for testcase_id, testcase_platforms in testcases.iteritems():
        if is_case_failed_similar(testcase_platforms):
            testcases[testcase_id] = set_case_skip(testcase_platforms)
    return testcases


@decorators.retries(5, delay=1, backoff=2)
def create_test_run(testsuite_id, version, title, testpalm_client, platform=None,
                    brand=None, parent_issue=None, build_url=None, project=TESTPALM_PROJECT):
    st_queue = None
    if parent_issue:
        st_queue = parent_issue.split('-')[0]
    testrun = {
        'title': title,
        'version': version,
        'parentIssue': {'id': parent_issue,
                        'trackerId': 'Startrek',
                        'groupId': st_queue},
        'properties': [{'key': 'OS', 'value': platform},
                       {'key': 'Brand/PartnerID', 'value': brand},
                       {'key': 'Build link', 'value': build_url}]
    }
    try:
        return testpalm_client.create_testrun_from_suite(testsuite_id, testrun, project)[0]['id']
    except HTTPError as e:
        if e.response.status_code == 406:  # empty suite
            logging.info('Suite %s seems to be empty, skip run creation', testsuite_id)
            return None
        raise


@decorators.retries(15, delay=1, backoff=2)
def update_testrun(project_id, testrun_id, tests_group, testpalm_client):
    testpalm_client.resolve_testrun_bulk(testrun_id, tests_group, project_id)


def post_testcase_statuses(allure_path, testruns, testpalm_client):
    # получаем для каждого кейса: id, статус выполенения, brand, OS
    all_testcases = get_testcases(allure_path)
    testcases = delete_platform_duplicates(all_testcases)
    testrun_with_testcases = defaultdict(list)
    for testcase_key in testcases:
        for platform in testcases[testcase_key]:
            if testruns.get(platform, None) is None:
                continue
            for brand in testcases[testcase_key][platform]:
                if testruns[platform].get(brand, None) is None:
                    continue
                for testrun in testruns[platform][brand]:
                    split = testcase_key.split('-')
                    testcase_project, testcase_id = '-'.join(split[:-1]), split[-1]
                    if testcase_id in testrun['case_ids'] and testcase_project == testrun['project_id']:
                        testrun_with_testcases["{}-{}".format(testrun['project_id'], testrun['testrun_id'])].extend(
                            to_run_testcase(testrun['case_ids'][testcase_id],
                                            testcases[testcase_key][platform][brand]['status']))

    for testrun_id, run_testcases in testrun_with_testcases.iteritems():
        logging.info('Updating {}'.format(testrun_id))
        for tests_group in split_list(run_testcases, CASES_COUNT_UPDATE_LIMIT):
            update_testrun(testrun_id.split("-")[0], testrun_id.split("-")[1], tests_group, testpalm_client)
            logging.info('Updated pack of cases')
    return testrun_with_testcases


def create_test_runs(testsuites, browser_version, issue, build_url, testpalm_client):
    """
    :return: {OS1: {brand1:run1_id, ..}, ..}
    """
    result = {}
    for testsuite in testsuites:
        testsuite_name = TESTRUN_TITLE.format(group='', browser_version=browser_version, brand=testsuite['brand'],
                                              os=testsuite['os'],
                                              part=' Part {}'.format(testsuite['part']) if 'part' in testsuite else '')

        project_id = testsuite.get("project_id", TESTPALM_PROJECT)
        testrun_id = create_test_run(testsuite['testsuite_id'], browser_version,
                                     testsuite_name, testpalm_client, testsuite['os'],
                                     testsuite['brand'], issue, build_url, project_id)
        if not testrun_id:
            continue
        testsuite_groups = testpalm_client.get_testrun_info(
            testrun_id, project_id,
            params={'include': 'testGroups.testCases.uuid,testGroups.testCases.testCase.id'})
        testsuite_cases = defaultdict(list)
        for group in testsuite_groups['testGroups']:
            for case_info in group['testCases']:
                testsuite_cases[unicode(case_info['testCase']['id'])].append(case_info['uuid'])
        if result.get(testsuite['os']) is None:
            result[testsuite['os']] = {testsuite['brand']: [
                dict(testrun_id=testrun_id,
                     project_id=project_id,
                     case_ids=testsuite_cases)]}
        else:
            result[testsuite['os']].setdefault(testsuite['brand'], []).append(
                dict(testrun_id=testrun_id,
                     project_id=project_id,
                     case_ids=testsuite_cases))
    return result


def testruns_to_string(testruns, testruns_with_cases):
    result = []
    for platform, os_testruns in testruns.iteritems():
        for brand, testrun_list in os_testruns.iteritems():
            for testrun in testrun_list:
                testrun_statuses = defaultdict(
                    int, Counter(case['status']
                                 for case in testruns_with_cases["{}-{}".format(testrun['project_id'],
                                                                                testrun['testrun_id'])]))
                untested_cases = (
                    sum(len(uuids) for uuids in testrun['case_ids'].values()) -
                    len(testruns_with_cases["{}-{}".format(testrun['project_id'], testrun['testrun_id'])])
                )
                created_cases = testrun_statuses['CREATED'] + untested_cases
                testrun_text = TESTRUN_TEXT.format(
                    testrun_url=TESTPALM_HOST + testrun['project_id'] + "/testrun/" + testrun['testrun_id'],
                    platform=platform, brand=brand,
                    passed=testrun_statuses['PASSED'],
                    failed=testrun_statuses['FAILED'],
                    broken=testrun_statuses['BROKEN'],
                    skipped=testrun_statuses['SKIPPED'],
                    created=created_cases)
                if testrun_statuses['FAILED'] + testrun_statuses['BROKEN'] + created_cases == 0:
                    testrun_text = '--' + testrun_text + '--'
                result.append(testrun_text)
    return result


def add_comment_to_startrek(issue_key, text, startrek_client):
    issue = startrek_client.issues[issue_key]
    return issue.comments.create(text=ISSUE_TEXT.format(testruns="\n".join(text)))


def report_to_testpalm(task):
    """
    Создаем раны по операционным системам и брендам, которые есть в отчете.
    В каждом ране помечаем кейсы, согласно их статусу в отчете.
    """

    if not task.Context.testruns:
        # создаем раны по всем кейсам is automated = true, status = actual,need changes. Делим их по брендам и ОСям
        after_autotests_suites = task.Parameters.after_autotests_suites.get(task.Parameters.platform, [])
        task.Context.testruns = create_test_runs(
            after_autotests_suites, task.Parameters.browser_version,
            task.Parameters.regression_issue, task.Parameters.build_url, task.testpalm_client)
    # помечаем нужными статусами кейсы в ранах
    if not task.Context.testruns_updated:
        task.Context.testruns_with_testcases = post_testcase_statuses(task.Context.allure_report_path,
                                                                      task.Context.testruns, task.testpalm_client)
        task.Context.testruns_updated = True

    if not task.Context.comment_text:
        task.Context.comment_text = testruns_to_string(task.Context.testruns, task.Context.testruns_with_testcases)
    task.startrek_client.issues[task.Parameters.regression_issue].comments.create(
        text=ISSUE_TEXT.format(testruns="\n".join(task.Context.comment_text)))
    divide_runs_by_groups(task.Context.testruns, task.Parameters.regression_issue, task.Parameters.group_issues,
                          task.testpalm_client, task.Parameters.browser_version, task.startrek_client)


def divide_runs_by_groups(test_runs, regression_issue, groups_issues, restpalm_client, browser_version,
                          startrek_client):
    groups_cases = defaultdict(lambda: defaultdict(lambda: defaultdict(lambda: defaultdict(list))))
    new_testpalm_client = TestpalmClientWrapper(restpalm_client)
    for platform, os_testruns in test_runs.iteritems():
        for brand, testrun_list in os_testruns.iteritems():
            for testrun in testrun_list:
                testrun_obj = new_testpalm_client.get_testrun(
                    project=testrun['project_id'], testrun=testrun['testrun_id'])
                case_statuses = testrun_obj.testcase_statuses
                for case in testrun_obj.testcases:
                    group = get_group_from_config(
                        case.mapped_attributes['Component'][0], TestingGroup('UnknownComponents', '', [], ['edragun']))
                    groups_cases[group][testrun['project_id']][platform][brand].append((case, case_statuses[case.id]))

    main_issue_comment = ''
    for group, _ in groups_cases.iteritems():
        created_runs = []
        group_issue = groups_issues.get(group.name)
        for project, __ in _.iteritems():
            for platform, os_testruns in __.iteritems():
                for brand, cases_with_statuses in os_testruns.iteritems():
                    cases = [case for case, status in cases_with_statuses]
                    statuses = {case.id: status for case, status in cases_with_statuses}
                    logging.info('Creating runs for {} group. brand: {}, platfrom: {}, project: {}'.format(
                        group.name, brand, platform, project
                    ))
                    logging.debug('Case statuses: {}'.format(statuses))
                    testrun_id = new_testpalm_client.create_testrun(
                        project=project,
                        title=TESTRUN_TITLE.format(group='[{}]'.format(group.name), os=platform, brand=brand, part='',
                                                   browser_version=browser_version),
                        cases=cases,
                        cases_statuses=statuses,
                        ticket=group_issue,
                        version=None,
                        current_environment=None,
                        properties=None,
                    )
                    testrun = new_testpalm_client.get_testrun(project=project, testrun=testrun_id)
                    created_runs.append(testrun)
        pretty_runs = '<br>'.join(testrun.pretty_str(True) for testrun in created_runs)
        if group_issue:
            issue_obj = startrek_client.issues[group_issue]
            startrek_client.issues.update(
                issue_obj,
                description=u'{}\n=== Результаты автотестов\n<#{}#>'.format(issue_obj.description, pretty_runs)
            )
        else:
            main_issue_comment += u'\nРаны после автотестов для группы {}: \n<#{}#>'.format(group.name, pretty_runs)
    if main_issue_comment:
        startrek_client.issues[regression_issue].comments.create(
            text=u'Эти раны не были добавлены в тикеты подгрупп:\n' + main_issue_comment)
