# -*- coding: utf-8 -*-
"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
import logging
import uuid
import zipfile
from StringIO import StringIO
from abc import abstractmethod
import os
import time
from os import mkdir
from xml.etree import ElementTree

from sandbox.common.utils import get_task_link
from sandbox.common.errors import TaskFailure
from sandbox import common
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.common.types.task import Status
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.browser.autotests.allure_parser import AllureReport
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL, BitBucket
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds
from sandbox.projects.browser.autotests import elastic_report
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import (
    AutotestsDump, AutotestsAllureReport, AutotestsReports, AutotestsAllureData)
from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, TEAMCITY_URL)
from sandbox.projects.browser.common.teamcity import run_teamcity_build
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment, AllureCommandLineEnvironment
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment


TEAMCITY_AUTOTESTS_BUILD = {
    'win': 'Browser_Tests_Functional_TeamcityAutotests_Win',
    'mac': 'Browser_Tests_Functional_TeamcityAutotests_Mac'
}

MAIN_MAC_FLAVOUR = 'high_sierra'
TEAMCITY_AUTOTESTS_POOLS = {
    'win7_x64': 'autotests-win7-x64',
    'win7_x64_uac': 'autotests-win7-x64-uac',
    'win7_x86': 'autotests-win7-x86',
    'win8_x64': 'autotests-win8-x64',
    'win10_x64': 'autotests-win10-x64',
    'high_sierra': 'autotests-mac-high-sierra',
    'yosemite': 'autotests-mac-yosemite',
    'elcapitan': 'autotests-mac-elcapitan',
    'sierra': 'autotests-mac-sierra',
    'mojave': 'autotests-mac-mojave',
}

BINARY_TEAMCITY_AUTOTESTS_POOLS = {
    'win7_x64': 'bautotests-win7-x64',
    'win7_x64_uac': 'bautotests-win7-x64',
    'win7_x86': 'bautotests-win7-x86',
    'win8_x64': 'bautotests-win8-x64',
    'win10_x64': 'bautotests-win10-x64',
    'high_sierra': 'autotests-mac-high-sierra',
    'yosemite': 'autotests-mac-yosemite',
    'elcapitan': 'autotests-mac-elcapitan',
    'sierra': 'autotests-mac-sierra',
    'mojave': 'autotests-mac-mojave',
}

MAX_RERUNS = 1
WAIT_BUILDS_TIME = 300

STATUSES_CLASSES = {
    "RUNNING": "status_executing",
    "SUCCESS": "status_success",
    "FAILURE": "status_exception",
    "UNKNOWN": "status_draft"
}


class BrowserAutotestsTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 500
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER  # because of teamcity access
        platfrom = 'linux_ubuntu_14.04_trusty'
        environments = [
            PipEnvironment('teamcity-client==4.8.2'),
            AllureCommandLineEnvironment('1.4.22'),
            SandboxJavaJdkEnvironment('1.8.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        framework_branch = sdk2.parameters.String(
            'Framework branch', default='master', required=True,
            description='Branch in <a href="{}/projects/AUTOTEST/repos/browser-test-framework/browse">autotests</a>'
                        ' repository'.format(DEFAULT_BITBUCKET_URL))
        framework_commit = sdk2.parameters.String('Framework repo commit', default='')

        default_tests_launch_arguments = sdk2.parameters.String(
            'Arguments string for tests launch', default='', required=False,
            description='')
        sleep_time = sdk2.parameters.Integer('Build check period.',
                                             description='How often check if teamcity build finished. In minutes.',
                                             default=15)
        wait_builds_time = sdk2.parameters.Integer('Build wait period.',
                                                   description='How long to wait for the completion of timcity builds. In minutes.',
                                                   default=WAIT_BUILDS_TIME)
        use_test_pools = sdk2.parameters.Bool('Use test agent pools', description='Used to test new agent images',
                                              default=False)
        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token  for teamcity and bitbucket',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    class Context(sdk2.Context):
        start_time = None
        download_xml_reports = True
        teamcity_builds = {}
        allure_report_path = None
        report_url = None
        builds_without_report = {}
        framework_commit = None

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(self.Parameters.oauth_vault))

    @property
    @common.utils.singleton
    def bitbucket_client(self):
        return BitBucket(DEFAULT_BITBUCKET_URL,
                         'x-oauth-token',
                         sdk2.Vault.data(self.Parameters.oauth_vault))

    @common.utils.singleton
    @decorators.retries(5, delay=120, backoff=1.2)
    def get_change(self):
        change = self.teamcity_client.Change(
            buildType__id=TEAMCITY_AUTOTESTS_BUILD['win'],  # any project with framework vcs
            version=self.get_framework_commit())
        change.id  # force lazy client to request change
        return change

    @common.utils.singleton
    def get_framework_commit(self):
        if self.Context.framework_commit is None:
            self.Context.framework_commit = self.Parameters.framework_commit or self.get_latest_commit()
        return self.Context.framework_commit

    @decorators.retries(5, delay=2, backoff=2)
    def get_latest_commit(self):
        return self.bitbucket_client.get_latest_commit(
            FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, self.real_framework_branch)

    @abstractmethod
    def get_teamcity_builds_to_launch(self):
        pass

    @common.utils.singleton
    def get_pool_by_name(self, name):
        return self.teamcity_client.Pool(name=name)

    @property
    @common.utils.singleton
    def real_framework_branch(self):
        real_branch = self.Parameters.framework_branch
        if real_branch.startswith('pull-requests'):
            real_branch = 'refs/{}/merge-pin'.format(real_branch)
        return real_branch

    def build_comment(self, build_name):
        return '{}\n{}\nLaunched by task sandbox-{}'.format(
            build_name, self.Parameters.description.encode('utf-8'), self.id)

    def wait_teamcity_builds(self):
        for name, info in self.Context.teamcity_builds.iteritems():
            if info['status'] == 'RUNNING':
                try:
                    build = self.teamcity_client.builds[info['build_id']]
                    state = build.state
                    status = build.status
                except:
                    logging.exception('Failed to get build #{} info'.format(info['build_id']))
                else:
                    if state == 'finished':
                        if status != 'SUCCESS' and len(info['reruns']) < MAX_RERUNS:
                            old_id = info['build_id']
                            try:
                                info['build_id'] = run_teamcity_build(
                                    self.teamcity_client,
                                    branch=self.Parameters.framework_branch,
                                    change=self.get_change(),
                                    build_type=info['build_type'],
                                    parameters=info['build_params'],
                                    comment='{}\nRerun build #{}'.format(self.build_comment(name), old_id),
                                    pool=self.get_pool_by_name(info['pool']),
                                ).id
                                info['reruns'].append(old_id)
                            except:
                                logging.exception('Failed to rerun build #{}'.format(old_id))
                        else:
                            info['status'] = status
                            self.Context.running_builds -= 1
            if not info['build_id']:
                try:
                    info['build_id'] = run_teamcity_build(
                        self.teamcity_client,
                        branch=self.Parameters.framework_branch,
                        change=self.get_change(),
                        build_type=info['build_type'],
                        parameters=info['build_params'],
                        comment=self.build_comment(name),
                        pool=self.get_pool_by_name(info['pool']),
                    ).id
                    info['status'] = 'RUNNING'
                except:
                    logging.exception('Failed to start build #{}'.format(info['build_id']))

        if self.Context.running_builds:
            if int(time.time()) - self.Context.start_time < self.Parameters.wait_builds_time * 60:
                raise sdk2.WaitTime(self.Parameters.sleep_time * 60)
            else:
                self.set_info(u"! WARNING: Таска исчерпала таймаут {} мин. Некоторые teamcity-сборки не завершены и остановлены."
                              u" Отчет автотестов будет сформирован частично, на основе завершенных успешных сборок."
                              u"".format(self.Parameters.wait_builds_time))
                self.stop_teamcity_builds()

    def download_results(self, results_dir, reports_dir):
        self.Context.failed_builds_with_names = {}
        for launch_name, info in self.Context.teamcity_builds.iteritems():
            build_id = info['build_id']
            build = self.teamcity_client.rest_api.builds.locator(
                id=build_id
            )
            if info['status'] != 'SUCCESS':
                self.Context.failed_builds_with_names[launch_name] = build.get().webUrl
            launch_dir = os.path.join(reports_dir, launch_name)
            mkdir(launch_dir)
            artifacts_names = [artifact['name'] for artifact in build.artifacts.get().file]
            if 'junit_report.xml' in artifacts_names:
                with open(os.path.join(launch_dir, 'junit_report.xml'), 'w') as f:
                    f.write(ElementTree.tostring(build.artifacts.content['junit_report.xml'].get()))
            if 'allure.zip' in artifacts_names:
                with zipfile.ZipFile(StringIO(build.artifacts.content['allure.zip'].get()),
                                     compression=zipfile.ZIP_DEFLATED) as myzipfile:
                    myzipfile.extractall(results_dir)

        if self.Context.failed_builds_with_names:
            self.set_info(
                'There are unsuccessful builds:<br>' +
                ', '.join(['<a href="{}">{}</a>'.format(url, launch_name)
                           for launch_name, url in self.Context.failed_builds_with_names.iteritems()]),
                do_escape=False)

    def add_task_link_to_allure_env(self, file_path):
        try:
            tree = ElementTree.parse(file_path)
            root = tree.getroot()
            param = ElementTree.SubElement(root, 'parameter')

            name = ElementTree.SubElement(param, 'name')
            name.text = 'Sandbox task'

            key = ElementTree.SubElement(param, 'key')
            key.text = 'Sandbox task'

            value = ElementTree.SubElement(param, 'value')
            value.text = get_task_link(self.id)

            tree.write(file_path)
        except Exception as e:
            self.set_info(str(e))
            self.set_info('Failed to add task link to allure report')

    def on_execute(self):
        if not self.Context.start_time:
            self.Context.start_time = int(time.time())

        if self.Parameters.use_test_pools:
            # raises Exception if someone outside browser infrastructure tries to use this param
            sdk2.Vault.data('browser-infra-blank-vault')
        if not self.Context.teamcity_builds:
            for build_name, (build_type, pool, params) in self.get_teamcity_builds_to_launch().iteritems():
                params['unused_unique_parameter'] = str(uuid.uuid4())
                self.Context.teamcity_builds[build_name] = {
                    'reruns': [], 'status': 'UNKNOWN', 'build_id': None, 'build_type': build_type,
                    'build_params': params, 'pool': pool + ('-test' if self.Parameters.use_test_pools else ''),
                }
            self.Context.running_builds = len(self.Context.teamcity_builds)
        self.wait_teamcity_builds()

        allure_source = str(self.path('allure_source'))
        xml_reports_dir = str(self.path('xml-reports'))
        mkdir(allure_source)
        mkdir(xml_reports_dir)
        self.download_results(allure_source, xml_reports_dir)
        self.add_task_link_to_allure_env(os.path.join(allure_source, 'environment.xml'))
        if self.Context.download_xml_reports:
            xml_reports_resource = AutotestsReports(self, 'Allure junit reports', xml_reports_dir)
            sdk2.ResourceData(xml_reports_resource).ready()

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('generate_allure')) as pl:
            self.Context.allure_report_path = str(self.path('allure-report'))
            report = AutotestsAllureReport(self, 'Allure report', self.Context.allure_report_path)
            status = sp.Popen(['allure', 'generate', allure_source, '-o', self.Context.allure_report_path],
                              stdout=pl.stdout, stderr=sp.STDOUT).wait()
            if status != 0:
                raise Exception('Failed to generate allure report')
            else:
                sdk2.ResourceData(report).ready()
            self.Context.report_url = report.http_proxy + '/index.html'
            self.set_info('REPORT: <a href="{}">{}</a>'.format(self.Context.report_url, self.Context.report_url),
                          False)

            try:
                tests_to_report, all_successful = elastic_report.get_tests_from_task_allure(self, self.Context.allure_report_path)
                elastic_report.send_autotests_report(bulk_upload=True,
                                                     data=elastic_report.construct_bulk_insert(tests_to_report))
                self.set_info("Tests' data sent to elastic")
                if not all_successful:
                    self.set_info("Some of the tests' parameters are absent, see common.log")
            except Exception as e:
                self.set_info(str(e))
                self.set_info("Failed to send tests' data to elastic")

            # lightweight allure data for automated report processing
            self.Context.allure_lightweight_data = str(self.path('allure-data', 'data'))
            os.makedirs(self.Context.allure_lightweight_data)
            lightweight = AutotestsAllureData(self, 'Allure lightweight data',
                                              self.Context.allure_lightweight_data)

            with zipfile.ZipFile(os.path.join(self.Context.allure_lightweight_data, "allure.zip"), 'w') as allure_zip:
                for dirpath, dnames, fnames in os.walk(
                        os.path.join(self.Context.allure_report_path, 'data')):
                    for f in fnames:
                        if not f.endswith(('.mp4', '.png')):
                            allure_zip.write(os.path.join(dirpath, f), f, compress_type=zipfile.ZIP_DEFLATED)
            sdk2.ResourceData(lightweight).ready()

            # tests_dump resource
            dump_path = str(self.path('tests_dump'))
            os.makedirs(dump_path)
            dump_resource = AutotestsDump(self, 'Autotests dump', dump_path)
            allure_report = AllureReport(os.path.join(self.Context.allure_report_path, 'data'), None)
            allure_report.dump_tests(dump_path)
            sdk2.ResourceData(dump_resource).ready()

        if self.Context.failed_builds_with_names:
            raise TaskFailure('There are unsuccessful teamcity builds')

    def stop_teamcity_builds(self):
        if self.Context.teamcity_builds:
            BrowserStopTeamcityBuilds(
                None,
                description='Stop builds for task {}'.format(self.id),
                teamcity_build_ids=[info['build_id'] for info in self.Context.teamcity_builds.values()],
                cancel_comment='Task sandbox-{} was stopped'.format(self.id),
                owner=self.Parameters.owner,
                oauth_vault=ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
            ).enqueue()

    def on_break(self, prev_status, status):
        if status == Status.STOPPED:
            self.stop_teamcity_builds()

    def link_to_build(self, build_id):
        return '{}/viewLog.html?buildId={}'.format(TEAMCITY_URL, build_id)

    @sdk2.report(title="Launched builds")
    def launched_build_tab(self):
        if not self.Context.teamcity_builds:
            return 'No builds has been launched yet'
        return '<br>'.join(
            [
                "<a href={}>{}</a>: <span class='status {}'>{}</span>{}".format(
                    self.link_to_build(info['build_id']), build_name, STATUSES_CLASSES[info['status']],
                    info['status'],
                    ' (rerun {})'.format(
                        ', '.join(['<a href={}>#{}</a>'.format(self.link_to_build(build_id), build_id)
                                   for build_id in info['reruns']]))
                    if info['reruns'] else ''
                ) if info['build_id'] else build_name
                for build_name, info in self.Context.teamcity_builds.iteritems()
            ]
        )
