# coding=utf-8
import os
import textwrap
import yaml

import sandbox
from sandbox.projects.browser.autotests.classes.regression_manager.desktop import (DBroRegressionManager,
                                                                                   DBroRegressionManagerWithComponents)
from sandbox.projects.browser.autotests.classes.testing_job import JobExecutors
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests_qa_tools.common import TEAMCITY_URL
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import BaseBrowserRegression
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.browser.autotests.regression_tasks.configs import dbro_test_suites_configs
from sandbox.projects.browser.autotests.testpalm_helper import TestpalmClientWrapper
from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, pretty_time_delta, html_link, BRANDED_BUILDS_PROJECT, TESTS_BUILDS_PROJECT)

TEST_SUITE_CONFIGS_PATH = os.path.dirname(dbro_test_suites_configs.__file__)
TEST_SUITE_CONFIGS = dbro_test_suites_configs.TEST_SUITE_CONFIGS


class BrowserEternalResource(sdk2.Resource):
    """
    A resource with no useful content, just to make parent task stored infinitely, like if it was launched by user.
    """
    ttl = 'inf'


class BrowserCalculateFunctionalitiesProfit(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String("Mode") as mode:
            mode.values.GET_INFO_FROM_TASK = mode.Value("Calculate profit for given regression task")
            mode.values.GET_INFO_FROM_PARAMS = mode.Value("Calculate profit for builds in params", default=True)

        with mode.value['GET_INFO_FROM_TASK']:
            regression_task = sdk2.parameters.Task('regression task')

        with mode.value['GET_INFO_FROM_PARAMS']:
            old_build_id = sdk2.parameters.Integer('Previous build id',
                                                   description='Use to calculate diff tickets')
            build_id = sdk2.parameters.Integer(
                'Browser build id',
                description='Id of browser teamcity build '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
                required=True
            )
            enable_autotests = sdk2.parameters.Bool('Use autotests', default=False)
            with enable_autotests.value[True]:
                browser_tests_build_id = sdk2.parameters.Integer(
                    'Tests build id',
                    description='Id of teamcity build with browser tests '
                                '(<a href="{base}/project.html?projectId={project}">'
                                '{project}</a>)'.format(base=TEAMCITY_URL, project=TESTS_BUILDS_PROJECT),
                )

            regression_type = sdk2.parameters.String(
                'Regression type', required=True, choices=[(_, _) for _ in TEST_SUITE_CONFIGS + ['custom']],
                ui=sdk2.parameters.String.UI('select')
            )
            with regression_type.value['custom']:
                test_suites_override = BaseBrowserRegression.Parameters.test_suites()
            with sdk2.parameters.RadioGroup("Diff type") as diff_type:
                diff_type.values.functionalities_diff = diff_type.Value("Functionalities diff", default=True)
                diff_type.values.component_diff = diff_type.Value("Component diff")
                diff_type.values.disabled = diff_type.Value("Diff disabled")
            scope_filter = sdk2.parameters.String(
                'Scope filter', default='', multiline=True,
                description='This filter will be applied to issues that are found'
                            ' in diff before computing components that will be checked in this regression. '
                            'For example, if "Priority: Blocker" filter, then components from blocker issues with'
                            'recent commits will be checked.'
                            '  See <a href="https://wiki.yandex-team.ru/browser/dev/infra/qa/regression-launcher/diff/#nastrojjkadiffavregressionnyxtaskax">documentation</a>')

            # these params are not really needed to calculate profit, but RegressionManager uses them to create jobs
            distribution_type = sdk2.parameters.String('Distribution type', required=True, default='brands',
                                                       choices=[(_, _) for _ in ['brands', 'partners']],
                                                       ui=None)
            distribution_name = sdk2.parameters.String('Brand/Partner name', required=True, default='yandex',
                                                       ui=None)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
            PipEnvironment('testpalm-api-client', version='4.0.2'),
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
            PipEnvironment('jsonschema==2.5.1'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def regression_task(self):
        return self.Parameters.regression_task

    @sandbox.common.utils.singleton
    def get_config(self, config_file):
        with open(os.path.join(TEST_SUITE_CONFIGS_PATH, config_file)) as config_file:
            return yaml.load(config_file)

    @sandbox.common.utils.singleton
    def regression_config(self, task):
        if task.Parameters.regression_type != 'custom':
            return self.get_config(task.Parameters.regression_type)
        else:
            return task.Parameters.test_suites_override

    def create_eternal_resource(self):
        dummy_file = self.path('dummy')
        dummy_file.touch()
        resource = BrowserEternalResource(self, 'Dummy resource', dummy_file)
        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        self.create_eternal_resource()
        task = self.regression_task if self.Parameters.mode == 'GET_INFO_FROM_TASK' else self
        new_manager = DBroRegressionManager(
            regression_config=self.regression_config(task),
            task_parameters=task.Parameters,
            task_context=task.Context,
            oauth_vault=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))
        old_manager = DBroRegressionManagerWithComponents(
            regression_config=self.regression_config(task),
            task_parameters=task.Parameters,
            task_context=task.Context,
            oauth_vault=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        )

        new_jobs = new_manager.regression_jobs
        old_jobs = old_manager.regression_jobs

        new_cases = set('{}-{}'.format(j['case_project'], j['case_id']) for j in new_jobs)
        old_cases = set('{}-{}'.format(j['case_project'], j['case_id']) for j in old_jobs)

        self.Context.added_cases = sorted(new_cases - old_cases)
        self.Context.removed_cases = sorted(old_cases - new_cases)

        self.Context.assessors_jobs = len([job for job in new_jobs if job['executor'] == JobExecutors.assessor.value])
        self.Context.manual_jobs = len([job for job in new_jobs if job['executor'] == JobExecutors.manual.value])

        self.Context.assessors_new_jobs_estimate = sum(
            job['estimate'] for job in new_jobs if job['executor'] == JobExecutors.assessor.value)
        self.Context.manual_new_jobs_estimate = sum(
            job['estimate'] for job in new_jobs if job['executor'] == JobExecutors.manual.value)
        self.Context.assessors_old_jobs_estimate = sum(
            job['estimate'] for job in old_jobs if job['executor'] == JobExecutors.assessor.value)
        self.Context.manual_old_jobs_estimate = sum(
            job['estimate'] for job in old_jobs if job['executor'] == JobExecutors.manual.value)

        self.Context.saved_assessors_time = (
            self.Context.assessors_old_jobs_estimate - self.Context.assessors_new_jobs_estimate)
        self.Context.saved_manual_time = (
            self.Context.manual_old_jobs_estimate - self.Context.manual_new_jobs_estimate)

        self.Context.total_saved = self.Context.saved_manual_time + self.Context.saved_assessors_time

        self.set_info(
            textwrap.dedent(
                '''
                <b>Jobs count</b>:
                Assessors jobs with functional diff: {assessors_jobs}
                Manual jobs with functional diff: {manual_jobs}

                <b>Estimate cases durations</b>:
                Assessors cases with component diff: {assessors_old}
                Assessors cases with functional diff: {assessors_new}

                Manual cases with component diff: {manual_old}
                Manual cases with functional diff: {manual_new}

                Assessors profit: {assessors_profit}
                Manual profit: {manual_profit}

                <b>Total profit: {total_profit}</b>
                '''.format(
                    assessors_jobs=self.Context.assessors_jobs,
                    manual_jobs=self.Context.manual_jobs,
                    assessors_old=pretty_time_delta(self.Context.assessors_old_jobs_estimate),
                    assessors_new=pretty_time_delta(self.Context.assessors_new_jobs_estimate),
                    manual_old=pretty_time_delta(self.Context.manual_old_jobs_estimate),
                    manual_new=pretty_time_delta(self.Context.manual_new_jobs_estimate),
                    assessors_profit=pretty_time_delta(self.Context.saved_assessors_time),
                    manual_profit=pretty_time_delta(self.Context.saved_manual_time),
                    total_profit=pretty_time_delta(self.Context.total_saved)
                )
            ),
            do_escape=False,
        )

    @sdk2.report(title='Added cases')
    def added_cases(self):
        if not self.Context.added_cases:
            return 'Cases not added or calculation in progress'
        return '<br/>'.join(
            html_link('https://testpalm.yandex-team.ru/testcase/{}'.format(case)) for case in self.Context.added_cases)

    @sdk2.report(title='Removed cases')
    def removed_cases(self):
        if not self.Context.removed_cases:
            return 'Cases not removed or calculation in progress'
        return '<br/>'.join(
            html_link('https://testpalm.yandex-team.ru/testcase/{}'.format(case))
            for case in self.Context.removed_cases)

    @property
    @sandbox.common.utils.singleton
    def testpalm_client(self):
        from testpalm_api_client.client import TestPalmClient
        return TestpalmClientWrapper(TestPalmClient(oauth_token=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)))
