# -*- coding: utf-8 -*-
import logging
import datetime
import requests
import time

import sandbox
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import notification as ctn
import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.autoduty_tasks.BrowserTestpalmAnalyst import BrowserTestpalmAnalyst
from sandbox.projects.browser.autotests.classes.autotests_result import IsolatesAutotestsResult
from sandbox.projects.browser.autotests.classes.testing_builds import ReleaseKits
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.PrepareBrowserRegressionBuilds import PrepareBrowserRegressionBuilds
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests.regression_tasks.RunDesktopBrowserAutotests import RunDesktopBrowserAutotests
from sandbox.projects.browser.autotests.RunNightlyBrowserAutotests import RunNightlyBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT, DEFAULT_MAX_FLAKY_RETRIES
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsReportResource

MASTER_BRANCH = "master"
MASTER_NEXT_BRANCH = "master-next"

DEFAULT_PROJECTS = {
    "android": {
        "builds_kit": ReleaseKits.android_autotests_nightly.name,
        "required_branches": [MASTER_BRANCH],
        "max_branches_count": 1,
        "autotests_timeout": 60 * 5,
        "max_flaky_retries": 1,  # see https://st.yandex-team.ru/ABRO-62408#6128d8b6e6186613a84edd7c
        "generate_allure_reports": True,
        "tested_application": "browser"
    },
    "android_searchapp": {
        "builds_kit": ReleaseKits.android_searchapp_nightly.name,
        "required_branches": [MASTER_NEXT_BRANCH, MASTER_BRANCH],
        "max_branches_count": 4,
        "autotests_timeout": 60 * 5,
        "max_flaky_retries": DEFAULT_MAX_FLAKY_RETRIES,
        "generate_allure_reports": True,
        "tested_application": "searchapp"
    },
    "ios": {
        "builds_kit": ReleaseKits.ios_autotests_nightly.name,
        "required_branches": [MASTER_NEXT_BRANCH, MASTER_BRANCH],
        "max_branches_count": 4,
        "autotests_timeout": 60 * 3,
        "max_flaky_retries": DEFAULT_MAX_FLAKY_RETRIES,
        "generate_allure_reports": True,
        "tested_application": "browser"
    },
    "ios_searchapp": {
        "builds_kit": ReleaseKits.ios_autotests_nightly.name,
        "required_branches": [MASTER_NEXT_BRANCH, MASTER_BRANCH],
        "max_branches_count": 4,
        "autotests_timeout": 60 * 3,
        "max_flaky_retries": DEFAULT_MAX_FLAKY_RETRIES,
        "generate_allure_reports": True,
        "tested_application": "searchapp",
    },
    "desktop_win": {
        "builds_kit": ReleaseKits.desktop_win.name,
        "required_branches": [MASTER_NEXT_BRANCH, MASTER_BRANCH],
        "max_branches_count": 4,
        "autotests_timeout": 60 * 5,
        "max_flaky_retries": 0,
        "generate_allure_reports": False,
        "tested_application": "browser"
    },
    "desktop_mac": {
        "builds_kit": ReleaseKits.desktop_mac.name,
        "required_branches": [MASTER_NEXT_BRANCH, MASTER_BRANCH],
        "max_branches_count": 4,
        "autotests_timeout": 60 * 4,
        "max_flaky_retries": DEFAULT_MAX_FLAKY_RETRIES,
        "generate_allure_reports": False,
        "tested_application": "browser"
    }
}

RELEASES_MAP = {
    'android': {
        'project_name': 'android',
        'release_platform': 'android'
    },
    'android_searchapp': {
        'project_name': 'pp-android',
        'release_platform': 'android'
    },
    'ios': {
        'project_name': 'ios',
        'release_platform': 'ios'
    },
    'ios_searchapp': {
        'project_name': 'pp-ios',
        'release_platform': 'ios'
    },
    'desktop_win': {
        'project_name': 'desktop',
        'release_platform': 'win'
    },
    'desktop_mac': {
        'project_name': 'desktop',
        'release_platform': 'mac'
    }
}

TESTPALM_ANALYST_PROJECTS = {
    "mobile": {
        "testpalm_projects": [
            'abro',
            'ibro',
            'mobsearchios',
        ],
        "tests_builds": [
            "android",
            "ios",
        ],
        "desktop_autotests_branch": '',
        "consider_platforms": [
            "apad",
            "aphone",
            "ios",
        ],
        "treat_blacklisted_cases_as_automated": True,
        "auto_set_ready_status": True,
        "report_recipients": ["report_of_mobile_autotests@yandex-team.ru"],
    }
}


def get_current_releases(project_name):
    _project_name = RELEASES_MAP[project_name]['project_name']
    _release_platform = RELEASES_MAP[project_name]['release_platform']
    releases_url = ('https://browser.yandex-team.ru/rest/projects/{}/releases/'
                    '?show_only_active=true&include_milestones=true'.format(_project_name))
    current_timestamp = int(time.mktime(datetime.datetime.today().timetuple()))
    logging.debug('current timestamp %s', current_timestamp)
    current_releases = requests.get(releases_url).json().get('items', [])
    logging.debug('current releases %s', current_releases)
    current_releases = [
        release for release in current_releases
        if _release_platform in release.get('platforms', []) and any(milestone['kind'] == 'release:rc' and milestone['date'] > current_timestamp
                                                                     for milestone in release['milestones'])
    ]
    res = sorted(current_releases, key=lambda release: release['milestones'][0]['date'])
    logging.debug('filtered current releases %s', res)
    return res


class BrowserNightlyAutotestsTrigger(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.BROWSER

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        notifications = (
            sdk2.Notification(
                [Status.Group.FINISH, Status.Group.BREAK],
                ['browser-autotests-bots'],
                ctn.Transport.EMAIL
            )
        )
        projects_and_builds = sdk2.parameters.JSON(
            'Projects and builds',
            default={}
        )

    class Context(sdk2.Context):
        projects = {}

    def task_href(self, task_id):
        return '<a href={link}>#{id}</a>'.format(link=get_task_link(task_id), id=task_id)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @property
    @sandbox.common.utils.singleton
    def projects_and_builds(self):
        return self.Parameters.projects_and_builds or DEFAULT_PROJECTS

    def prepare_builds(self):
        existing_branches = [branch_info['displayId'] for branch_info in
                             self.clients.bitbucket.get_branches('stardust', 'browser', 'master', 'MODIFICATION')]
        tasks = {}
        for project, config in self.projects_and_builds.iteritems():
            release_branches = [
                release["real_branch"] for release in get_current_releases(project) if release["real_branch"] in existing_branches]
            branches = config["required_branches"] + release_branches
            branches = branches[:config["max_branches_count"]]
            tasks[project] = {
                branch: PrepareBrowserRegressionBuilds(
                    self,
                    browser_branch=branch,
                    build_type=config['builds_kit'],
                    add_autotests_build=True,
                    tags=['browser-autotests-nightly-{}'.format(project),
                          branch]
                ).enqueue() for branch in branches}

        for project, project_branches in tasks.iteritems():
            self.Context.projects[project] = {
                branch: {
                    "builds_task_id": task.id,
                    "autotests_task_id": None,
                    "autotests_report_url": None,
                    "autotests_results": None,
                    "problems": [],
                    "complete_status": False,
                    "html_summary": u"Нет",
                }
                for branch, task in project_branches.iteritems()}

    def launch_autotests(self, builds_task, project, branch):
        autotests_task = RunNightlyBrowserAutotests(
            self,
            browser_tests_build_id=builds_task.Context.teamcity_builds.get("autotests", 0),
            browser_build_id=builds_task.Context.teamcity_builds.get("browser", 0),
            fake_build_id=builds_task.Context.teamcity_builds.get("fake_browser", 0),
            tags=['browser-autotests-nightly-{}'.format(project),
                  branch],
            timeout=self.projects_and_builds[project]['autotests_timeout'],
            max_flaky_retries=self.projects_and_builds[project].get('max_flaky_retries',
                                                                               DEFAULT_MAX_FLAKY_RETRIES),
            generate_allure_reports=self.projects_and_builds[project].get('generate_allure_reports',
                                                                                     False),
            tested_application=self.projects_and_builds[project].get('tested_application', 'browser'),
            statistics_launch_type="Nightly"
        ).enqueue()
        return autotests_task

    def process(self, project, branch):
        branch_context = self.Context.projects[project][branch]
        if not branch_context["complete_status"]:
            builds_task = sdk2.Task.find(id=branch_context["builds_task_id"], children=True).limit(1).first()
            if builds_task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
                return branch_context

            if builds_task.status != Status.SUCCESS:
                branch_context['problems'].append(
                    u'Таска подготовки сборок {} не успешна'.format(self.task_href(builds_task.id)))
                branch_context['complete_status'] = True
                return branch_context

            if branch_context["autotests_task_id"] is None:
                autotests_task = self.launch_autotests(builds_task, project, branch)
                branch_context["autotests_task_id"] = autotests_task.id
                return branch_context

            autotests_task = sdk2.Task.find(id=branch_context["autotests_task_id"], children=True).limit(1).first()
            if autotests_task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
                return branch_context

            # finish
            branch_context['complete_status'] = True
            if autotests_task.status != Status.SUCCESS:
                branch_context['problems'].append(
                    u'Таска автотестов {} не успешна'.format(self.task_href(autotests_task.id)))
            branch_context['autotests_report_url'] = autotests_task.Context.report_url
            if not branch_context['autotests_report_url']:
                branch_context['problems'].append(
                    u'Отчет автотестов {} не найден'.format(self.task_href(autotests_task.id)))
            else:
                try:
                    task_type = RunDesktopBrowserAutotests if project in ["desktop_win", "desktop_mac"] else RunBrowserAutotests
                    autotests_result_resource = AutotestsReportResource.find(
                        task=autotests_task.find(task_type=task_type).first(),
                        state=ctr.State.READY).first()
                    results = IsolatesAutotestsResult(str(sdk2.ResourceData(autotests_result_resource).path),
                                                      self.clients)
                    branch_context['html_summary'] = results.report.html_summary
                    # branch_context['problems'] += results.report.problems
                except Exception as e:
                    branch_context['problems'].append(
                        u'Не удалось обработать результат {} {}'.format(self.task_href(autotests_task.id), e.message))
                    return branch_context

        return branch_context

    def complete_all_tasks(self):
        result = True
        for project, project_branches in self.Context.projects.iteritems():
            for branch in project_branches:
                result = self.process(project, branch)["complete_status"] and result
        return result

    def on_execute(self):

        with self.memoize_stage.check_projects:
            unknown_projects = [_p for _p in self.projects_and_builds if _p not in RELEASES_MAP]
            if unknown_projects:
                self.set_info(
                    u'<font color="red"><b>Внимание!</b></font> Проекты {} отсутствуют в списке ожидаемых. Могут возникнуть странности<br/>'.format(
                        u",".join(unknown_projects)), do_escape=False)

        with self.memoize_stage.run_builds:
            self.prepare_builds()

        if not self.complete_all_tasks():
            raise sdk2.WaitTime(60 * 10)

        # report
        info = u''
        fail_task = False
        for project, project_branches in self.Context.projects.iteritems():
            info += u'<ul><li><big><b>{}</b></big>'.format(project)
            for branch, branch_context in project_branches.iteritems():
                info += u'<ul><li><b>{}</b>'.format(branch)
                info += u'<ul>'

                if branch_context['problems']:
                    info += u'<li><font color="red"><b>Проблемы:</b></font>'
                    info += u'<ul>'
                    for item in branch_context['problems']:
                        info += u'<li>{}</li>'.format(item)
                    info += u'</ul>'
                    info += u'</li>'
                    fail_task = True
                info += u'<li>Сборки: {}</li>'.format(self.task_href(branch_context['builds_task_id']))
                info += u'<li>Автотесты: {}</li>'.format(
                    self.task_href(branch_context['autotests_task_id']) if branch_context['autotests_task_id'] else u"не запущены")
                info += u'<li>Отчет автотестов: {}</li>'.format(
                    u'<a href="{href}">{href}</a>'.format(
                        href=branch_context['autotests_report_url']) if branch_context['autotests_report_url'] else u"нет")
                # info += u'<li>Результаты:{}</li>'.format(branch_context['html_summary'])
                info += u'</ul>'
                info += u'</li></ul>'
            info += u'</li></ul>'

        self.set_info(info, do_escape=False)
        self.launch_testpalm_analyst_task()
        if fail_task:
            raise TaskFailure(u"Что то пошло не так. См сообщения о проблемах")

    def launch_testpalm_analyst_task(self):

        def _get_buiils(data):
            builds_tasks_ids = [
                self.Context.projects.get(item, {}).get(MASTER_BRANCH, {}).get("builds_task_id") for item in data["tests_builds"]
            ]
            if not any(builds_tasks_ids):
                return None
            tests_builds = []
            for _task_id in builds_tasks_ids:
                _task = sdk2.Task.find(id=_task_id, children=True).limit(1).first()
                if not _task or _task.status != Status.SUCCESS:
                    return None
                tests_build_id = _task.Context.teamcity_builds.get("autotests")
                if not tests_build_id:
                    return None
                tests_builds.append(tests_build_id)
            return tests_builds

        for name, data in TESTPALM_ANALYST_PROJECTS.iteritems():
            tests_builds = _get_buiils(data)
            if not tests_builds:
                self.set_info(u" Не удалось запустить таск актуализации разметки кейсов для группы {}, то то не так со сборками".format(name))
                continue
            task = BrowserTestpalmAnalyst(
                self,
                testpalm_projects=data["testpalm_projects"],
                tests_builds=tests_builds,
                desktop_autotests_branch=data["desktop_autotests_branch"],
                consider_platforms=data["consider_platforms"],
                treat_blacklisted_cases_as_automated=data["treat_blacklisted_cases_as_automated"],
                auto_set_ready_status=data["auto_set_ready_status"],
                report_recipients=data["report_recipients"]
            ).enqueue()
            self.set_info(u'Создана таска актуализации разметки {}-кейсов: {}'.format(name, self.task_href(task.id)),
                          do_escape=False)
