"""
Author: Olga Kochetova <myxomopla@yandex-team.ru>
"""
import logging
import re
from collections import defaultdict
from copy import deepcopy
import datetime
import requests
import time

import sandbox
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.autotests.BrowserAutotestRun import BrowserAutotestRun
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.common.errors import TaskFailure
from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserAutotestNightlyRun import BrowserAutotestNightlyRun, TIMEZONE
import sandbox.common.types.client as ctc

ISOLATES_BUILD_TYPE = 'Browser_Tests_Build_{}'

BROWSER_BRANDED_BETA = {
    'mac': 'BrowserBrandedDistributives_BrandedBetaMac',
    'win': 'BrowserBrandedDistributives_BrandedBetaWin',
    'linux': 'BrowserBrandedDistributives_BrandedBetaLinuxNew'
}

DEFAULT_AUTOTESTS_SET = 'short_nightly_{}.json'
FULL_AUTOTEST_SET = 'all_nightly_beta_{}.json'
RELEASE_ID_RE = re.compile(r'master-(?P<release>\d+\.\d+)\.\d+/rc')
PRE_ID_RE = re.compile(r'master-(?P<release>\d+\.\d+)/pre')

MAX_BRANCHES_WITH_FULL_AUTOTESTS = 3


def get_current_releases():
    releases_url = ('https://browser.yandex-team.ru/rest/releases/'
                    '?show_only_active=true&include_milestones=true')
    current_timestamp = int(time.mktime(datetime.datetime.today().timetuple()))
    logging.debug('current timestamp %s', current_timestamp)
    current_releases = requests.get(releases_url).json()['items']
    logging.debug('current releases %s', current_releases)
    current_releases = [
        release for release in current_releases
        if any(milestone['kind'] == 'release:rc' and milestone['date'] > current_timestamp
               for milestone in release['milestones'])
    ]
    res = sorted(current_releases, key=lambda release: release['milestones'][0]['date'])
    logging.debug('filtered current releases %s', res)
    return res


# TODO remove it, dublicate
def get_framework_branch(browser_branch):
    if browser_branch in ['master', 'master-next', 'master-mobile', '<default>']:
        return 'master'

    release_branch_match = re.match(RELEASE_ID_RE, browser_branch)
    if release_branch_match:
        return 'feature/{}'.format(release_branch_match.group('release'))
    pre_branch_match = re.match(PRE_ID_RE, browser_branch)
    if pre_branch_match:
        return 'feature/{}'.format(pre_branch_match.group('release'))

    raise TaskFailure('Unknown type of branch: {}'.format(browser_branch))


class BrowserReleaseBuildsNightlyTrigger(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.BROWSER

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        notifications = (
            sdk2.Notification(
                [Status.Group.FINISH, Status.Group.BREAK],
                ['browser-autotests-bots'],
                ctn.Transport.EMAIL
            )
        )
        launch_autotests_delay = sdk2.parameters.Integer(
            'Delay before running autotests after browser builds', default=None,
        )

    class Context(sdk2.Context):
        child_autotest_tasks = {}

    def task_href(self, task):
        return '<a href={link}>task</a>'.format(link=get_task_link(task.id))

    @property
    @sandbox.common.utils.singleton
    def bitbucket_client(self):
        return bitbucket.BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                                   sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    def on_execute(self):
        if not self.Context.child_autotest_tasks:
            created_tasks = defaultdict(dict)
            launch_time = None
            now = datetime.datetime.now(TIMEZONE)
            if self.Parameters.launch_autotests_delay:
                launch_time = now + datetime.timedelta(hours=self.Parameters.launch_autotests_delay)

            existing_branches = [branch_info['displayId'] for branch_info in
                                 self.bitbucket_client.get_branches('stardust', 'browser', 'master', 'MODIFICATION')]

            releases = [('master', ['win', 'mac', 'linux'])] + [('master-next', ['win', 'mac'])] + [
                (r['real_branch'], r['platforms']) for r in get_current_releases() if r['real_branch'] not in ["master", "master-next"]]

            full_autotests_launch_count = 0
            for branch, platforms in releases:
                if branch not in existing_branches:
                    continue
                should_run_full_autotests = full_autotests_launch_count < MAX_BRANCHES_WITH_FULL_AUTOTESTS
                full_autotests_launch_count += 1
                common_params = {
                    'browser_branch': branch,
                    'framework_branch': get_framework_branch(branch),
                    'teamcity_builds_comment': 'Nightly branded beta',
                    'teamcity_builds_tags': ['n-beta'],
                    'browser_build_params': {'partner-names': '@nightly',
                                             'brand-names': 'yandex int tb by'},
                    'launch_autotests_at': launch_time,
                    'check_tests_status': not should_run_full_autotests,
                    'notifications': (
                        sdk2.Notification([Status.Group.BREAK], ['browser-autotests-bots'], ctn.Transport.EMAIL)
                    ),
                }
                for platform in platforms:
                    platform = platform.capitalize()
                    params = deepcopy(common_params)
                    #  see BYIN-12370
                    if platform.lower() == 'mac':
                        params['browser_build_params'].update({'sandbox.disable_notarization': True})
                    params.update({
                        'platform': platform,
                        'browser_build_type_id': BROWSER_BRANDED_BETA[platform.lower()],
                        'isolates_build_type_id': ISOLATES_BUILD_TYPE.format(platform),
                        'config_file': (FULL_AUTOTEST_SET if should_run_full_autotests
                                        else DEFAULT_AUTOTESTS_SET).format(
                            platform.lower()) if platform != 'Linux' else None,
                        'without_fake': not should_run_full_autotests or platform == 'Linux',
                        'tags': ['browser-autotests-nightly-{}'.format(platform),
                                 'browser-autotests-nightly-{}'.format(branch)],
                    })
                    autotest_task = BrowserAutotestNightlyRun(self, **params)
                    created_tasks[branch][platform] = autotest_task.enqueue()
                self.Context.child_autotest_tasks = {branch: {platform_: task.id
                                                              for platform_, task in tasks.iteritems()}
                                                     for branch, tasks in created_tasks.iteritems()}
            raise sdk2.WaitTask(
                tasks=[task_id for branch, tasks in created_tasks.iteritems() for _, task_id in tasks.items()],
                statuses=list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True,
                timeout=(6 + self.Parameters.launch_autotests_delay
                         if self.Parameters.launch_autotests_delay else 12) * 60 * 60
            )
        info = ''
        fail_task = False
        for branch, tasks in self.Context.child_autotest_tasks.iteritems():
            info += '<p> <b>{}:</b><ul>'.format(branch)
            for platform, task_id in tasks.iteritems():
                info += '<li>'
                task = sdk2.Task[task_id]
                timeouted = task.status not in list(Status.Group.FINISH + Status.Group.BREAK)
                autotest_task = task.find(BrowserAutotestRun).first()
                info += '{platform} {task} is in status {status}{timeouted}.{allure}'.format(
                    platform=platform.upper(), task=self.task_href(task),
                    status="<span class='status status_{}'>{}</span>".format(task.status.lower(), task.status),
                    timeouted=' (timeouted)' if timeouted else '',
                    allure=' <a href="{}">Allure report</a>'.format(autotest_task.Context.report_url)
                    if autotest_task and autotest_task.Context.report_url else ''
                )
                if task.Context.build_problems:
                    info += '<ul>'
                    for build_problem in task.Context.build_problems:
                        info += '<li>{}</li>'.format(build_problem)
                    info += '</ul>'
                    fail_task = True
                if timeouted or task.status != Status.SUCCESS:
                    fail_task = True
                info += '</li>'
            info += '</ul></p>'
        self.set_info(info, do_escape=False)
        if fail_task:
            raise TaskFailure("Smth happened during autotests launching")
