# coding=utf-8
import json
import logging
import os
import yaml
from collections import defaultdict, OrderedDict

import jinja2

import sandbox

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.browser.autotests.classes.functionalities import get_all_layer_functionalities
from sandbox.projects.browser.autotests.configs.functionalities import UPLOAD_CONFIG_NAME
from sandbox.projects.browser.autotests.testpalm_helper import TestpalmClientWrapper
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT, is_dev_sandbox

TESTPALM_PROJECTS_TEST = ['test_project']
TESTPALM_FIELDS = ['Functionality', 'Depends on Functionality']

FUNCTIONALITIES_CONFIGS_PATH = os.path.join(__file__, '..', '..', 'configs', 'functionalities')


class BrowserFunctionalitiesHtml(sdk2.Resource):
    ttl = 14


def group_functionalities_by_component(functionalities_with_descriptions):
    """
    :param functionalities_with_descriptions: dict {str: str}
    :return:
    """
    functionalities = get_all_layer_functionalities(functionalities_with_descriptions.keys())
    res = defaultdict(set)
    for functionality in functionalities:
        res[functionality.component].add(str(functionality))
    return {
        component: OrderedDict(
            [(f, functionalities_with_descriptions.get(f))
             for f in sorted(res[component])
             if f != component]
        )
        for component in res
    }


def get_functionalities_descriptions(project):
    functionalities = {}
    for r, d, f in os.walk(os.path.abspath(os.path.join(FUNCTIONALITIES_CONFIGS_PATH, project))):
        for component_file in f:
            if component_file.endswith('.yaml'):
                with open(os.path.join(r, component_file), 'r') as f:
                    functionalities_from_file = yaml.load(f)
                    if functionalities_from_file:
                        functionalities.update(functionalities_from_file)
    return functionalities


def get_testpalm_projects(project):
    with open(os.path.abspath(os.path.join(FUNCTIONALITIES_CONFIGS_PATH, project, UPLOAD_CONFIG_NAME)), 'r') as f:
        return json.load(f)['testpalm_projects']


class BrowserUpdateFunctionalities(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        update_definitions = sdk2.parameters.Bool(
            'Update projects definitions', default=True,
            description='If not set, then task will only generate html files,'
                        ' without changing testpalm projects definitions.'
                        ' This makes sense only for testing purposes')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('testpalm-api-client', version='4.0.2'),
        ]
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='browser-update-functionalities')
            ],
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    @sandbox.common.utils.singleton_property
    def result_folder(self):
        os.mkdir(str(self.path('htmls')))
        return self.path('htmls')

    def save_html(self, functionalities_with_descriptions, project):
        grouped_functionalities = group_functionalities_by_component(functionalities_with_descriptions)

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        page = env.get_template('functionalities.jinja').render(functionalities=grouped_functionalities)

        resource_path = str(self.result_folder.joinpath(project + '.html'))
        with open(resource_path, 'w') as f:
            f.write(page)

    def update_definitions(self, functionalities_from_configs, projects):
        functionalities_from_configs = set(functionalities_from_configs)
        for project in projects:
            definitions = {definition['title']: definition
                           for definition in self.testpalm_client.get_definitions(project)}
            for field in TESTPALM_FIELDS:
                definition = definitions[field]
                logging.debug('For project %s in field %s got values %s', project, field, definition['values'])
                new_functionalities = functionalities_from_configs - set(definition['values'])
                if new_functionalities:
                    logging.debug('New functionalities from config: %s', new_functionalities)
                    definition['values'].extend(new_functionalities)
                    self.testpalm_client.update_definition(definition, project)
                else:
                    logging.debug('No new functionalities from config for this field')

    def on_execute(self):
        for project_folder in next(os.walk(os.path.abspath(FUNCTIONALITIES_CONFIGS_PATH)))[1]:
            functionalities_with_descriptions = get_functionalities_descriptions(project_folder)
            if self.Parameters.update_definitions:
                testpalm_projects = (TESTPALM_PROJECTS_TEST if is_dev_sandbox()
                                     else get_testpalm_projects(project_folder))
                self.update_definitions(functionalities_with_descriptions.keys(), testpalm_projects)
            self.save_html(functionalities_with_descriptions, project_folder)

        resource = BrowserFunctionalitiesHtml(self, 'Browser functionalities', self.result_folder)
        sdk2.ResourceData(resource).ready()

    @sandbox.common.utils.singleton_property
    def testpalm_client(self):
        from testpalm_api_client.client import TestPalmClient
        return TestpalmClientWrapper(TestPalmClient(oauth_token=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)))
