"""
Author: Nikolay v Isaev <nik-isaev@yandex-team.ru>
"""
import os

import sandbox
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.classes.autotests_bundle import DesktopAutotestsBouldle
from sandbox.projects.browser.autotests.classes.autotests_result import BtrReport, TestStatuses
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.RunDesktopBrowserAutotests import RunDesktopBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsReportResource

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.task import Status
import sandbox.common.types.client as ctc
from sandbox.projects.common.teamcity import TeamcityArtifacts
from sandbox.sandboxsdk.environments import PipEnvironment


LAUNCH_CONFIGS = {
    'branded.beta.win': {
        'python_win_10_x64': [
            'brocase-13918',
            'brocase-13919',
        ]
    },
    'branded.dev.win': {
        'python_win_10_x64': [
            'brocase-14362',
            'brocase-14364',
        ]
    },
    'branded.canary.win': {
        'python_win_10_x64': [
            'brocase-14363',
            'brocase-14365',
        ]
    }
}


class RunBroteamBrowserAutotests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        teamcity_build_id = sdk2.parameters.Integer('Teamcity build id', required=True)
        platform = sdk2.parameters.String('Platform', choices=[(_, _) for _ in ['win', 'mac']], default='win')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    class Requirements(sdk2.Requirements):

        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [PipEnvironment('teamcity-client==3.0.0'),
                        PipEnvironment('PyYAML', version='3.11'),
                        PipEnvironment('junit-xml', version='1.9')]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    @property
    @common.utils.singleton
    def builds(self):
        names_map = {
            "build_tests.win": "tests_build",
            'branded.beta.win': "browser_build",
            'branded.dev.win': "browser_build",
            'branded.canary.win': "browser_build"
        }
        result = {}
        for build in self.clients.teamcity.Build(id=self.Parameters.teamcity_build_id).snapshot_dependencies:
            if build.build_type.name in names_map:
                if build.status != 'SUCCESS':
                    raise TaskFailure("Build: #{} status not SUCCESS".format(build.id))
                if result.get(names_map[build.build_type.name]) is None:
                    result[names_map[build.build_type.name]] = build
                else:
                    raise ValueError(
                        'Duplication in {}. There are probably incorrect dependencies'.format(names_map[build.build_type.name]))
        return result

    def on_execute(self):
        from junit_xml import TestSuite

        child = self.find().first()
        if not child:
            if not all(_b_type in self.builds for _b_type in ['tests_build', 'browser_build']):
                raise TaskFailure("Required snapshot_dependencies builds not found")

            bundle = DesktopAutotestsBouldle(self.builds['tests_build'], self.builds['browser_build'],
                                             None, self.clients)
            child = RunDesktopBrowserAutotests(
                self,
                description=self.Parameters.description,
                launch_config=LAUNCH_CONFIGS[self.builds['browser_build'].build_type.name],
                browser_tests_build_id=self.builds['tests_build'].id,
                branch=bundle.browser_branch,
                commit=bundle.browser_commit,
                build_extra_args=bundle.build_extra_args,
                ignore_blacklists=False
            )
            raise sdk2.WaitTask(
                child.enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True
            )
        artifacts_path = str(self.path('artifacts'))
        os.mkdir(artifacts_path)
        junits_path = os.path.join(artifacts_path, 'junit_reports')
        os.mkdir(junits_path)

        resource = AutotestsReportResource.find(task=child).first()
        btr_report = BtrReport(str(sdk2.ResourceData(resource).path), self.clients)
        if not btr_report.tests:
            raise TaskFailure("No tests found")
        xml_results = btr_2_junit_xml_string(btr_report._data)
        for platform, _ts in xml_results.iteritems():
            with open(os.path.join(junits_path, '{}_junit.xml'.format(platform)), "w") as _f:
                TestSuite.to_file(_f, _ts, prettyprint=False)

        with open(os.path.join(artifacts_path, 'allure_report.html'), 'w') as f:
            f.write('Allure report is not available')
        sdk2.ResourceData(TeamcityArtifacts(self, 'Junit reports', artifacts_path)).ready()


def btr_2_junit_xml_string(btr_report_data):
    from junit_xml import TestSuite, TestCase

    result = {}
    for platform, data in btr_report_data.iteritems():
        _tests = []
        for build_id, cases in data.iteritems():
            for case_id, tests in cases.iteritems():
                for test in tests:
                    _case = TestCase(
                        name=u"{}.{}.{}".format(case_id, test['binary'], test['name']),
                        log=test.get("log", "")
                    )
                    if test['status'] not in TestStatuses.PASSED.value:
                        if test['status'] in TestStatuses.SKIPPED.value:
                            _case.add_skipped_info("skipped")
                        else:
                            _case.add_failure_info("Test status: {} not success".format(test['status']))
                    _tests.append(_case)
        result[platform] = [TestSuite("All {} tests".format(platform), _tests)]
    return result
