# -*- coding: utf-8 -*-
import re

import sandbox
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.classes.autotests_bundle import (
    IsolateAutotestBundle, AndroidAutotestsBundle, DesktopAutotestsBouldle, SearchappIsolateAutotestBundle,
    AndroidSearchappAutotestBundle)
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.CollectAutotestsStatistics import CollectAutotestsStatistics
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests.regression_tasks.RunDesktopBrowserAutotests import RunDesktopBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import (
    html_link, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT, DEFAULT_MAX_FLAKY_RETRIES)
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds
from sandbox.sandboxsdk.environments import PipEnvironment


class RunNightlyBrowserAutotests(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
        ]

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    class Context(sdk2.Context):
        autotests_task_id = None
        report_url = None

    class Parameters(sdk2.Parameters):

        browser_tests_build_id = sdk2.parameters.Integer(
            'Tests build id',
            description='Id of teamcity build with browser tests',
            required=True)
        browser_build_id = sdk2.parameters.Integer(
            'Brended or apk build id',
            description='Id of teamcity build with browser',
            default=0)
        fake_build_id = sdk2.parameters.Integer(
            'Fake build id',
            description='Id of teamcity build with fake browser',
            default=0)
        launch_config = sdk2.parameters.JSON(
            'Launch config',
            description='Launch config for yandex.autotests.binary.run_tests script. Leave blank to run all tests'
        )
        max_flaky_retries = sdk2.parameters.Integer(
            'Number of retries flaky tests',
            default=DEFAULT_MAX_FLAKY_RETRIES
        )
        generate_allure_reports = sdk2.parameters.Bool(
            'Generate allure reports', default=False)
        timeout = sdk2.parameters.Integer(
            'Timeout',
            description='Timeout in minutes',
            default=DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT
        )
        statistics_launch_type = sdk2.parameters.String(
            'Launch type for statistics',
            description=u'Раздел статистики запусков автотестов: Nightly, Regression, Pull-request, Custom. Если поле пусто - статистика не отправится.',
            default=""
        )
        tested_application = sdk2.parameters.String(
            'Tested application: browser or searchapp', default='browser')

    @property
    @sandbox.common.utils.singleton
    def builds(self):
        result = {"tests_build": self.clients.teamcity.Build(id=self.Parameters.browser_tests_build_id)}
        if self.Parameters.browser_build_id:
            result["browser_build"] = self.clients.teamcity.Build(id=self.Parameters.browser_build_id)
        if self.Parameters.fake_build_id:
            result["fake_build"] = self.clients.teamcity.Build(id=self.Parameters.fake_build_id)
        return result

    def wait_builds(self, builds):
        if any(b.state != 'finished' for b in builds):
            raise sdk2.WaitTask(
                BrowserWaitTeamcityBuilds(
                    self,
                    description='Wait browser regression builds',
                    mode='WAIT_GIVEN',
                    builds=' '.join(str(build.id) for build in builds),
                    oauth_vault=ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
                ).enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )
        else:
            if any(b.status != 'SUCCESS' for b in builds):
                raise TaskFailure('Одна из сборок упала')

    def get_autotests_bundle(self):
        # Android
        if re.match(re.compile(r".*\.android", re.IGNORECASE), self.builds["tests_build"].build_type.name):
            if self.Parameters.tested_application == 'searchapp':
                return AndroidSearchappAutotestBundle(self.builds["tests_build"],
                                                      self.builds.get("browser_build"),
                                                      self.builds.get("fake_build"),
                                                      self.clients)

            if not self.builds.get("browser_build"):
                raise TaskFailure(u"Для запуска Android-тестов браузера требуется browser_build")
            return AndroidAutotestsBundle(self.builds["tests_build"], self.builds["browser_build"], self.clients)
        # Desktop
        elif self.builds["tests_build"].build_type.name in ['build_tests.win', 'build_tests.mac']:
            return DesktopAutotestsBouldle(self.builds["tests_build"],
                                           self.builds.get("browser_build"),
                                           self.builds.get("fake_build"),
                                           self.clients)
        # Any searchapp
        elif self.Parameters.tested_application == 'searchapp':
            return SearchappIsolateAutotestBundle(self.builds["tests_build"], self.clients)
        # Default
        else:
            return IsolateAutotestBundle(self.builds["tests_build"], self.clients)

    def on_execute(self):
        self.wait_builds([_b for _b in self.builds.values() if _b])
        with self.memoize_stage.launch_and_wait_autotests:
            autotests_bundle = self.get_autotests_bundle()
            launch_config = self.Parameters.launch_config or autotests_bundle.get_launch_config_all_cases(
                ignore_blacklists=False)
            # launch autotestst
            task_type = RunDesktopBrowserAutotests if isinstance(autotests_bundle, DesktopAutotestsBouldle) else RunBrowserAutotests
            autotests_task = task_type(
                self,
                description=u"Nightly autotests",
                launch_config=launch_config,
                browser_tests_build_id=self.Parameters.browser_tests_build_id,
                tested_application=autotests_bundle.tested_application,
                build_extra_args=autotests_bundle.build_extra_args,
                number_of_shards=autotests_bundle.number_of_shards,
                max_flaky_retries=self.Parameters.max_flaky_retries,
                branch=autotests_bundle.browser_branch,
                commit=autotests_bundle.browser_commit,
                timeout=self.Parameters.timeout,
                kill_timeout=(self.Parameters.timeout + 30) * 60,
                generate_allure_reports=self.Parameters.generate_allure_reports,
                use_test_bitbucket=False,
                tags=self.Parameters.tags).enqueue()
            self.set_info(
                u'Запуск автотестов: {}'.format(html_link(get_task_link(autotests_task.id))),
                do_escape=False)
            self.Context.autotests_task_id = autotests_task.id

            # Enqueue autotests statistics task
            if self.Parameters.statistics_launch_type:
                CollectAutotestsStatistics(
                    self,
                    autotests_task=autotests_task,
                    launch_type=self.Parameters.statistics_launch_type,
                    tags=self.Parameters.tags
                ).enqueue()

            raise sdk2.WaitTask(autotests_task,
                                [Status.Group.FINISH, Status.Group.BREAK])

        errors = []
        autotests_task = sdk2.Task.find(id=self.Context.autotests_task_id, children=True).limit(1).first()
        if autotests_task.status != Status.SUCCESS:
            errors.append(u'Таска  автотестов {} не успешна статус={}'.format(
                html_link(get_task_link(autotests_task.id)),
                autotests_task.status))
        self.Context.report_url = autotests_task.Context.report_url
        if not self.Context.report_url:
            errors.append(u"Отчет автотестов не найден")
        else:
            self.set_info(u"Отчет автотестов: {}".format(self.Context.report_url))
        if errors:
            self.set_info(u"Имеются проблемы: <br>{}".format(u"<br>".join(errors)))
            raise TaskFailure()
