"""
Author: Nikolay V. Isaev <nik-isaev@yandex-team.ru>
"""
import logging

from sandbox import common
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.BrowserAutotestRun import BrowserAutotestRun
from sandbox.projects.browser.autotests_qa_tools.common import SPECIFICATION_NAME
from sandbox.projects.browser.autotests.UpdateBrowserAutotestsSpecification import UpdateBrowserAutotestsSpecification
from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsAllureData
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL, BitBucket
from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.types import resource as ctr
from sandbox.projects.common import decorators


class RunUpdateBrowserAutotestsSpecification(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):

        framework_commit = None
        changes_detected = None

    class Parameters(sdk2.Parameters):

        framework_branch = sdk2.parameters.String('Autotests framework branch. For example feature/19.9', required=True)

        specification_file = sdk2.parameters.String(
            'Autotests specification file name',
            default=SPECIFICATION_NAME)
        force = sdk2.parameters.Bool('Force update/create specification', default=False)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity and bb',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @common.utils.singleton
    def bitbucket_client(self):
        return BitBucket(DEFAULT_BITBUCKET_URL,
                         'x-oauth-token',
                         sdk2.Vault.data(self.Parameters.oauth_vault))

    @common.utils.singleton
    def get_framework_commit(self):
        if self.Context.framework_commit is None:
            self.Context.framework_commit = self.get_latest_commit(self.Parameters.framework_branch)
        return self.Context.framework_commit

    @decorators.retries(5, delay=2, backoff=2)
    def get_latest_commit(self, branch):
        return self.bitbucket_client.get_latest_commit(
            FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, branch)

    def is_changes_detected(self):

        if self.Context.changes_detected is None:
            framework_commit = self.get_framework_commit()
            bundle_commit = self.get_latest_commit("bundle/{}".format(self.Parameters.framework_branch))
            logging.debug(' ---- framework_commit:{} bundle_commit:{}'.format(framework_commit, bundle_commit))
            merge_base = self.bitbucket_client.merge_base(FRAMEWORK_BITBUCKET_PROJECT,
                                                          FRAMEWORK_BITBUCKET_REPO,
                                                          [framework_commit, bundle_commit])
            logging.debug(' ---- merge_base commit:{} '.format(merge_base))

            self.Context.changes_detected = not bool(merge_base == framework_commit)

        return self.Context.changes_detected

    def on_execute(self):

        if self.Parameters.force or self.is_changes_detected():
            with self.memoize_stage.run_autotests:
                raise sdk2.WaitTask(
                    BrowserAutotestRun(
                        self,
                        description='Get fake report for autotests specification',
                        framework_branch=self.Parameters.framework_branch,
                        framework_commit=self.get_framework_commit(),
                        build_id=123,  # use a fake random build_id because the tests will not really run
                        fake_build_id=123,
                        browser_tests_build_id=123,
                        config_file=self.Parameters.specification_file,
                        browser_version=123,
                        tags=['Update_Browser_Autotests_Specification', self.Parameters.framework_branch.replace("/", "_")]
                    ).enqueue(),
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    True,
                )

            for child in self.find():
                if child.status != Status.SUCCESS:
                    raise TaskFailure('Child task {} is not successful'.format(child.type))

            with self.memoize_stage.update_specification:

                allure_report = AutotestsAllureData.find(task=child, state=ctr.State.READY).first()

                raise sdk2.WaitTask(
                    UpdateBrowserAutotestsSpecification(
                        self,
                        allure_report=allure_report,
                        framework_branch=self.Parameters.framework_branch,
                        framework_commit=self.get_framework_commit(),
                        specification_file=self.Parameters.specification_file,
                        tags=['Update_Browser_Autotests_Specification', self.Parameters.framework_branch.replace("/", "_")]
                    ).enqueue(),
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    True,
                )
        else:
            self.set_info("No changes detected for {}".format(self.Parameters.framework_branch))
