# -*- coding: utf-8 -*-
"""
Author: Nikolay V. Isaev <nik-isaev@yandex-team.ru>
"""
import json
import time
import os
import shutil

from collections import OrderedDict
from datetime import datetime

from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, ROBOT_BRO_QA_INFRA_SSH_KEY_VAULT, extract_zip_resource,
    FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, FRAMEWORK_DEFAULT_REVIEWERS,  SPECIFICATION_NAME,
    SPECIFICATION_PATH)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsAllureData
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox import common
from sandbox import sdk2

from sandbox.sdk2.helpers import process
from sandbox.projects.browser.autotests.allure_parser import AllureReport
from sandbox.projects.browser.common.git import repositories, ConfigureGitEnvironment, GitEnvironment
from sandbox.projects.common import decorators


@decorators.retries(3, delay=30, backoff=2, exceptions=process.subprocess.CalledProcessError)
def retried_git(*args, **kwargs):
    git_check(*args, **kwargs)


def git_check(*args, **kwargs):
    command_line = ['git'] + list(args)
    with process.ProcessLog(task=sdk2.task.Task.current, logger='git') as pl:
        process.subprocess.check_call(command_line, stdout=pl.stdout, stderr=pl.stderr, **kwargs)


def git_out(*args, **kwargs):
    command_line = ['git'] + list(args)
    return process.subprocess.check_output(command_line, stderr=process.subprocess.STDOUT, **kwargs)


def _get_specification(launch_config_template, allure_report):

    cases_spec = {}
    for test in allure_report.tests:

        test_id = test.test_id
        if test_id is None:
            continue

        case_data = cases_spec.setdefault(test_id, {"platforms": {}})
        case_platform_data = case_data["platforms"].setdefault(test.platform, {"packs": [], "blacklisted": False})

        case_platform_data["packs"] = list(set(case_platform_data["packs"] + [test.test_environment_parameters.get("autotests_pack_name")]))
        case_platform_data["blacklisted"] = case_platform_data["blacklisted"] or test.test_environment_parameters.get("blacklisted")

    return dict(launch_config=launch_config_template,
                cases=_sort_dict(cases_spec))


def _sort_dict(source):
    if isinstance(source, list):
        return sorted(source)
    elif isinstance(source, dict):
        _res = OrderedDict()
        for key in sorted(source.keys()):
            _res[key] = _sort_dict(source[key])
        return _res
    else:
        return source


class UpdateBrowserAutotestsSpecification(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = (
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment(email='robot-bro-qa-infra@yandex-team.ru', username='robot-bro-qa-infra'),
        )

    class Context(sdk2.Context):

        changes_branch = ''
        allure_report_path = ''

    class Parameters(sdk2.Parameters):

        allure_report = sdk2.parameters.Resource('Allure report', required=True, resource_type=AutotestsAllureData)
        framework_branch = sdk2.parameters.String('Autotests framework branch. For example feature/19.9', required=True)
        framework_commit = sdk2.parameters.String(
            'Autotests framework commit', required=True)
        specification_file = sdk2.parameters.String(
            'Autotests specification file name',
            default=SPECIFICATION_NAME)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity and bb',
                                                 default=ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
            ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key for bitbucket',
                default=ROBOT_BRO_QA_INFRA_SSH_KEY_VAULT)

    @property
    @common.utils.singleton
    def bitbucket_client(self):
        return bitbucket.BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                                   sdk2.Vault.data(self.Parameters.oauth_vault))

    @property
    @common.utils.singleton
    def repo(self):
        return repositories.Autotest.browser_test_framework(
            filter_branches=False)

    def framework_repo_path(self, *args):
        return str(self.path('browser-test-framework', *args))

    def prepare_repo(self):

        self.Context.changes_branch = "wp/robots/update_specifications/{}/{}-{}".format(
            self.Parameters.framework_branch.replace("/", "-"),
            datetime.now().date(),
            str(time.time()).replace(".", ""))
        self.Context.bundle_branch = "bundle/{}".format(self.Parameters.framework_branch)

        # FIXME update_cache is used here, because without it cloning branch fails on branch, which is a suffix of
        # another branch. e.g. cloning feature/19.10 fails if branch bundle/feature/19.10 exists.
        self.repo.update_cache_repo()
        self.repo.clone(self.framework_repo_path(), self.Context.bundle_branch)

        git_check('checkout', '-b', self.Context.changes_branch,
                  cwd=self.framework_repo_path())

        # backup specifications
        shutil.copytree(self.framework_repo_path(SPECIFICATION_PATH),
                        'specifications_backup')

        git_check('fetch', 'origin', self.Parameters.framework_branch, cwd=self.framework_repo_path())
        git_check('merge', '--no-commit', self.Parameters.framework_commit, cwd=self.framework_repo_path())

        # restore specifications
        if os.path.isdir(self.framework_repo_path(SPECIFICATION_PATH)):
            shutil.rmtree(self.framework_repo_path(SPECIFICATION_PATH))
        shutil.copytree('specifications_backup',
                        self.framework_repo_path(SPECIFICATION_PATH))

        git_check('add', self.framework_repo_path(SPECIFICATION_PATH),
                  cwd=self.framework_repo_path())

    def updale_local_specification_file(self):

        with open(
                self.framework_repo_path('launch_configs', self.Parameters.specification_file),
                'r') as config_file:
            config_temlpate = json.load(config_file)

        specification = _get_specification(config_temlpate, AllureReport(
            self.Context.allure_report_path, None))

        with open(
                self.framework_repo_path(SPECIFICATION_PATH, self.Parameters.specification_file),
                'w') as specification_file:
            json.dump(specification, specification_file, indent=4)

        return bool(
            git_out("diff", "HEAD", "--name-only", cwd=self.framework_repo_path())
        )

    def commit_changes_and_create_pr(self):

        with sdk2.ssh.Key(self, self.Parameters.ssh_key_vault, None):

            git_check('commit', "-a", '-m', '"update tests specifications"',
                      cwd=self.framework_repo_path())

            retried_git('push', '--set-upstream', 'origin', self.Context.changes_branch,
                        cwd=self.framework_repo_path())

        pr = self.bitbucket_client.create_pr(
            FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO,
            "update tests specifications",
            "// startrek-update-only-from:{}Update tests specifications for {}".format(os.linesep, datetime.now().date()),
            self.Context.changes_branch,
            self.Context.bundle_branch,
            FRAMEWORK_DEFAULT_REVIEWERS)

        self.set_info(
            'PR created: <a href="{}">{}</a>'.format(pr.web_url, pr.web_url),
            False)

    def on_execute(self):

        self.Context.allure_report_path = extract_zip_resource(
            self,
            str(sdk2.ResourceData(self.Parameters.allure_report).path))

        self.prepare_repo()

        has_changes = self.updale_local_specification_file()
        if has_changes:
            self.commit_changes_and_create_pr()
        else:
            self.set_info('There are no changes for the {} branch'.format(self.Context.bundle_branch))
