# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link

import sandbox
from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.autoduty_tasks.DBroAutotestsToIssues import DBroAutotestsToIssues

from sandbox.projects.browser.autotests.regression_tasks.RunDesktopBrowserAutotests import RunDesktopBrowserAutotests
from sandbox.projects.browser.autotests.classes.autotests_bundle import get_desktop_autotest_bundle
from sandbox.projects.browser.autotests.classes.testing_builds import ReleaseKits
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.PrepareBrowserRegressionBuilds import PrepareBrowserRegressionBuilds
from sandbox.projects.browser.autotests_qa_tools.common import (
    BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT)
from sandbox.sandboxsdk.environments import PipEnvironment


BUILDS_KIT_NAME = ReleaseKits.desktop_win_binary_tests.name


class BrowserAnalyzeAutotestsWithRetryes(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        autotest_task = None
        browser_builds = None
        builds_task = None
        browser_commit = None

    class Parameters(sdk2.Task.Parameters):
        browser_branch = sdk2.parameters.String(
            'Browser branch',
            default='master',
            description=u'Ветка браузера для сборки',
            required=True)
        browser_commit = sdk2.parameters.String(
            'Browser commit',
            description=u'Коммит (пустой = latest)',
            default='')
        max_flaky_retries = sdk2.parameters.Integer(
            'Number of retries flaky tests',
            default=10,
            description=u'Количество повторов запусков автотестов')
        launch_fails_2_issues_task = sdk2.parameters.Bool(
            'Create issues for failure tests', default=True,
            description=u"Создать тикеты для стабильно падающих тестов")

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    def task_href(self, task_id):
        return '<a href={link}>#{id}</a>'.format(link=get_task_link(task_id), id=task_id)

    def get_task(self, task_id):
        task = sdk2.Task.find(id=task_id, children=True).limit(1).first()
        if not task:
            raise RuntimeError(u"Task: {} not found".format(task_id))
        return task

    def wait_task(self, task_id):
        task = self.get_task(task_id)
        if task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
            raise sdk2.WaitTime(60 * 10)
        else:
            return task

    def prepare_builds(self):
        builds_task = PrepareBrowserRegressionBuilds(
            self,
            browser_branch=self.Parameters.browser_branch,
            browser_commit=self.Context.browser_commit,
            build_type=BUILDS_KIT_NAME,
            add_autotests_build=True,
            tags=[
                'flaky_analyze',
                self.Parameters.browser_branch,
            ]
        ).enqueue()

        self.Context.builds_task = builds_task.id
        self.set_info(u'Запущена таска подготовки сборок: {}'.format(self.task_href(self.Context.builds_task)),
                      do_escape=False)

    def get_autotests_task_params(self):
        if not self.Context.autotests_task_params:
            autotests_bundle = get_desktop_autotest_bundle(
                self.Context.browser_builds["autotests"], None, None,
                self.clients)
            launch_config = autotests_bundle.get_launch_config_all_cases()
            self.Context.autotests_task_params = {
                "description": u"'Flaky autotests for task # {}".format(self.id),
                "launch_config": launch_config,
                "browser_tests_build_id": self.Context.browser_builds["autotests"],
                "max_flaky_retries": self.Parameters.max_flaky_retries,
                "branch": self.Parameters.browser_branch,
                "commit": self.Context.browser_commit,
                "timeout": DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT * 3,
                "tags": [
                    'flaky_analyze',
                    self.Parameters.browser_branch,
                ]
            }
        return self.Context.autotests_task_params

    def launch_autotests_task(self):
        autotests_task = RunDesktopBrowserAutotests(
            self,
            **self.get_autotests_task_params()).enqueue()

        self.set_info(u'Запуск автотестов: {}'.format(self.task_href(autotests_task.id)),
                      do_escape=False)
        self.Context.autotest_task = autotests_task.id
        raise sdk2.WaitTime(60 * 10)

    def launch_and_wait_autotests(self):
        if not self.Context.autotest_task:
            self.launch_autotests_task()

        autotest_task = self.wait_task(self.Context.autotest_task)
        if autotest_task.status != Status.SUCCESS:
            self.set_info(
                u'--  WARNING! Таска  автотестов: {} не успешна. Статус: {}'.format(
                    self.task_href(autotest_task.id),
                    autotest_task.status),
                do_escape=False)
            raise TaskFailure('Autotests task not success')

    def on_execute(self):
        # prepare builds
        with self.memoize_stage.prepare_builds:
            self.Context.browser_commit = self.Parameters.browser_commit or self.clients.bitbucket.get_latest_commit(
                BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO, self.Parameters.browser_branch)
            self.prepare_builds()

        if not self.Context.browser_builds:
            builds_task = self.wait_task(self.Context.builds_task)
            if builds_task.status != Status.SUCCESS:
                self.set_info(
                    u'Таска подготовки сборок: {} не успешна. Статус: {}'.format(
                        self.task_href(self.Context.builds_task),
                        builds_task.status),
                    do_escape=False)
                raise TaskFailure("Builds_task #{} failed".format(self.Context.builds_task))
            self.Context.browser_builds = builds_task.Context.teamcity_builds

        # launch autotests
        self.launch_and_wait_autotests()

        # Create and launch issues task
        issues_task = DBroAutotestsToIssues(
            self,
            browser_branch=self.Parameters.browser_branch,
            win_autotests=self.get_task(self.Context.autotest_task)
        )
        self.set_info(u'Анализ падений: {}'.format(self.task_href(issues_task.id)),
                      do_escape=False)
        if self.Parameters.launch_fails_2_issues_task:
            issues_task.enqueue()
