# -*- coding: utf-8 -*-
import os
import smtplib
from email.mime.text import MIMEText

import sandbox
from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr
from sandbox.projects.browser.autotests.allure_parser import TESTS_DUMP_FILENAME
from sandbox.projects.browser.autotests.autoduty_tasks.DBroAutotestsToIssues import DBroAutotestsToIssues
from sandbox.projects.browser.autotests_qa_tools.common import is_dev_sandbox, RESULT_DIR
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsDump
from sandbox.projects.browser.autotests_qa_tools.classes.autoduty.flaky_analyzer import FlakyAnalyzer


FAILS_TABLE_PATH = 'fails_table.json'
FLAKY_STAT_RESULT_PATH = 'flaky_stat_result.json'
STABLE_TESTS_PATH = 'stable_tests.json'
SMTP_URL = 'yabacks.yandex.ru'
SMTP_PORT = 25
EMAIL_SENDER = 'robot-bro-qa-infra@yandex-team.ru'
EMAIL_RECIPIENTS = [
    'nik-isaev@yandex-team.ru',
    'browser-autotests-bots@yandex-team.ru',
]


class FlakyAutotestsAnalystResult(sdk2.Resource):
    share = True


class BrowserFlakyAutotestsAnalyst(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):

        issues_task = None

    class Parameters(sdk2.Task.Parameters):

        autotests_tasks = sdk2.parameters.List('Autotests tasks', required=True)
        min_success_tasks_count = sdk2.parameters.Integer(
            'Min success tasks count', default=5,
            description=u"Минимальное число успешных тасок автотестов"
        )
        launch_fails_2_issues_task = sdk2.parameters.Bool(
            'Create issues for failure tests', default=True,
            description=u"Создать тикеты для стабильно падающих тестов"
        )

    def task_href(self, task_id):
        return '<a href={link}>#{id}</a>'.format(link=get_task_link(task_id), id=task_id)

    def get_task(self, task_id):
        task = sdk2.Task.find(id=task_id, children=True).limit(1).first()
        if not task:
            raise RuntimeError(u"Task: {} not found".format(task_id))
        return task

    def get_result_resources_paths(self, autotests_tasks):

        res = []
        for a_task in autotests_tasks:
            _resource = AutotestsDump.find(task=a_task, state=ctr.State.READY).first()
            if not _resource:
                raise RuntimeError(u"Не найден ресурс c отчётом у таски {}".format(a_task.id))
            res.append(
                os.path.join(str(sdk2.ResourceData(_resource).path), TESTS_DUMP_FILENAME)
            )
        return res

    @property
    @sandbox.common.utils.singleton
    def out_dir(self):
        data_path = str(self.path(RESULT_DIR))
        os.mkdir(data_path)
        return data_path

    def save_results(self):
        resource = FlakyAutotestsAnalystResult(self, "Flaky autotests analyst result", self.out_dir)
        sdk2.ResourceData(resource).ready()

    def on_execute(self):

        all_autotests_tasks = [self.get_task(_t) for _t in self.Parameters.autotests_tasks]
        info = u"Таски автотестов:"
        for _t in all_autotests_tasks:
            info += u"<br>{} - {}".format(self.task_href(_t.id), _t.status)
        self.set_info(info, do_escape=False)
        success_tasks = [_t for _t in all_autotests_tasks if _t.status == Status.SUCCESS]
        if not len(success_tasks) or len(success_tasks) < self.Parameters.min_success_tasks_count:
            raise RuntimeError(u"Успешных тасок не достаточно")

        dump_paths = self.get_result_resources_paths(success_tasks)

        analyzer = FlakyAnalyzer()
        analyzer.get_failures_table(dump_paths,
                                    save_to_file=os.path.join(self.out_dir, FAILS_TABLE_PATH))
        flakes_stat_result = analyzer.analyse_failures_table(
            os.path.join(self.out_dir, FAILS_TABLE_PATH),
            save_to_file=os.path.join(self.out_dir, FLAKY_STAT_RESULT_PATH))

        self.set_info("Flaky:{} slow_flaky: {} stable: {}".format(len(flakes_stat_result['flaky']),
                                                                  len(flakes_stat_result['slow_flaky']),
                                                                  len(flakes_stat_result['stable'])))
        stable_tests = []
        for item in flakes_stat_result['stable']:
            stable_tests += item.keys()
        info = u"Стабильные падения зафиксированы у {} штук тестов<br>".format(len(stable_tests))
        info += u"<br>".join(stable_tests)
        self.set_info(info, do_escape=False)

        if self.Parameters.launch_fails_2_issues_task and stable_tests:
            issues_task = DBroAutotestsToIssues(
                self,
                win_autotests=success_tasks[0],
                stable_tests_only=stable_tests
            ).enqueue()

            self.Context.issues_task = issues_task.id
            message = u'Запущена таска заведения тикетов на стабильные падения автотестов: {}'.format(
                self.task_href(self.Context.issues_task))
            message += u"<br>Необходимо просмотреть и полайкать тикеты и отправить кейсы в блэклисты"
            self.set_info(message, do_escape=False)
            if not is_dev_sandbox():
                self.send_email(message)

        else:
            self.set_info(u'Таска заведения тикетов на стабильные падения не запущена')

        self.save_results()

    def send_email(self, massage):

        text = u'<a href="{}" target="_blank">SANDBOX-таска #{}</a><br>'.format(get_task_link(self.id), self.id)
        text += massage
        message = MIMEText(text, _subtype='html', _charset='utf-8')
        message['Subject'] = 'New issues for stable failure autotests'
        message['From'] = EMAIL_SENDER
        message['To'] = ', '.join(EMAIL_RECIPIENTS)
        email_client = smtplib.SMTP(SMTP_URL, SMTP_PORT)
        try:
            email_client.ehlo_or_helo_if_needed()
            email_client.sendmail(EMAIL_SENDER, EMAIL_RECIPIENTS, message.as_string())
        finally:
            email_client.quit()
