# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link

import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.autoduty_tasks.BrowserFlakyAutotestsAnalyst import BrowserFlakyAutotestsAnalyst
from sandbox.projects.browser.autotests.BrowserAutotestRun import BrowserAutotestRun
from sandbox.projects.browser.autotests.classes.testing_builds import ReleaseKits
from sandbox.projects.browser.autotests.regression_tasks.PrepareBrowserRegressionBuilds import PrepareBrowserRegressionBuilds


BUILDS_KIT_NAME = ReleaseKits.desktop_win_binary_tests.name


class BrowserRunFlakyAutotests(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        autotest_tasks = []
        browser_builds = None
        builds_task = None

    class Parameters(sdk2.Task.Parameters):
        browser_branch = sdk2.parameters.String(
            'Browser branch',
            default='master',
            description=u'Ветка браузера для сборки',
            required=True)
        browser_commit = sdk2.parameters.String(
            'Browser commit',
            description=u'Коммит (пустой = latest)',
            default='')
        number_of_repetitions = sdk2.parameters.Integer(
            'Number of tests repetitions',
            default=10,
            description=u'Количество повторов запусков автотестов')
        launch_fails_2_issues_task = sdk2.parameters.Bool(
            'Create issues for failure tests', default=True,
            description=u"Создать тикеты для стабильно падающих тестов")

    def task_href(self, task_id):
        return '<a href={link}>#{id}</a>'.format(link=get_task_link(task_id), id=task_id)

    def get_task(self, task_id):
        task = sdk2.Task.find(id=task_id, children=True).limit(1).first()

        if not task:
            raise RuntimeError(u"Task: {} not found".format(task_id))
        return task

    def wait_task(self, task_id):
        task = self.get_task(task_id)
        if task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
            raise sdk2.WaitTime(60 * 10)
        else:
            return task

    def prepare_builds(self):
        builds_task = PrepareBrowserRegressionBuilds(
            self,
            browser_branch=self.Parameters.browser_branch,
            browser_commit=self.Parameters.browser_commit,
            build_type=BUILDS_KIT_NAME,
            add_autotests_build=True,
            tags=[
                'flaky',
                self.Parameters.browser_branch,
            ]
        ).enqueue()

        self.Context.builds_task = builds_task.id
        self.set_info(u'Запущена таска подготовки сборок: {}'.format(self.task_href(self.Context.builds_task)),
                      do_escape=False)

    def get_autotests_task_params(self):
        if not self.Context.autotests_task_params:

            self.Context.autotests_task_params = {
                "description": u"'Flaky autotests for task # {}".format(self.id),
                "framework_branch": "master",
                "config_file": "binary_only_win.json",
                "browser_version": 'custom',
                "build_id": 0,
                "fake_build_id": 0,
                "browser_tests_build_id": self.Context.browser_builds["autotests"],
                "tags": [
                    'flaky',
                    self.Parameters.browser_branch,
                ]
            }
        return self.Context.autotests_task_params

    def launch_autotests_task(self):
        autotests_task = BrowserAutotestRun(
            self,
            **self.get_autotests_task_params()).enqueue()

        self.set_info(u'Запуск автотестов: {}'.format(self.task_href(autotests_task.id)),
                      do_escape=False)
        self.Context.autotest_tasks.append(autotests_task.id)
        raise sdk2.WaitTime(60 * 10)

    def launch_and_wait_autotests(self):
        if not self.Context.autotest_tasks:
            self.launch_autotests_task()

        last_task = self.wait_task(self.Context.autotest_tasks[-1])
        if last_task.status != Status.SUCCESS:
            self.set_info(
                u'--  WARNING! Таска  автотестов: {} не успешна. Статус: {}'.format(
                    self.task_href(last_task.id),
                    last_task.status),
                do_escape=False)

        if self.Parameters.number_of_repetitions > len(self.Context.autotest_tasks):
            self.launch_autotests_task()

    def on_execute(self):
        # prepare builds
        with self.memoize_stage.prepare_builds:
            self.prepare_builds()

        if not self.Context.browser_builds:
            builds_task = self.wait_task(self.Context.builds_task)
            if builds_task.status != Status.SUCCESS:
                self.set_info(
                    u'Таска подготовки сборок: {} не успешна. Статус: {}'.format(
                        self.task_href(self.Context.builds_task),
                        builds_task.status),
                    do_escape=False)
                raise RuntimeError("Builds_task #{} failed".format(self.Context.builds_task))
            self.Context.browser_builds = builds_task.Context.teamcity_builds

        # launch autotests
        self.launch_and_wait_autotests()

        # launch Analyst task
        analyst_task = BrowserFlakyAutotestsAnalyst(
            self,
            autotests_tasks=self.Context.autotest_tasks,
            min_success_tasks_count=self.Parameters.number_of_repetitions - 1,
            launch_fails_2_issues_task=self.Parameters.launch_fails_2_issues_task
        ).enqueue()

        self.set_info(u'Анализатор флаков: {}'.format(self.task_href(analyst_task.id)),
                      do_escape=False)
