# -*- coding: utf-8 -*-
import os
import smtplib
from email.mime.text import MIMEText

import sandbox
from sandbox import sdk2
from sandbox.common.utils import get_task_link
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.classes.testcases_analyst import TestcasesAnalyst, ReportStatuses
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT, RESULT_DIR
from sandbox.sandboxsdk.environments import PipEnvironment

RESULT_FILENAME = 'index.html'
SMTP_URL = 'yabacks.yandex.ru'
SMTP_PORT = 25
EMAIL_SUBJECT = 'Testpalm cases markup status report'
EMAIL_SENDER = 'robot-bro-qa-infra@yandex-team.ru'


class BrowserCasesReportResource(sdk2.Resource):
    share = True


class BrowserTestpalmAnalyst(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
            PipEnvironment('testpalm-api-client', version='4.0.2')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        result_info = None

    class Parameters(sdk2.Task.Parameters):

        testpalm_projects = sdk2.parameters.List('Testpalm projects', required=True)
        tests_builds = sdk2.parameters.List(
            'Tests builds',
            description='List of teamcity builds "Browser_Tests_BuildTests" with browser tests')
        desktop_autotests_branch = sdk2.parameters.String(
            'desktop_autotests_branch',
            description=u'Ветка фреймворка десктопных python-автотестов (только для десктопа)',
            default='')
        consider_platforms = sdk2.parameters.List(
            'Platforms', default=[],
            description=u'Учитывать платформы: apad, aphone, ios, win7x64 и/или пр'
        )
        treat_blacklisted_cases_as_automated = sdk2.parameters.Bool(
            'Treat blacklists as automated', default=True,
            description=u"Учитывать blacklisted-кейсы как автоматизированные"
        )
        auto_set_ready_status = sdk2.parameters.Bool(
            'Auto set ready status', default=False,
            description=u"Автоматически проставить 'Automation Status'='Ready', если значение в кейсе отсутствует"
        )
        report_recipients = sdk2.parameters.List('Report recipients "login@yandex-team.ru"')

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @property
    @sandbox.common.utils.singleton
    def out_dir(self):
        data_path = str(self.path(RESULT_DIR))
        os.mkdir(data_path)
        return data_path

    def on_execute(self):
        analyst = TestcasesAnalyst(self.clients)
        result_data = analyst.check_automated_mapping(
            projects=self.Parameters.testpalm_projects,
            tests_builds=self.Parameters.tests_builds,
            consider_platforms=self.Parameters.consider_platforms,
            treat_blacklisted_cases_as_automated=self.Parameters.treat_blacklisted_cases_as_automated,
            desktop_autotests_branch=self.Parameters.desktop_autotests_branch or None
        )

        self.Context.result_info = {key.name: [_c.url for _c in cases] for key, cases in result_data.iteritems()}
        report_html = analyst.render_report(result_data,
                                            self.Parameters.testpalm_projects,
                                            self.Parameters.tests_builds)
        self.save_report_resource(report_html)

        self.set_info(u'<a href="{}" target="_blank"># Отчет</a>'.format(self.Context.report_url), do_escape=False)
        if result_data.get(ReportStatuses.WillBeAutoMarked):
            info = u"Обнаружены не размеченные кейсы ({}) шт.".format(len(result_data.get(ReportStatuses.WillBeAutoMarked)))
            if self.Parameters.auto_set_ready_status:
                info += u' Кейсы будут размечены автоматически'
                analyst.mark_cases_as_automated(result_data.get(ReportStatuses.WillBeAutoMarked))
            else:
                info += u" Автоматическая разметка выключена параметром 'auto_set_ready_status'"
            self.set_info(info)
        else:
            self.set_info(u"Кейсов для автоматической разметки не обнаружено")

        if self.Parameters.report_recipients:
            self.send_email()

    def save_report_resource(self, report_html):
        out_path = str(self.path(RESULT_DIR))
        os.mkdir(out_path)
        with open(os.path.join(out_path, RESULT_FILENAME), "w") as _f:
            _f.write(report_html.encode('utf-8'))
        resource = BrowserCasesReportResource(self, "Cases condition report", out_path)
        sdk2.ResourceData(resource).ready()
        self.Context.report_url = "{}/{}".format(resource.http_proxy, RESULT_FILENAME)

    def send_email(self):

        text = u'<a href="{}" target="_blank">SANDBOX-таска #{}</a>'.format(get_task_link(self.id), self.id)
        text += u' <br><a href="{}" target="_blank">Отчёт</a>'.format(
            self.Context.report_url)
        message = MIMEText(text, _subtype='html', _charset='utf-8')
        message['Subject'] = EMAIL_SUBJECT
        message['From'] = EMAIL_SENDER
        message['To'] = ', '.join(self.Parameters.report_recipients)
        email_client = smtplib.SMTP(SMTP_URL, SMTP_PORT)
        try:
            email_client.ehlo_or_helo_if_needed()
            email_client.sendmail(EMAIL_SENDER, self.Parameters.report_recipients, message.as_string())
        finally:
            email_client.quit()
