# -*- coding: utf-8 -*-
import os
import re

import sandbox
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.common.types.task import Status
from sandbox.common.types import resource as ctr
from sandbox.projects.browser.autotests.autoduty_tasks.DBroIssuesToBlacklistsAndWork import DBroIssuesToBlacklistsAndWork
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests_qa_tools.classes.autoduty.failures_analyzer import FailuresAnalyzer, AllureFailuresAnalyzer
from sandbox.projects.browser.autotests_qa_tools.common import html_link, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, extract_zip_resource
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsAllureData, AutotestsReportResource
from sandbox.sandboxsdk.environments import PipEnvironment


REPORT_TYPES = {
    'btr': AutotestsReportResource,
    'allure': AutotestsAllureData
}


class DBroAutotestsToIssues(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('ids==1.3.66', custom_parameters=["requests==2.18.4"]),
            PipEnvironment('ids-staff==0.11.12', custom_parameters=["requests==2.18.4"]),
            PipEnvironment('testpalm-api-client', version='4.0.2'),
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    class Parameters(sdk2.Parameters):
        browser_branch = sdk2.parameters.String("Browser branch", default="master")
        win_autotests = sdk2.parameters.Task('Win autotests task')
        mac_autotests = sdk2.parameters.Task('Mac autotests task')
        with sdk2.parameters.CheckGroup('Issues types') as issue_types:
            issue_types.values.BinaryTestsFailedIssue = issue_types.Value('BinaryTestsFailedIssue', checked=True)
            issue_types.values.PythonInstallUpdateTestIssue = issue_types.Value('PythonInstallUpdateTestIssue', checked=False)
            issue_types.values.PythonInstallUpdateTestIssueInfra = issue_types.Value('PythonInstallUpdateTestIssueInfra', checked=False)
            issue_types.values.PythonLegacyTestIssue = issue_types.Value('PythonLegacyTestIssue', checked=False)
            issue_types.values.AndroidTestsFailedIssue = issue_types.Value('AndroidTestsFailedIssue', checked=False)
            issue_types.values.IosTestsFailedIssue = issue_types.Value('IosTestsFailedIssue', checked=False)
        cases_priority_only = sdk2.parameters.String(
            'Cases priority only',
            default="Blocker,Critical,Normal",
            description="Comma-separated list of test cases priority for issues creation"
        )
        stable_tests_only = sdk2.parameters.JSON(
            u'Список стабильно падающих тестов, разрешенных для заведения задач. Если пуст - разрешены все тесты',
            default=[])
        new_issues_assignee = sdk2.parameters.String('New issues assignee',
                                                     description='Leave empty for default assignee')
        report_type = sdk2.parameters.String('Report type', required=True, default='btr',
                                             choices=[(_, _) for _ in ['btr', 'allure']])

    def get_autotest_tasks(self):
        autotests_tasks = [_t for _t in [self.Parameters.win_autotests, self.Parameters.mac_autotests] if _t]
        if not autotests_tasks:
            TaskFailure(u"Не указано ни одной таски с автотестами")
        return autotests_tasks

    def get_reports(self, autotests_tasks):

        # ret reports
        reports = []
        report_class = REPORT_TYPES[self.Parameters.report_type]
        for a_task in autotests_tasks:
            _resource = report_class.find(task=a_task,
                                          state=ctr.State.READY).first()
            if not _resource:
                raise RuntimeError(u"Не найден ресурс c отчётом у таски {}".format(a_task.id))
            reports.append(
                os.path.join(str(sdk2.ResourceData(_resource).path))
            )

        if not self.Parameters.report_type == 'allure':
            return reports
        else:
            result = []
            for _path in reports:
                result.append(extract_zip_resource(self, _path))
            return result

    def on_execute(self):

        autotests_tasks = self.get_autotest_tasks()
        with self.memoize_stage.wait_autotests_tasks:
            raise sdk2.WaitTask(
                tasks=autotests_tasks,
                wait_all=True,
                statuses=[Status.Group.FINISH, Status.Group.BREAK]
            )
        for task in autotests_tasks:
            if task.status != Status.SUCCESS:
                self.set_info(u'Таска с тестами {} не успешна статус={}'.format(
                    html_link(get_task_link(task.id)),
                    task.status),
                    do_escape=False)
                raise TaskFailure()

        reports = self.get_reports(autotests_tasks)

        if not self.Parameters.report_type == 'allure':
            analyzer = FailuresAnalyzer(self.oauth_vault, task_link=get_task_link(self.id))
        else:
            analyzer = AllureFailuresAnalyzer(self.oauth_vault, task_link=get_task_link(self.id))

        cases_priority_filter = re.sub(r'\s?', "", self.Parameters.cases_priority_only).split(",") if self.Parameters.cases_priority_only else None
        templates = analyzer.get_issues_templates(reports, self.Parameters.issue_types,
                                                  cases_priority_filter=cases_priority_filter,
                                                  tests_names_filter=self.Parameters.stable_tests_only)
        if templates:
            pr_issue = analyzer.create_main_ticket()
            create_issues_params = {
                "issues": templates,
                "parent": pr_issue
            }
            if self.Parameters.new_issues_assignee:
                create_issues_params['assignee'] = self.Parameters.new_issues_assignee

            issues_data = analyzer.create_issues(**create_issues_params)

            next_step_task = DBroIssuesToBlacklistsAndWork(
                self,
                issues_data=issues_data,
                pr_issue=pr_issue,
                browser_branch=self.Parameters.browser_branch
            )

            tickets = u"\n".join(html_link(u"{}/{}".format(self.clients.st_base_url, issue_key)) for issue_key in issues_data)
            self.set_info(u'Созданы тикеты:\n{}'.format(tickets), do_escape=False)
            self.set_info(
                u'После просмотра и утверждения запустите: {}'.format(html_link(get_task_link(next_step_task.id))),
                do_escape=False)
        else:
            self.set_info(u"Падений согласно фильтрам не зафиксированно. Тикетов не заведено")
