# -*- coding: utf-8 -*-
import logging
import subprocess
import time
import os

import sandbox
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests_qa_tools.classes.autoduty.failures_analyzer import FailuresAnalyzer
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests_qa_tools.common import (
    html_link, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, ROBOT_BRO_QA_INFRA_SSH_KEY_VAULT)
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.sandboxsdk.environments import PipEnvironment


REVIEWERS = ["nik-isaev", "hpotter"]


class DBroIssuesToBlacklistsAndWork(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 3 * 1024  # 3GB
        cores = 1
        dns = misc.DnsType.DNS64
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        environments = [
            GitEnvironment('2.19'),
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
            PipEnvironment('virtualenv', '15.1.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def yaml(self):
        from ruamel.yaml import YAML
        yaml = YAML()
        yaml.indent(mapping=2, sequence=2, offset=2)
        return yaml

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    class Parameters(sdk2.Parameters):
        issues_data = sdk2.parameters.JSON(
            'Autotests tasks')
        pr_issue = sdk2.parameters.String('Ticket for PR link', required=True)
        browser_branch = sdk2.parameters.String('Browser repo branch', default='master')
        reviewers = sdk2.parameters.List('Reviewers', default=REVIEWERS)
        skip_update_blacklist_and_create_pr = sdk2.parameters.Bool(
            'Skip update blacklist and create PR', default=False)

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @sandbox.common.utils.singleton_property
    def git(self):
        return GitCli(self.browser_path, config={
            'user.name': ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
            'user.email': '{}@yandex-team.ru'.format(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        })

    def update_autotests_configs(self, actual_issues, hpe):

        for issue_key, issue_data in actual_issues.iteritems():
            if issue_data["type"] != 'binary':
                raise RuntimeError(
                    u"Тикет {} содержит не поддерживаемый тип  тестов {}. Поддерживается только 'binary'".format(
                        issue_key, issue_data["type"]))

            for test_case_id, source_file in issue_data.get("blacklists_targets", []):
                logging.info("Update {} for issue {}".format(source_file, issue_key))
                script_path = os.path.join(os.path.dirname(__file__), 'issues_to_blacklist.py')
                command = [str(hpe.python_executable), script_path] + [
                    os.path.abspath(self.browser_path),
                    test_case_id,
                    source_file,
                    issue_key,
                ]
                logging.debug('Running {}'.format(command))
                env = os.environ.copy()
                with sdk2.helpers.ProcessLog(self, logger="script-log") as pl:
                    subprocess.check_call(command, env=env, stdout=pl.stdout, stderr=pl.stdout)

    def update_blacklist_and_create_pr(self, actual_issues, hpe):

        pr_branch = 'wp/robots/{}/{}'.format(self.Parameters.pr_issue, int(time.time() * 1000))
        try:
            logging.info('Checkout browser branch %s', self.Parameters.browser_branch)
            vcs_root = repositories.Stardust.browser()
            vcs_root.clone(self.browser_path, self.Parameters.browser_branch)

            self.git.checkout('-f', '-B', pr_branch, self.Parameters.browser_branch)
        except subprocess.CalledProcessError:
            raise errors.TemporaryError('Checkout error')

        self.update_autotests_configs(actual_issues, hpe)

        try:
            self.git.commit(
                '-a', '-m',
                'Add autotests to blacklists. Committed by sandbox task: {}.'.format(
                    get_task_link(self.id)
                ),
            )
            with sdk2.ssh.Key(self, ROBOT_BRO_QA_INFRA_SSH_KEY_VAULT, None):
                self.git.push('origin', '{}:refs/heads/{}'.format(pr_branch, pr_branch))
        except subprocess.CalledProcessError:
            raise errors.TaskFailure('Failed to commit or push changes')

        pr = self.clients.bitbucket.create_pr(
            'stardust', 'browser',
            'Add autotests to blacklists.',
            description='Created from sandbox task {}'.format(get_task_link(self.id)),
            from_ref=pr_branch, to_ref=self.Parameters.browser_branch,
            reviewers=self.Parameters.reviewers
        )
        self.set_info(u'<a href="{}">Пулл-реквест</a> на занесение тикетов в блэклисты</a>'.format(pr.web_url),
                      do_escape=False)

    def info_about_blacklists(self, actual_issues):
        platforms = {}
        for issie_key, issue_data in actual_issues.iteritems():
            for test in issue_data['tests']:
                platforms.setdefault(test[1], {}).setdefault(test[0], set()).add(issie_key)

        message = u'Необходимо вручную занести в блэклисты следующие кейсы \n'
        for platform, cases in platforms.iteritems():
            message += u'{} \n'.format(platform)
            for case in sorted(cases.keys()):
                message += u'    {}: {} \n'.format(case, ','.join(cases[case]))
        self.set_info(message, do_escape=False)

    def on_execute(self):
        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.1',
            packages=('ruamel.ordereddict==0.4.13', 'ruamel.yaml==0.15.35'),
        ) as hpe:
            analyzer = FailuresAnalyzer(self.oauth_vault)

            actual_issues = analyzer.get_actual_issues_drafts(self.Parameters.issues_data)
            actual_tickets = u"\n".join(html_link(u"{}/{}".format(self.clients.st_base_url, issue_key)) for issue_key in actual_issues.keys())
            self.set_info(u'Подтвержденные тикеты будут добавлены в блэклисты и отправлены в работу:\n{}'.format(actual_tickets if actual_issues else u"Нет"), do_escape=False)

            no_actual_issues = filter(lambda x: x not in actual_issues.keys(), self.Parameters.issues_data.keys())
            no_actual_tickets = u"\n".join(html_link(u"{}/{}".format(self.clients.st_base_url, issue_key)) for issue_key in no_actual_issues)
            self.set_info(u'Не подтвержденные тикеты будут пропущены:\n{}'.format(no_actual_tickets if no_actual_issues else u"Нет"), do_escape=False)

            if not actual_issues:
                raise RuntimeError(u"Нет актуальных тикетов в списке")

            if not self.Parameters.skip_update_blacklist_and_create_pr:
                self.update_blacklist_and_create_pr(actual_issues, hpe)
            else:
                self.info_about_blacklists(actual_issues)

            analyzer.send_issues_to_work(actual_issues)
