# -*- coding: utf-8 -*-
from copy import deepcopy

from sandbox.projects.browser.autotests.allure_parser import AllureReport
from sandbox.projects.browser.autotests.classes.btr_log_parser import BtrReport
from sandbox.projects.browser.autotests.classes.testing_platforms import TestingPlatforms
from sandbox.projects.browser.autotests.classes.test_statuses import TestStatuses


def aggregate_tests_status(tests):

    statuses = set(test.status for test in tests)
    if not statuses:
        return TestStatuses.NOT_FOUND.name
    if all(_s in TestStatuses.PASSED.value for _s in statuses):
        return TestStatuses.PASSED.name
    elif any(_s in TestStatuses.FAILED.value for _s in statuses):
        return TestStatuses.FAILED.name
    elif any(_s in TestStatuses.SKIPPED.value for _s in statuses):
        return TestStatuses.SKIPPED.name
    elif any(_s in TestStatuses.NOT_RUN.value for _s in statuses):
        return TestStatuses.NOT_RUN.name


def get_autotests_statuses(autotests_result, jobs):
    result = {}
    for job in jobs:

        if autotests_result:
            platform = autotests_result.get_report_platform_name(job['testrun_environment'])
            tests = autotests_result.report.get_case_tests("{}-{}".format(job['case_project'], job['case_id']),
                                                           platform) if platform else []
            status = aggregate_tests_status(tests)
            result[job['uuid']] = status
        else:
            result[job['uuid']] = TestStatuses.NOT_FOUND.name
    return result


class AutotestsResult(object):

    def __init__(self, report_path, ya_resources):
        self.report = self.report_cls(report_path, ya_resources)

    def get_report_platform_name(self, job_testrun_environment):
        raise NotImplementedError()

    @property
    def report_cls(self):
        raise NotImplementedError()


class IsolatesAutotestsResult(AutotestsResult):

    report_cls = BtrReport

    def get_report_platform_name(self, job_testrun_environment):
        artifact_name = TestingPlatforms.get_isolates_artifact_name(job_testrun_environment)
        if artifact_name:
            artifact_name = artifact_name.replace(".json", "")
        return artifact_name


class AllureAutotestsResult(AutotestsResult):

    report_cls = AllureReport

    def get_report_platform_name(self, job_testrun_environment):
        return TestingPlatforms.get_allure_platform_name(job_testrun_environment)


def allure_2_btr(allure_report, build_id):
    result = {}
    for test in allure_report.tests:
        isolane_config_name = TestingPlatforms.get_isolates_artifact_name(test.platform)
        if not isolane_config_name:
            continue
        _platform = isolane_config_name.replace(".json", "")
        _test_result = {
            "status": "SUCCESS" if test.is_passed else test.status,
            "binary": "python_framework_tests",
            "name": test.name
        }
        if test.status not in TestStatuses.SKIPPED.value and test.status not in TestStatuses.NOT_RUN.value:
            _test_result["elapsed_time_ms"] = test.duration
        else:
            _test_result["elapsed_time_ms"] = 0

        if not test.is_passed:
            _test_result['log'] = test.test_log
        result.setdefault(_platform, {}).setdefault(str(build_id), {}).setdefault(test.test_id, []).append(_test_result)
    return result


def merge_btr_reports(report1=None, report2=None):
    report1 = report1 or {}
    report2 = report2 or {}
    result = deepcopy(report1)
    for _platform, builds in report2.iteritems():
        for _build_id, cases in builds.iteritems():
            for _case, tests in cases.iteritems():
                for _test in tests:
                    test_data = deepcopy(_test)
                    result.setdefault(_platform, {}).setdefault(_build_id, {}).setdefault(_case, []).append(test_data)
    return result
