# -*- coding: utf-8 -*-
import os
import json
import re
import time

import sandbox
from sandbox.projects.browser.autotests.classes.test_statuses import TestStatuses
from sandbox.projects.browser.autotests_qa_tools.common import BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO


RESULT_FILENAME = "results.json"
LAUNCH_INFO_FILENAME = "launch_info.json"
EXPECTED_SUCCESS = 0.5
TESTS_DUMP_FILENAME = '_tests_dump.json'
PYTHON_TESTS_BINARYS = ["python_framework_tests"]
DEFAULT_OWNERS = ["nik-isaev", "hpotter"]
ANDROID_TESTS_OWNERS = ['dendnk968', 'dolf']
PRIORITIES = [
    'Blocker',
    'Critical',
    'Normal',
    'Low',
    'Minor',
]


def get_owners(bitbucket_client, project, repo, commit, files):
    owners = bitbucket_client.get_owners(project, repo, commit, list(files))
    res = {_file: [] for _file in files}
    for _owner in owners:
        for _file in _owner["files"]:
            res[_file].append(_owner["owner"]["emailAddress"].split("@")[0])
    return res


class BtrTest(object):

    def __init__(self, platform, build_id, case_id, test_data, test_version,
                 tested_application,
                 launch_info,
                 ya_resources):
        self.platform = platform
        self.build_id = build_id
        self.case_id = case_id
        self._test_data = test_data
        self.test_version = test_version
        self.tested_application = tested_application
        self.launch_info = launch_info if launch_info else {}
        self.ya_resources = ya_resources

    @property
    def task_id(self):
        return str(self.launch_info.get('task_id', 0))

    @property
    def status(self):
        return self._test_data['status']

    @property
    def name(self):
        return self._test_data['name']

    @property
    def binary(self):
        return self._test_data['binary']

    @property
    def full_name(self):
        return "{}.{}".format(self._test_data['binary'], self._test_data['name'])

    @property
    def log(self):
        return self._test_data.get('log', '')

    @property
    def test_log(self):
        return self.log

    @property
    def test_id(self):
        return self.case_id

    @property
    def is_passed(self):
        return self.status in TestStatuses.PASSED.value

    @sandbox.common.utils.singleton_property
    def testpalm_case(self):
        if not self.test_id:
            return None
        _case_id = re.split(r'-(\d+)$', self.test_id)
        return self.ya_resources.testpalm.get_case(_case_id[0], _case_id[1], include_fields=('id', 'attributes'))

    @property
    def is_failure(self):
        return self.status in TestStatuses.FAILED.value

    @property
    def filepath(self):
        return self._test_data.get('sources')[0] if self._test_data.get('sources') else None

    @property
    def component(self):
        return self._test_data.get('components')[0] if self._test_data.get('components') else None

    @property
    def feature(self):
        return self.component

    @sandbox.common.utils.singleton_property
    def priority(self):
        case = self.testpalm_case
        result = case.priority if case and case.priority else ['Normal']
        return sorted(result,
                      key=self.priority_sort)[0]

    @staticmethod
    def priority_sort(value):
        if value in PRIORITIES:
            return PRIORITIES.index(value)
        else:
            return len(PRIORITIES)

    @property
    def test_type(self):
        return 'binary' if self.binary not in PYTHON_TESTS_BINARYS else 'python'

    @property
    def tested_builds(self):
        return {
            'binary_tests': self.build_id
        }

    @property
    def elapsed_time_ms(self):
        return self._test_data.get('elapsed_time_ms')


class BtrReport(object):

    def __init__(self, resource_path, ya_resources, sandbox_task_id=None):
        self.resource_path = resource_path
        self.ya_resources = ya_resources

    @property
    def sandbox_task(self):
        return self.launch_info.get('task_id')

    @property
    def platform_id(self):
        return self.metadata['platform_id']

    @property
    @sandbox.common.utils.singleton
    def launch_info(self):
        if not os.path.exists(os.path.join(self.resource_path, LAUNCH_INFO_FILENAME)):
            return {}
        with open(os.path.join(self.resource_path, LAUNCH_INFO_FILENAME), 'r') as _f:
            data = json.load(_f)
        return data

    @property
    @sandbox.common.utils.singleton
    def _data(self):
        with open(os.path.join(self.resource_path, RESULT_FILENAME), 'r') as _f:
            data = json.load(_f)
        return data

    @property
    @sandbox.common.utils.singleton
    def branch(self):
        return self.launch_info.get('parameters', {}).get('branch')

    @property
    @sandbox.common.utils.singleton
    def tests(self):
        result = []
        for _platform, builds in self._data.iteritems():
            for _build, cases in builds.iteritems():
                for _case, tests in cases.iteritems():
                    for _test_data in tests:
                        result.append(BtrTest(platform=_platform,
                                              build_id=_build,
                                              case_id=_case,
                                              test_data=_test_data,
                                              test_version=self.branch,
                                              tested_application=self.tested_application,
                                              launch_info=self.launch_info,
                                              ya_resources=self.ya_resources))
        return result

    @sandbox.common.utils.singleton_property
    def failures(self):
        return filter(lambda x: x.is_failure, self.tests)

    def get_case_tests(self, case_id, report_platform_name):
        return [test for test in self.tests if test.case_id == case_id and test.platform == report_platform_name]

    @property
    @sandbox.common.utils.singleton
    def summary(self):
        res = {
            "total": len(self.tests),
            "platforms": {}
        }
        for _test in self.tests:
            status_count = res['platforms'].setdefault(_test.platform, {"_total": 0, "_was_launched": 0}).setdefault(_test.status, 0)
            res['platforms'][_test.platform][_test.status] = status_count + 1
            res['platforms'][_test.platform]['_total'] += 1
            if _test.status not in TestStatuses.SKIPPED.value:
                res['platforms'][_test.platform]['_was_launched'] += 1
        return res

    @property
    @sandbox.common.utils.singleton
    def html_summary(self):
        html_string = u'<ul><li>Всего тестов:{}</li>'.format(self.summary['total'])
        for _platform, _platform_results in self.summary['platforms'].iteritems():
            html_string += u'<li><b>{}</b><ul>'.format(_platform)
            for status in sorted(_platform_results.keys()):
                html_string += u'<li>{}: {}</li>'.format(status, _platform_results[status])
            html_string += u'</ul></li>'
        html_string += u'</ul>'
        return html_string

    @property
    @sandbox.common.utils.singleton
    def problems(self):
        results = []
        for _platform, _platform_results in self.summary['platforms'].iteritems():
            if _platform_results.get('_was_launched', 0) > 0:
                if float(_platform_results.get('SUCCESS', 0)) / _platform_results['_was_launched'] < EXPECTED_SUCCESS:
                    results.append(u'На платформе {} успешны меньше половины запущеных тестов'.format(_platform))
        return results

    def dump_tests(self, target_path):
        tests_dump = []
        for _t in self.tests:
            tests_dump.append(
                {
                    "name": _t.name,
                    "type": _t.binary,
                    "platform": _t.platform,
                    "status": _t.status,
                    "test_id": _t.test_id,
                    "feature": "",
                    "failure_message": "",
                    "log": _t.log[-300:] if not _t.is_passed else ""
                })
        dump_file = os.path.join(target_path, TESTS_DUMP_FILENAME)
        if os.path.exists(dump_file):
            os.remove(dump_file)

        with open(dump_file, 'w') as _f:
            json.dump(
                {
                    "sandbox_task": self.sandbox_task,
                    "tests": tests_dump
                }, _f, indent=4)
        return dump_file

    @property
    @sandbox.common.utils.singleton
    def browser_build_id(self):
        browser_build = self.launch_info.get('parameters', {}).get(
            'build_extra_args', {}).get('python_tests-extra-args', {}).get('build_id')
        if browser_build:
            return browser_build

        # appium_tests
        appium_app_url = self.launch_info.get('parameters', {}).get(
            'build_extra_args', {}).get('appium_tests-extra-args', [])
        if not appium_app_url:
            return None
        appium_app_url = appium_app_url[0]
        _r = re.search(r"/builds/id:\d{1,}", appium_app_url)
        if not _r:
            return None
        return _r.group(0).split(":")[1]

    @property
    @sandbox.common.utils.singleton
    def fake_build_id(self):
        return self.launch_info.get('parameters', {}).get(
            'build_extra_args', {}).get('python_tests-extra-args', {}).get('fake_build_id')

    @property
    def tested_application(self):
        return self.launch_info.get('parameters', {}).get('tested_application')

    def get_yt_statistics(self, common_fields=None):
        common_fields = common_fields or {}
        result = []
        for test in self.tests:
            test_data = dict(
                # tests info
                case_id=test.case_id,
                test_name=test.name,
                binary=test.binary,
                platform=test.platform,
                status=test.status,
                elapsed_time_ms=int(test.elapsed_time_ms) if test.elapsed_time_ms is not None else None,
                # builds info
                tests_build_id=int(test.build_id),
                browser_build_id=int(self.browser_build_id) if self.browser_build_id is not None else None,
                fake_build_id=int(self.fake_build_id) if self.fake_build_id is not None else None,
                tests_build_branch=self.ya_resources.get_build_revision(test.build_id)[0],
                tests_build_commit=self.ya_resources.get_build_revision(test.build_id)[1],
                runner_branch=self.launch_info.get('parameters', {}).get('branch'),
                runner_commit=self.launch_info.get('parameters', {}).get('commit'),
                tested_application=self.tested_application,
                # task info
                task_id=int(self.launch_info['task_id']) if self.launch_info.get('task_id') is not None else None,
                date=int(self.launch_info['date']) if self.launch_info.get('date') is not None else None
            )
            _build_queued_date = self.ya_resources.get_build_revision(test.build_id)[2]
            test_data['tests_build_queued_date'] = int(time.mktime(_build_queued_date.timetuple())) if _build_queued_date else None
            test_data.update(common_fields)
            result.append(test_data)
        return result

    def get_tests_owners(self, tests_list=None, branch="master"):
        if tests_list is None:
            tests_list = self.tests

        binary_tests = [test for test in tests_list if test.test_type == 'binary' and test.platform not in ['aphone', 'apad']]
        android_tests = [test for test in tests_list if test.test_type == 'binary' and test.platform in ['aphone', 'apad']]
        if binary_tests:
            browser_files = set([
                test.filepath[0] if isinstance(test.filepath, list) else test.filepath
                for test in binary_tests if test.filepath
            ])

            browser_commit = self.ya_resources.bitbucket.get_latest_commit(
                BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO, branch)
            browser_owners = get_owners(
                self.ya_resources.bitbucket,
                BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO,
                browser_commit, browser_files)

            for item in binary_tests:
                item.owners = browser_owners.get(item.filepath[0] if isinstance(item.filepath, list) else item.filepath,
                                                 DEFAULT_OWNERS)
        for item in android_tests:
            item.owners = ANDROID_TESTS_OWNERS
