import json
import logging
from collections import defaultdict

FUNCTIONALITY_TAG_PREFIX = 'func_'
STARCASES_NOT_NEEDED_TAG = 'StarCasesNotNeed'
STARCASES_DONE_TAG = 'StarCasesDone'

FUNCTIONALITY_ST_FIELD_SCHEMA = {
    'funcs': list
}


class Functionality(object):
    SEPARATOR = ':'

    def __init__(self, layers):
        assert layers
        self.layers = layers

    def __ne__(self, other):
        return not self.__eq__(other)

    def __eq__(self, other):
        return self.layers == other.layers

    def __unicode__(self):
        return self.SEPARATOR.join(self.layers)

    def __repr__(self):
        return unicode(self)

    def __hash__(self):
        return hash(self.__unicode__())

    @property
    def component(self):
        return self.layers[0]

    @classmethod
    def from_str(cls, functionality_string):
        if functionality_string.startswith('func_'):
            functionality_string = functionality_string[5:]
        return cls(functionality_string.split(cls.SEPARATOR))


class FunctionalityNode(object):
    def __init__(self):
        self.children = defaultdict(FunctionalityNode)
        # This field tells if issue has this level of functionality as a leaf.
        # E.g. if one issue has functionality A:B:C and another issue just A:B then we check both A:B:C and A:B.
        self.check = False

    def get_or_insert(self, key):
        return self.children[key]

    def get_or_none(self, key):
        return self.children.get(key)

    def get_all_layer_functionalities(self):
        res = set()
        for child_name, child_node in self.children.iteritems():
            res.update({Functionality([child_name] + functionality.layers)
                        for functionality in child_node.get_all_layer_functionalities()})
            res.add(Functionality([child_name]))
        return res


class FunctionalitiesTree(object):
    def __init__(self, functionalities):
        root = FunctionalityNode()
        for functionality in functionalities:
            tail = root
            for layer in functionality.layers:
                tail = tail.get_or_insert(layer)
            tail.check = True
        self.root = root

    def should_check_functionality(self, functionality):
        tail = self.root
        for layer in functionality.layers:
            tail = tail.get_or_none(layer)
            if not tail:
                return False
            if tail.check:
                return True
        return False

    def should_check_depends_on_functionality(self, functionality):
        tail = self.root
        if not tail.children:  # no functionalities from diff, should check nothing
            return False
        for layer in functionality.layers:
            if not tail.children:
                return True
            tail = tail.get_or_none(layer)
            if not tail:
                return False
        return True

    def get_all_layer_functionalities(self):
        return self.root.get_all_layer_functionalities()


def get_functionality_from_issue_field(field_content):
    """
    Functionality field structure: json with fields, described in FUNCTIONALITY_ST_FIELD_SCHEMA
    """

    def is_properly_formatted(json_object):
        properly_formatted = True
        for required_field, filed_type in FUNCTIONALITY_ST_FIELD_SCHEMA.iteritems():
            if required_field not in json_object:
                logging.warning('Field %s is required, but not found', required_field)
                properly_formatted = False
                continue
            if not isinstance(json_object[required_field], filed_type):
                logging.warning('Field %s should by instance of %s, but is %s',
                                required_field, filed_type, json_object[required_field].type())
                properly_formatted = False
                continue
        return properly_formatted

    if not field_content:
        return None

    try:
        content = json.loads(field_content)
    except ValueError:
        logging.exception('Functionality field is not a proper json, will ignore it')
        return None

    if not is_properly_formatted(content):
        return None

    return filter(None, {functionality.strip() for functionality in content['funcs']})


def get_components_common(issue_obj):
    components = {component.display for component in getattr(issue_obj, 'components', [])}
    return components


def get_components(issue_obj):
    components = {component.display for component in getattr(issue_obj, 'components', [])}

    if 'Installer+Updater' in components or 'Browser-Update' in components:
        components.discard('Installer+Updater')
        components.discard('Browser-Update')
        components.update({'Updater', 'Installer'})
    return components


def get_issue_functionalities(issue_obj):
    logging.debug('Getting functionalities for issue %s', issue_obj.key)
    functionality_field = getattr(issue_obj, 'functionality', u'')
    tags = getattr(issue_obj, 'tags', [])
    components = get_components(issue_obj)
    logging.debug('Functionality field: %s', functionality_field)
    logging.debug('Issue tags: %s', tags)
    logging.debug('Issue components %s', components)

    res = set()
    if STARCASES_NOT_NEEDED_TAG in tags:
        logging.debug('Starcases not needed, no functionalities for this issue')
        return set()

    if STARCASES_DONE_TAG in tags:
        logging.debug('Starcases done, will use functionalities logic')
        functionalities_from_field = get_functionality_from_issue_field(functionality_field)
        functionalities_from_tags = {Functionality.from_str(tag)
                                     for tag in tags if tag.startswith(FUNCTIONALITY_TAG_PREFIX)}
        if functionalities_from_field:
            logging.debug('Functionality field is filled, will use functionalities from there')
            res = {Functionality.from_str(f) for f in functionalities_from_field}
        elif functionalities_from_tags:
            logging.debug('Got functionalities from tags')
            res = functionalities_from_tags
        else:
            logging.debug('No functionalities in field or tags. Use just components')
            res = {Functionality.from_str(component) for component in components}
    else:
        logging.debug('Starcases not done, will use just components')
        res = {Functionality.from_str(component) for component in components}
    logging.debug('Result functionalities: %s', res)
    return res


def get_testcase_functionalities(testcase_obj):
    functionalities_strings = testcase_obj.mapped_attributes.get('Functionality', [])
    component = testcase_obj.mapped_attributes['Component'][0]
    return {Functionality.from_str(f) for f in functionalities_strings + [component]}


def get_testcase_depends_on_functionalities(testcase_obj):
    functionalities_strings = testcase_obj.mapped_attributes.get('Depends on Functionality', [])
    return {Functionality.from_str(f) for f in functionalities_strings}


def should_run_case_in_regression(func_tree, testcase):
    case_functionalities = get_testcase_functionalities(testcase)
    case_depends_on_functionalities = get_testcase_depends_on_functionalities(testcase)
    return (
        any(func_tree.should_check_functionality(functionality)
            for functionality in case_functionalities) or
        any(func_tree.should_check_depends_on_functionality(functionality)
            for functionality in case_depends_on_functionalities)
    )


def get_all_layer_functionalities(functionalities_strings):
    tree = FunctionalitiesTree([Functionality.from_str(f) for f in functionalities_strings])
    return tree.get_all_layer_functionalities()
