# -*- coding: utf-8 -*-
import itertools
import jinja2
import logging
import time
from copy import deepcopy

import sandbox
from sandbox.projects.browser.autotests.classes.autotests_result import get_autotests_statuses
from sandbox.projects.browser.autotests.classes.test_statuses import TestStatuses
from sandbox.projects.browser.autotests.classes.testing_job import TestrunTemplate, JobExecutors
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression.conf import TestingGroup
from sandbox.projects.browser.autotests.testpalm_helper import version_url
from sandbox.projects.browser.autotests_qa_tools.common import is_dev_sandbox


TICKET_JINJA_PACKAGE_PATH = 'projects.browser.autotests.templates'
WAIT_AUTOTESTS_COMMENT = u"\n**!!(жел)Дождитесь обновления тикета о результатах автотестов!!**"
REGRESSION_COMPLETE_COMMENT = u"\n**!!(зел)Создание регрессии для группы завершено.!!**"
AUTOTESTS_SECTION_MARK = u"\n======(autotests-end-section)======"
RUN_CREATION_PAUSE = 0.5

logger = logging.getLogger(__file__)


class GroupRegressionManager(object):

    assessor_files_order = None
    honey_logs_cases = {}

    def __init__(self,
                 sandbox_task_parameters,
                 main_context,
                 group_context,
                 oauth_vault):
        self._parameters = sandbox_task_parameters
        self._main_context = main_context
        self._context = group_context
        self.clients = YaClients(oauth_vault)
        if is_dev_sandbox():
            logger.setLevel(logging.DEBUG)
            logger.addHandler(logging.StreamHandler())

    @property
    def manual_version_template(self):
        raise NotImplementedError()

    def get_manual_run_title(self):
        raise NotImplementedError()

    @property
    def ticket_jinja_template_package(self):
        raise NotImplementedError()

    @property
    def ticket_summary_template(self):
        raise NotImplementedError()

    @property
    def assessors_ticket_summary_template(self):
        raise NotImplementedError()

    @property
    def st_version(self):
        raise NotImplementedError()

    @property
    def _group_ticket_properties(self):
        properties = self.settings['group_ticket_markup']
        properties['assignee'] = self.group.head
        properties['followers'] = self.group.participants
        return properties

    @property
    def _assessors_ticket_properties(self):
        properties = self.settings['asessor_ticket_markup']
        properties['assignee'] = self.group.head
        properties['followers'] = self.group.participants
        return properties

    @property
    def group_ticket_properties(self):
        return self._group_ticket_properties

    @property
    def assessors_ticket_properties(self):
        return self._assessors_ticket_properties

    @property
    def assessor_version_template(self):
        raise NotImplementedError()

    @property
    def group_ticket_queue(self):
        return self.group.manual_ticket_queue

    @property
    def assessors_ticket_queue(self):
        return self.group.assessors_ticket_queue

    @property
    def distributed_jobs(self):
        return self._context["distributed_jobs"]

    def get_assessor_run_title(self, *args, **kwargs):
        raise NotImplementedError()

    @property
    @sandbox.common.utils.singleton
    def settings(self):
        settings = deepcopy(self._main_context.settings)
        # merge group markup
        settings['group_ticket_markup']['tags'] = list(set(
            settings['group_ticket_markup']['tags'] + self.group.manual_ticket_tags))
        settings['group_ticket_markup']['components'] = list(set(
            settings['group_ticket_markup']['components'] + self.group.manual_ticket_components))
        settings['asessor_ticket_markup']['tags'] = list(set(
            settings['asessor_ticket_markup']['tags'] + self.group.asessors_ticket_tags))
        settings['asessor_ticket_markup']['components'] = list(set(
            settings['asessor_ticket_markup']['components'] + self.group.asessors_ticket_components))
        return settings

    @distributed_jobs.setter
    def distributed_jobs(self, value):
        if self._context["distributed_jobs"] is not None:
            raise RuntimeError("Group jobs already distributed")
        self._context["distributed_jobs"] = value

    @property
    @sandbox.common.utils.singleton
    def group(self):
        return TestingGroup(**self._context["group"])

    @property
    def main_ticket(self):
        return self._context["main_ticket"]

    @property
    def group_issue_key(self):
        return self._context["group_issue_key"]

    @group_issue_key.setter
    def group_issue_key(self, value):
        if self._context["group_issue_key"] is not None:
            raise RuntimeError("Group_ticket already fixed")
        self._context["group_issue_key"] = value

    @property
    def group_issue(self):
        if self.group_issue_key is None:
            raise RuntimeError(u"Group ticket seems not yet created")
        return self.clients.startrek.issues[self.group_issue_key]

    @property
    def assessors_tickets(self):
        return self._context["assessors_tickets"]

    @property
    def max_run_duration(self):
        return self._parameters.max_run_duration

    @property
    def runs(self):
        return self._context["runs"]

    @property
    @sandbox.common.utils.singleton
    def group_ticket_template(self):
        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))
        return env.get_template('group_ticket.jinja')

    @property
    @sandbox.common.utils.singleton
    def assessor_ticket_template(self):
        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))
        return env.get_template('assessors_ticket.jinja')

    def render_assessor_ticket_template(self, *args, **kwargs):
        raise NotImplementedError()

    def render_group_ticket_template(self):
        return self.group_ticket_template.render(
            launch_comment=self._parameters.manual_launch_comment
        )

    @staticmethod
    def new_group_context(main_ticket, group, no_automated_jobs, automated_jobs):
        return {
            "main_ticket": main_ticket,
            "group": group.summary,
            "group_issue_key": None,
            "no_automated_jobs": no_automated_jobs,
            "automated_jobs": automated_jobs,
            "jobs_statistics": {
                "autotests_success": [],
                "sent_to_work": [],
                "autotests_status": {}
            }
        }

    def _save_job_statistics(self, target, jobs):
        self._context["jobs_statistics"].setdefault(target, []).extend(_j["uuid"] for _j in jobs)

    @property
    def tested_builds(self):
        raise NotImplementedError()

    @staticmethod
    def _jobs_2_run_sets_by_duration(jobs, max_run_duration):
        result = [[]]
        current_split_estimate = 0
        for job in jobs:
            if max_run_duration is not None and current_split_estimate + job['estimate'] > max_run_duration * 60 * 1000:
                if result[-1]:
                    result.append([])
                    current_split_estimate = 0
            result[-1].append(job)
            current_split_estimate += job['estimate']
        return result

    def _jobs_2_run_templates(self, jobs, max_run_duration=None):
        def _key(x):
            return (x['case_project'], x['component'], x['suite_info'], x['testrun_environment'], x['build_info'])

        result = []
        # split by project, component,  suite_id and testrun_environment
        jobs = sorted(jobs, key=_key)
        for _group_data, _grouped_jobs in itertools.groupby(jobs, key=_key):
            # split jobs to run sets by duration
            _run_sets = self._jobs_2_run_sets_by_duration(_grouped_jobs, max_run_duration)
            for _part, _run_set in enumerate(_run_sets, 1):
                result.append(TestrunTemplate(
                    project=_group_data[0],
                    component=_group_data[1],
                    jobs=_run_set,
                    environment=_group_data[3],
                    build_info=_group_data[4],
                    suite_info=_group_data[2],
                    part=_part)
                )
        return result

    def _jobs_2_agregate_run_templates(self, jobs, max_run_duration):
        def _key(x):
            # split by project, component,  suite_id and testrun_environment
            return (x['case_project'], x['suite_info'], x['testrun_environment'], x['build_info'])

        result = []
        jobs = sorted(jobs, key=_key)
        for _group_data, _grouped_jobs in itertools.groupby(jobs, key=_key):
            # split jobs to run sets by duration
            _run_sets = self._jobs_2_run_sets_by_duration(_grouped_jobs, max_run_duration)
            for _part, _run_set in enumerate(_run_sets, 1):
                result.append(TestrunTemplate(
                    project=_group_data[0],
                    component="all_components",
                    jobs=_run_set,
                    environment=_group_data[2],
                    build_info=_group_data[3],
                    suite_info=_group_data[1],
                    part=_part)
                )
        return result

    def _jobs_2_manual_run_templates(self, jobs, max_run_duration):
        _agregate = self._jobs_2_agregate_run_templates(
            filter(lambda x: x["suite_info"]["aggregate_manual_runs"], jobs),
            max_run_duration=max_run_duration
        )
        _no_argegate = self._jobs_2_run_templates(
            filter(lambda x: not x["suite_info"]["aggregate_manual_runs"], jobs),
            max_run_duration=max_run_duration
        )
        return _agregate + _no_argegate

    def _jobs_2_assessors_run_templates(self, jobs, max_run_duration):
        _agregate = self._jobs_2_agregate_run_templates(
            filter(lambda x: x["suite_info"]["aggregate_asessors_runs"], jobs),
            max_run_duration=max_run_duration
        )
        _no_argegate = self._jobs_2_run_templates(
            filter(lambda x: not x["suite_info"]["aggregate_asessors_runs"], jobs),
            max_run_duration=max_run_duration
        )
        return _agregate + _no_argegate

    def jobs_2_executors_run_templates(self, jobs, max_run_duration=None, force_send_to_manual_testing=False):
        assessor = self._jobs_2_assessors_run_templates(
            jobs=[_j for _j in jobs if _j["executor"] == JobExecutors.assessor.value and not force_send_to_manual_testing],
            max_run_duration=max_run_duration
        )
        manual = self._jobs_2_manual_run_templates(
            jobs=[_j for _j in jobs if _j["executor"] != JobExecutors.assessor.value or force_send_to_manual_testing],
            max_run_duration=None
        )
        return {
            "assessor": assessor,
            "manual": manual
        }

    def _run_template_2_run_cases_list(self, run_template_cases):
        cases = []
        for _id, _data in run_template_cases.iteritems():
            _d = {_key: _value for _key, _value in _data.iteritems()}
            _d.update({"id": _id})
            cases.append(_d)
        return cases

    def _get_testpalm_version(self, template, project, unique, description=None):
        if unique:
            result = self.clients.testpalm.create_unique_version(
                project,
                template,
                description)
        else:
            result = self.clients.testpalm.get_or_create_version(
                template, project, description)
        return result

    @sandbox.common.utils.singleton
    def get_assessor_version(self, run_template):
        return self._get_testpalm_version(
            self.assessor_version_template,
            run_template["project"],
            unique=True)

    def get_manual_version(self, run_template):
        return self._get_testpalm_version(
            self.manual_version_template,
            run_template["project"],
            unique=False)

    def get_manual_run_properties(self, run_template):
        return {}

    def create_manual_runs(self, run_templates):
        created_runs = []
        for run_template in run_templates:
            version = self.get_manual_version(run_template)
            logger.debug(" ############## manual_run_version = {}".format(version))

            run_title = self.get_manual_run_title(run_template)

            testrun_properties = self.get_manual_run_properties(run_template)
            run_id = self.clients.testpalm.create_testrun(
                version=version,
                cases=self._run_template_2_run_cases_list(run_template['cases']),
                project=run_template["project"],
                title=run_title,
                properties=testrun_properties,
                current_environment=run_template["environment"],
                tags=run_template["suite_info"]["tags"]
            )

            run_instance = self.clients.testpalm.get_testrun(run_id,
                                                             run_template["project"])
            logger.info("manual run added: {}".format(run_instance.url))
            created_runs.append({"suite_info": run_template["suite_info"],
                                 "instance": run_instance})
            # TODO fix it
            try:
                self.group_issue.remotelinks.create('relates', 'testrun/{}/{}'.format(run_instance.project, run_instance.id),
                                                    'ru.yandex.testpalm')
            except Exception as e:
                logger.warning(e.message)

            time.sleep(RUN_CREATION_PAUSE)
        return created_runs

    def create_assessor_runs(self, run_temlpates):
        raise NotImplementedError()

    def add_assessor_ticket(self, runs):
        description = self.render_assessor_ticket_template(runs)
        issue_dict = {
            'queue': self.assessors_ticket_queue,
            'summary': self.assessors_ticket_summary_template,
            'description': description,
            "links": [
                {
                    "relationship": "relates",
                    "issue": self.group_issue_key
                }
            ]
        }
        issue_dict.update(self.assessors_ticket_properties)
        if self.main_ticket:
            issue_dict['parent'] = self.main_ticket
        res = self.clients.startrek.issues.create(**issue_dict)['key']
        logger.info("ASSESSORS TICKET CREATED: {}/{}".format(self.clients.st_base_url, res))
        return res

    def create_group_ticket(self):
        if self.group_issue_key is not None:
            raise RuntimeError(u"Group ticket already created: {}".format(self.group_issue_key))

        description = self.render_group_ticket_template()
        issue_dict = {
            'queue': self.group_ticket_queue,
            'summary': self.ticket_summary_template,
            'description': description
        }
        issue_dict.update(self.group_ticket_properties)
        if self.main_ticket:
            issue_dict['parent'] = self.main_ticket
        self.group_issue_key = self.clients.startrek.issues.create(**issue_dict)['key']

        logger.info("GROUP TICKET CREATED: {}/{}".format(self.clients.st_base_url,
                                                         self.group_issue_key))
        return self.group_issue_key

    def _update_issue_description(self, comment, mark=None):

        def _modifier(issue):
            if mark is not None and mark in issue.description:
                return {
                    'description': issue.description.replace(mark, u"\n{}{}".format(comment, mark), 1)
                }
            else:
                return {
                    'description': u'{}\n{}'.format(
                        issue.description,
                        comment)
                }

        self.clients.safely_update_issue(
            self.group_issue_key,
            modifier=_modifier
        )
        logger.info("group ticket updated {}/{}".format(self.clients.st_base_url,
                                                        self.group_issue_key))

    def comment_about_created_human_work(self,
                                         manual_runs=None,
                                         assessor_runs=None,
                                         assessor_ticket=None,
                                         title=None,
                                         mark=None):
        # TODO 2. format asessor issue link
        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))

        def _key(x):
            return (x["suite_info"]["url"])
        manual_runs = sorted(manual_runs, key=_key)
        grouped_manual_runs = {}
        for suite_url, _grouped_runs in itertools.groupby(manual_runs, key=_key):
            grouped_manual_runs[suite_url] = [testrun["instance"].pretty_str(True) for testrun in _grouped_runs]

        comment = env.get_template('group_issue_human_work.jinja').render(
            grouped_manual_runs=grouped_manual_runs,
            manual_testpalm_versions=set(set(version_url(run["instance"].project,
                                                         run["instance"].version)
                                         for run in manual_runs)),
            assessor_ticket=assessor_ticket if assessor_ticket else "",
            title=title
        )
        self._update_issue_description(comment, mark=mark)

    def comment_about_autotests_work(self, jobs):

        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))
        comment = env.get_template('group_issue_autotests_work.jinja').render(
            autotests_jobs_count=len(jobs)
        )

        self._update_issue_description(u"{}{}{}".format(comment,
                                                        WAIT_AUTOTESTS_COMMENT if jobs else REGRESSION_COMPLETE_COMMENT,
                                                        AUTOTESTS_SECTION_MARK))

    def comment_about_autotests_result(self, failed_jobs, total_autotests_jobs):

        env = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path=self.ticket_jinja_template_package))
        comment = env.get_template('group_issue_autotests_results.jinja').render(
            failed_count=len(failed_jobs),
            total_count=len(total_autotests_jobs)
        )

        self._update_issue_description(comment, mark=AUTOTESTS_SECTION_MARK)
        self.delete_wait_autotests_comment()
        logger.info("group ticket updated {}/{}".format(self.clients.st_base_url,
                                                        self.group_issue_key))

    def delete_wait_autotests_comment(self):
        self.clients.safely_update_issue(
            self.group_issue_key,
            modifier=lambda x: {"description": x.description.replace(WAIT_AUTOTESTS_COMMENT, "")}
        )
        logger.info("group ticket updated {}/{}".format(self.clients.st_base_url,
                                                        self.group_issue_key))

    def comment_about_regression_complete(self):
        self._update_issue_description(REGRESSION_COMPLETE_COMMENT)

    def run_regression(self):
        # create_group_ticket
        self.create_group_ticket()
        # create_runs
        manual_runs, assessor_runs, assessor_ticket = self.send_jobs_2_work(
            self._context["no_automated_jobs"])
        # update issue description
        self.comment_about_created_human_work(
            manual_runs,
            assessor_runs,
            assessor_ticket)
        # update issue description
        self.comment_about_autotests_work(self._context["automated_jobs"])
        return manual_runs, assessor_runs, assessor_ticket

    def continue_regression(self, autotest_result):
        manual_runs = []
        assessor_runs = []
        assessor_ticket = None
        if self._context["automated_jobs"]:

            results = get_autotests_statuses(autotest_result, self._context["automated_jobs"])
            no_success_jobs = [j for j in self._context["automated_jobs"] if results[j["uuid"]] != TestStatuses.PASSED.name]

            self._save_job_statistics("autotests_success", [j for j in self._context["automated_jobs"] if j not in no_success_jobs])
            self._context["jobs_statistics"]["autotests_status"] = results

            self.comment_about_autotests_result(no_success_jobs,
                                                self._context["automated_jobs"])
            if no_success_jobs:
                manual_runs, assessor_runs, assessor_ticket = self.send_jobs_2_work(
                    no_success_jobs,
                    force_send_to_manual_testing=False)

                # update issue description
                self.comment_about_created_human_work(
                    manual_runs,
                    assessor_runs,
                    assessor_ticket,
                    title=u"==== Перепроверка после автотестов",
                    mark=AUTOTESTS_SECTION_MARK)

                self.comment_about_regression_complete()

        return manual_runs, assessor_runs, assessor_ticket

    def send_jobs_2_work(self, jobs, force_send_to_manual_testing=False):

        self._save_job_statistics("sent_to_work", jobs)
        distributed_run_templates = self.jobs_2_executors_run_templates(
            jobs,
            max_run_duration=self.max_run_duration,
            force_send_to_manual_testing=force_send_to_manual_testing
        )

        manual_runs = []
        assessor_runs = []
        assessor_ticket = None

        if distributed_run_templates['manual']:
            manual_runs = self.create_manual_runs(
                distributed_run_templates['manual'])

        if distributed_run_templates['assessor']:
            assessor_runs = self.create_assessor_runs(distributed_run_templates['assessor'])
            assessor_ticket = self.add_assessor_ticket(assessor_runs)

        return manual_runs, assessor_runs, assessor_ticket
