# -*- coding: utf-8 -*-
import jinja2
import logging
import urlparse
import uuid
from copy import deepcopy

import sandbox
from sandbox.projects.browser.autotests.classes.autotests_result import IsolatesAutotestsResult
from sandbox.projects.browser.autotests.classes.functionalities import (
    get_components_common, FunctionalitiesTree, Functionality, should_run_case_in_regression,
    get_issue_functionalities)
from sandbox.projects.browser.autotests.classes.diff_tickets import get_diff_tickets, get_issue_obj, sort_issues
from sandbox.projects.browser.autotests.classes.testing_job import TestingJob, JobExecutors
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression.conf import TestingGroup, get_groups_conf_by_activities
from sandbox.projects.browser.autotests_qa_tools.common import (
    is_dev_sandbox, html_link, split_list, REGRESSION_GROUPS_ACTIVITIES)

logger = logging.getLogger(__file__)


class RegressionManager(object):

    regression_type_name = None
    autotests_result_parser = IsolatesAutotestsResult

    class SuiteInfo(object):
        def __init__(self, id, project, assessors_platforms, manual_platforms,
                     filter_by_diff=None, tags=None, brand=None,
                     comment=None, aggregate_manual_runs=False,
                     aggregate_asessors_runs=False, honey_logs=False, disable_assessors=False):
            self.manual_platforms = tuple(manual_platforms)
            self.assessors_platforms = tuple(assessors_platforms)
            self.project = project
            self.id = id
            self.tags = tuple(tags or [])
            self.filter_by_diff = filter_by_diff or False
            self.aggregate_manual_runs = aggregate_manual_runs or False
            self.aggregate_asessors_runs = aggregate_asessors_runs or False
            self.honey_logs = honey_logs or False
            self.disable_assessors = disable_assessors or False
            self.comment = comment or ""

    @property
    def settings(self):
        if not self._context.settings:
            self._context.settings = self.regression_config['settings']
        return self._context.settings

    @property
    def _main_ticket_properties(self):
        properties = self.settings['main_ticket_markup']
        if self._parameters.main_regression_ticket_assignee:
            properties['assignee'] = self._parameters.main_regression_ticket_assignee
        return properties

    @property
    def main_ticket_properties(self):
        return self._main_ticket_properties

    def is_assessors(self, case):
        return ('Asessors' in case.mapped_attributes.get('Checklist', []) or
                'Assessors' in case.mapped_attributes.get('Checklist', []))

    def get_component(self, case):
        try:
            return case.mapped_attributes['Component'][0]
        except:
            raise RuntimeError(u"Кейс {} не имеет разметки компонента".format(case.url))

    @property
    @sandbox.common.utils.singleton
    def testing_groups(self):
        return [self.get_named_group(name) for name in self.settings["groups"]]

    @sandbox.common.utils.singleton
    def get_named_group(self, group_name):
        all_groups = get_groups_conf_by_activities(REGRESSION_GROUPS_ACTIVITIES)
        if group_name not in all_groups:
            raise RuntimeError(u"Данные группы {} не найдены в конфиге групп".format(group_name))
        group_info = deepcopy(all_groups[group_name])

        if self.settings['group_ticket_assignee'] == "regression_assignee" and self._parameters.main_regression_ticket_assignee:
            if not group_info.get('hardcore_assignee'):
                group_info['participants'].insert(0, self._parameters.main_regression_ticket_assignee)

        return TestingGroup(name=group_name, **group_info)

    def get_group_by_component(self, component):
        for group in self.testing_groups:
            if component in group.components:
                return group
        return None

    def get_group_by_testpalm_project(self, testpalm_project):
        for group in self.testing_groups:
            if testpalm_project in group.testpalm_projects:
                return group
        return None

    def get_group(self, job):
        method = self.settings['allocate_cases_to_groups']
        if method == 'by_testpalm_project':
            result = self.get_group_by_testpalm_project(job['case_project'])
        elif method == 'by_testpalm_component':
            result = self.get_group_by_component(job['component'])
        else:
            raise RuntimeError(u"Unknown allocate_cases_to_groups method: {}".format(method))
        return result or self.default_group

    @property
    def default_group(self):
        return self.get_named_group(self.settings['default_group'])

    @property
    def _diff_get_components_method(self):
        return get_components_common

    @property
    def ticket_jinja_template_package(self):
        raise NotImplementedError()

    @property
    def tested_builds(self):
        raise NotImplementedError()

    @property
    @sandbox.common.utils.singleton
    def autotests_bundle(self):
        raise NotImplementedError()

    @property
    @sandbox.common.utils.singleton
    def tested_application(self):
        raise NotImplementedError()

    def publish_diff_tickets(self):
        raise NotImplementedError()

    def is_automated_in_testpalm(self, case, testrun_environment):
        return case.mapped_attributes.get('Automation Status', []) == ['Ready']

    def is_automated(self, case, testrun_environment):
        return bool(
            self.autotests_bundle is not None and
            self.is_automated_in_testpalm(case, testrun_environment) and
            self.autotests_bundle.is_automated_in_tests(
                "{}-{}".format(
                    case.project, case.id),
                testrun_environment)
        )

    def get_automated_info(self):
        return None

    def create_main_ticket(self):
        raise NotImplementedError()

    @classmethod
    def load_suites_info(cls, regression_config):
        return [cls.SuiteInfo(**item) for item in regression_config["testsuites"]["testsuites"]]

    @property
    def main_ticket(self):
        return self._context.main_ticket

    @main_ticket.setter
    def main_ticket(self, value):
        if self._context.main_ticket is not None:
            raise RuntimeError("Main regression ticket already fixed")
        self._context.main_ticket = value

    @property
    def all_manual_oses(self):
        res = set()
        for suite in self.test_suites:
            res.update(suite.manual_platforms)
        return res

    @property
    def main_ticket_template(self):
        env = jinja2.Environment(
            loader=jinja2.PackageLoader('projects.browser.autotests.templates',
                                        package_path=self.ticket_jinja_template_package))
        return env.get_template('main_ticket.jinja')

    def __init__(self, regression_config, task_parameters, task_context, oauth_vault):
        self.regression_config = regression_config
        self.test_suites = self.load_suites_info(regression_config)
        self._parameters = task_parameters
        self._context = task_context
        self.oauth_vault = oauth_vault
        self.clients = YaClients(oauth_vault)
        if is_dev_sandbox():
            logger.setLevel(logging.DEBUG)
        self._context.regression_type_name = self.regression_type_name

    @property
    @sandbox.common.utils.singleton
    def regression_jobs(self):
        return self._suites_2_jobs()

    def load_suite_cases(self, suite_id, suite_project):
        return self.clients.testpalm.get_testsuite_cases(
            suite_id,
            suite_project,
            include_fields=('isAutotest', 'stats', 'id', 'estimatedTime',
                            'name', 'attributes', 'properties', 'status', 'removed'))

    def should_check_case(self, case, suite):
        return True

    def get_builds_info(self, *args, **kwargs):
        return None

    @sandbox.common.utils.singleton
    def _suites_2_jobs(self):

        jobs = []
        for suite in self.test_suites:
            cases = self.load_suite_cases(suite.id,
                                          suite.project)
            suite_object = self.clients.testpalm.get_testsuite(suite.id, suite.project)
            for case in cases:
                if case.status != 'ACTUAL':
                    continue
                if not self.should_check_case(case, suite):
                    continue

                if suite.disable_assessors:
                    executor = JobExecutors.manual.value
                    testrun_environments = suite.manual_platforms or suite.assessors_platforms
                elif self.is_assessors(case):
                    executor = JobExecutors.assessor.value
                    testrun_environments = suite.assessors_platforms
                else:
                    executor = JobExecutors.manual.value
                    testrun_environments = suite.manual_platforms

                if not testrun_environments:
                    logger.warning(u'Для {}-кейса {} нет платформ'.format(executor, case.url))

                case_grouping_fields = []
                for field in getattr(suite_object, "groups", []):
                    grouping_values = case.attributes.get(field, [])
                    case_grouping_fields.append(grouping_values[0] if grouping_values and grouping_values[0] else None)

                _jobs = [TestingJob(uuid=uuid.uuid4().hex,
                                    case_id=case.id,
                                    case_project=case.project,
                                    testrun_environment=_env,
                                    build_info=self.get_builds_info(suite),
                                    executor=executor,
                                    component=self.get_component(case),
                                    is_automated=self.is_automated(case, _env),
                                    estimate=case.estimate,
                                    case_grouping_fields=case_grouping_fields,
                                    suite_info=dict(id=suite.id, tags=suite.tags, project=case.project,
                                                    url=suite_object.pretty_str(),
                                                    aggregate_manual_runs=suite.aggregate_manual_runs,
                                                    aggregate_asessors_runs=suite.aggregate_asessors_runs,
                                                    honey_logs=suite.honey_logs
                                                    ))
                         for _env in testrun_environments]

                jobs += _jobs
        return jobs

    def _jobs_2_groups(self, jobs):
        """
        :param cases list(TestingJob)
        :return: {TestingGroup: [TestingJob, TestingJob, ...]}
        """
        result = {}
        for job in jobs:
            _group = self.get_group(job)
            result.setdefault(
                _group, []).append(job)
        return result

    def get_or_create_group_manager(self, group,
                                    no_automated_jobs=None, automated_jobs=None):
        if no_automated_jobs is None:
            no_automated_jobs = []
        if automated_jobs is None:
            automated_jobs = []

        group_context = self._context.groups.get(group.name)

        if group_context is None:
            self._context.groups[group.name] = self.group_regresson_manager_class.new_group_context(
                self._context.main_ticket,
                group,
                no_automated_jobs=no_automated_jobs,
                automated_jobs=automated_jobs)

        return self.get_group_from_context(group.name)

    @sandbox.common.utils.singleton
    def get_group_from_context(self, group_name):
        return self.group_regresson_manager_class(
            sandbox_task_parameters=self._parameters,
            main_context=self._context,
            group_context=self._context.groups[group_name],
            oauth_vault=self.oauth_vault
        )

    def create_regression(self):
        if not self.regression_jobs:
            raise RuntimeError("An empty set of cases, taking into account the diff")

        grouped_jobs = self._jobs_2_groups(self.regression_jobs)
        if self.settings['create_main_ticket']:
            self.create_main_ticket()

        logger.info("Main_ticket = {}/{}".format(self.clients.st_base_url, self._context.main_ticket))
        result = {}
        for group, jobs in grouped_jobs.iteritems():
            no_automated_jobs = [_j for _j in jobs if not _j["is_automated"]]
            automated_jobs = [_j for _j in jobs if _j["is_automated"]]
            group_manager = self.get_or_create_group_manager(
                group,
                no_automated_jobs=no_automated_jobs,
                automated_jobs=automated_jobs)
            manual_runs, assessor_runs, assessor_ticket = group_manager.run_regression()
            result[group_manager.group] = dict(manual_runs=manual_runs,
                                               group_ticket=group_manager.group_issue_key,
                                               assessors_runs=assessor_runs,
                                               assessors_ticket=assessor_ticket)

        runs_and_tickets = self.prepare_regression_result_structure(result)
        return {
            "runs_and_tickets": runs_and_tickets,
            "info_message": self.summary_message(*runs_and_tickets)
        }

    def continue_regression(self, autotests_results_path, issues_filter=""):

        autotests_report = self.autotests_result_parser(autotests_results_path, self.clients) if autotests_results_path else None
        issues_filter = issues_filter.replace(" ", "")
        issues_list = issues_filter.split(",") if issues_filter else None
        result = {}
        for group_name, group_context in self._context.groups.iteritems():
            if issues_list is not None and group_context["group_issue_key"] not in issues_list:
                continue
            group_manager = self.get_group_from_context(group_name)
            manual_runs, assessor_runs, assessor_ticket = group_manager.continue_regression(autotests_report)
            result[group_manager.group] = dict(manual_runs=manual_runs,
                                               group_ticket=group_manager.group_issue_key,
                                               assessors_runs=assessor_runs,
                                               assessors_ticket=assessor_ticket)

        runs_and_tickets = self.prepare_regression_result_structure(result)
        return {
            "runs_and_tickets": runs_and_tickets,
            "info_message": self.summary_message(*runs_and_tickets)
        }

    def summary_message(self, main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs):

        message = u'Главный тикет на регрессию: {}</br>'.format(
            html_link(urlparse.urljoin(self.clients.st_base_url, main_ticket))
        ) if main_ticket else ''

        message += u'Тикеты на ручное тестирование:</br>{}</br>'.format(
            '<br/>'.join(
                '{}: {}'.format(
                    group.name,
                    html_link(urlparse.urljoin(self.clients.st_base_url, ticket)))
                for group, ticket in manual_tickets.iteritems())
        ) if manual_tickets else ''

        message += u'Тикеты на тестирование асессорами:</br>{}</br>'.format(
            '<br/>'.join(
                '{}: {}'.format(
                    group.name,
                    html_link(urlparse.urljoin(self.clients.st_base_url, ticket)))
                for group, ticket in assessors_tickets.iteritems())
        ) if assessors_tickets else ''

        return message

    def prepare_regression_result_structure(self, results):
        main_ticket = self.main_ticket
        manual_tickets = {}
        manual_runs = {}
        assessors_tickets = {}
        assessors_runs = {}

        for group, result in results.iteritems():
            manual_tickets[group] = result["group_ticket"]
            if result["assessors_ticket"]:
                assessors_tickets[group] = result["assessors_ticket"]
            if result["manual_runs"]:
                manual_runs[group] = self.restore_runs_suite_structure(result["manual_runs"])
            if result["assessors_runs"]:
                assessors_runs[group] = self.restore_runs_suite_structure(result["assessors_runs"])

        return main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs

    def restore_runs_suite_structure(self, runs):
        sorted_runs = {}
        for run in runs:
            sorted_runs.setdefault(run["suite_info"]["id"], []).append(run["instance"])
        return {self._get_suite_info_object(suite_id): suite_runs for suite_id, suite_runs in sorted_runs.iteritems()}

    def _get_suite_info_object(self, suite_id):
        for suite in self.test_suites:
            if suite.id == suite_id:
                return suite
        raise RuntimeError(u"Incorrect suite id: {}".format(suite_id))

    # Base diff
    def get_diff_issues(self):
        if not self._parameters.old_build_id:
            raise RuntimeError(u"Не указана предыдущая сборка")

        if not self._context.diff_issues:
            new_issues, maybe_new_issues, old_issues, report = get_diff_tickets(
                project='stardust', repo='browser',
                old_build_id=self._parameters.old_build_id,
                new_build_id=self._parameters.build_id,
                teamcity_client=self.clients.teamcity,
                bb_client=self.clients.bitbucket,
                startrek_client=self.clients.prod_startrek)

            self._context.diff_issues = (
                [issue.key for issue in new_issues],
                [issue.key for issue in maybe_new_issues],
                [issue.key for issue in old_issues],
            )

        return (
            [get_issue_obj(self.clients.prod_startrek, issue) for issue in issues]
            for issues in self._context.diff_issues
        )

    def get_filtered_diff(self, filter=None):
        def filter_issues(all_issues):
            if not all_issues:
                return all_issues
            res = []
            for issues in split_list(all_issues, 100):
                diff_issues_filter = ' OR '.join('Key:{}'.format(issue.key) for issue in issues)
                res_filter = '({}) AND ({})'.format(diff_issues_filter, filter)
                res.extend(self.clients.prod_startrek.issues.find(res_filter))
            return sort_issues(res, self.clients.prod_startrek)

        new_issues, maybe_new_issues, old_issues = self.get_diff_issues()
        if filter:
            logging.debug('Filtering diff with filter %s', filter)
            new_issues, maybe_new_issues, old_issues = [
                filter_issues(issue_group) for issue_group in
                [new_issues, maybe_new_issues, old_issues]
            ]
            setattr(
                self._context, 'filtered_diff_{}_{}'.format(len(filter), hash(filter)),
                (
                    [issue.key for issue in new_issues],
                    [issue.key for issue in maybe_new_issues],
                    [issue.key for issue in old_issues],
                )
            )

        else:
            logging.debug('No filter provided, all diff issues will be used')
        return new_issues, maybe_new_issues, old_issues

    @property
    def diff_components(self):
        if not self._context.diff_components:
            new_issues, maybe_new_issues, old_issues = self.get_filtered_diff(self._parameters.scope_filter)

            components = set()
            for issue_group in [new_issues, maybe_new_issues, old_issues]:
                for issue in issue_group:
                    components.update(self._diff_get_components_method(issue))
            self._context.diff_components = sorted(components)
            logging.info('Diff components: %s', self._context.diff_components)
        return self._context.diff_components

    def is_case_in_diff(self, case):
        if self._parameters.diff_type == "disabled":
            return True
        elif self._parameters.diff_type == "functionalities_diff":
            return self.is_case_in_functionalities_diff(case)
        else:
            return self.is_case_in_component_diff(case)

    def is_case_in_component_diff(self, case):
        return self.get_component(case) in self.diff_components

    def is_case_in_functionalities_diff(self, case):
        return should_run_case_in_regression(self.diff_functionalities_tree, case)

    @sandbox.common.utils.singleton_property
    def diff_functionalities_tree(self):
        return FunctionalitiesTree(Functionality.from_str(f) for f in self.diff_functionalities)

    @property
    def diff_functionalities(self):
        if not self._context.diff_functionalities:
            new_issues, maybe_new_issues, old_issues = self.get_filtered_diff(self._parameters.scope_filter)

            functionalities = set()
            for issue_group in [new_issues, maybe_new_issues, old_issues]:
                for issue in issue_group:
                    functionalities.update(get_issue_functionalities(issue))

            self._context.diff_functionalities = [
                unicode(functionality)
                for functionality in functionalities
            ]
            logging.debug('diff functionalities: %s', self._context.diff_functionalities)
        return self._context.diff_functionalities
