import os
import json
import requests
import datetime
import logging

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox.projects.browser.autotests.allure_parser import AllureReport

ELASTIC_TESTING_URL = 'http://test.elastic.browser.yandex.net'
ELASTIC_URL = 'http://elastic.browser.yandex.net:9200'

AUTOTEST_TASK_PARAMETERS = [
    'browser_version',
    'distribution_type',
    'platform',
    'framework_branch',
    'browser_branch',
    'browser_build_type_id',
    'isolates_build_type_id',
    'browser_build_params',
    'teamcity_builds_tags',
    'config_branch',
    'config_file',
    'build_id',
    'fake_build_id',
    'browser_tests_build_id',
    'build_type',
    'brands',
    'tests_folders',
    'tests_collect_by',
    'test_collect_condition',
    'tests_launch_platforms',
    'tests_additional_params',
    'regression_issue',
    'setup_type',
    'custom_setup',
]

TEST_PARAMETERS = [
    'full_name',
    'name',
    'suite',
    'test_type',
    'component',
    'duration',
    'start_time',
    'stop_time',
    'platform',
    'test_id',
    'status',
    'failure_message',
]


def get_index_name():
    return 'autotests-{}'.format(datetime.datetime.utcnow().strftime('%Y.%m.%d'))


def send_autotests_report(bulk_upload=False, data=None):
    """
    :param bool bulk_upload: flag indicating that bulk API should be used
    :param str data: data to send to elastic
    """

    if not data:
        logging.error('No data to send to elastic')
        raise Exception('No data to send to elastic')

    base_url = '{elastic_url}/{index_name}/tests/{bulk}'.format(
        elastic_url=ELASTIC_URL,
        index_name=get_index_name(),
        bulk='_bulk/' if bulk_upload else '',
    )

    session = requests.Session()
    retry = Retry(total=5, backoff_factor=0.3,
                  method_whitelist=frozenset(['GET', 'POST']))
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)

    if bulk_upload:
        headers = {'Content-Type': 'application/x-ndjson'}
    else:
        headers = {'Content-Type': 'application/json'}

    response = session.post(base_url, data=data, headers=headers)
    if not bulk_upload:
        logging.debug(response)
    response.raise_for_status()

    response = response.json()
    # bulk API response is {items: [{index: {status: ...}}]}
    # single document response is {status: ...}
    report_statuses = response.get('items') or [response]
    inserted = 0
    for status in report_statuses:
        if status.get('index', status).get('result') == 'created':
            inserted += 1
        else:
            logging.info(status)
    logging.info('Data to elastic sent: %d tests', inserted)


def construct_bulk_insert(report_commands):
    """
    :param list[dict] report_commands: list of dicts with test data
    :returns str: string with ndjson for bulk insert
    """

    if not report_commands:
        logging.error('No commands for bulk insert')
        raise Exception('No commands for bulk insert')

    insert_command = {
        "index": {
            "_index": get_index_name(),
            "_type": "tests",
        },
    }

    bulk_commands = []
    for report_command in report_commands:
        bulk_commands.append(json.dumps(insert_command))
        bulk_commands.append(json.dumps(report_command).replace(r'\n', ' '))
    bulk_commands.append('')  # bulk upload should always end with \n
    return '\n'.join(bulk_commands)


def extract_task_context(autotests_task):
    """
    :param sdk2.Task autotests_task: autotest launcher task
    :returns dict: task params
    """

    logging.info('Getting info about task #%s', autotests_task.id)
    param_dict = dict(autotests_task.Parameters)
    task_context = {
        param_name: param_dict.get(param_name)
        for param_name in AUTOTEST_TASK_PARAMETERS
    }
    task_context['sandbox_task_id'] = autotests_task.id
    task_context['task_type'] = str(autotests_task.type)
    return task_context


def get_test_parameters(test):
    logging.debug(test.full_name)
    all_parameters_exist = True
    full_test_info = {}
    for parameter_name in TEST_PARAMETERS:
        full_test_info[parameter_name] = getattr(test, parameter_name, None)

        if full_test_info[parameter_name] is None:
            logging.error('%s not found for test %s', parameter_name, test.full_name)
            all_parameters_exist = False
    return full_test_info, all_parameters_exist


def get_tests_from_task_allure(autotests_task, allure_report_path):
    """
    :param sdk2.Task autotests_task: autotest launcher task
    :param sdk2.Resource allure_resource: resource with allure AutotestsAllureReport
    :returns None|list[dict]: list of tests data dicts
    """

    task_info = extract_task_context(autotests_task)
    if autotests_task.parent:
        parent_task_info = extract_task_context(autotests_task.parent)
        task_info.update((key, value)
                         for key, value in parent_task_info.iteritems()
                         if value is not None)
    allure_report = AllureReport(os.path.join(allure_report_path, 'data'),
                                 ya_resources=None)

    all_successful = True
    report_commands = []
    logging.info('%d tests found', len(allure_report.tests))
    for test in allure_report.tests:
        full_test_info, all_parameters_exist = get_test_parameters(test)
        full_test_info.update({'task_info': task_info})
        report_commands.append(full_test_info)
        all_successful = all_successful and all_parameters_exist
    return report_commands, all_successful
