# -*- coding: utf-8 -*-
import logging
import urlparse

import itertools
import uuid

import jinja2
from collections import defaultdict

import time

import sandbox
from sandbox.projects.browser.autotests.regression.common import load_suites_info
from sandbox.common.types.task import Status
from sandbox.common.types import notification as ctn
from sandbox.projects.browser.autotests.regression.BrowserWaitTestpalmRuns import BrowserWaitTestpalmRuns
from sandbox.projects.browser.autotests.regression.assessors.BrowserStartHitmanProcess import (
    BrowserStartHitmanProcesses)
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport import (
    ROBOT_USERAGENT, ST_TEST_BASE_API_URL, ST_BASE_API_URL
)
from sandbox.projects.browser.autotests.testpalm_helper import (
    TestpalmClientWrapper, testsuite_url, version_url
)
from sandbox.projects.browser.autotests_qa_tools.common import (
    html_link, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, is_dev_sandbox, TEAMCITY_URL)
from sandbox import sdk2
from sandbox.projects.browser.autotests.regression.conf import MAX_ASSESSOR_RUN
from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt


class RunsCreator(object):
    def __init__(self, tickets_queue, task, testpalm_version_template, run_title_template,
                 ticket_jinja_template_package, ticket_jinja_template_name, ticket_summary_template,
                 default_group,
                 ticket_properties=None, ticket_jinja_template_args=None,
                 testrun_properties=None, unique_testpalm_version=True):
        self.ticket_properties = ticket_properties if ticket_properties else {}
        self.ticket_jinja_template_args = ticket_jinja_template_args if ticket_jinja_template_args else {}
        self.ticket_summary_template = ticket_summary_template
        self.ticket_jinja_template_name = ticket_jinja_template_name
        self.ticket_jinja_template_package = ticket_jinja_template_package
        self.default_group = default_group
        self.testrun_properties = testrun_properties if testrun_properties else {}
        self.run_title_template = run_title_template
        self.unique_testpalm_version = unique_testpalm_version
        self.testpalm_version_template = testpalm_version_template
        self.task = task
        self.tickets_queue = tickets_queue

    @property
    @sandbox.common.utils.singleton
    def jinja_template(self):
        env = jinja2.Environment(
            loader=jinja2.PackageLoader(self.ticket_jinja_template_package))
        template = env.get_template(self.ticket_jinja_template_name)
        template.globals['html'] = html_link
        template.globals['testsuite_url'] = testsuite_url
        return template

    def create_group_ticket(self, group, runs, versions, main_ticket=None):
        description = self.jinja_template.render(
            runs=runs, group=group,
            testpalm_versions=[version_url(project, version) for project, version in versions.iteritems()],
            **self.ticket_jinja_template_args)
        issue_dict = {
            'queue': self.tickets_queue,
            'summary': self.ticket_summary_template.format(group=group.name),
            'priority': 'critical',
            'description': description,
            'assignee': group.head,
            'unique': uuid.uuid4().hex,
        }
        issue_dict.update(self.ticket_properties)
        if main_ticket:
            issue_dict['parent'] = main_ticket
        return self.task.startrek_client.issues.create(**issue_dict)['key']

    @sandbox.common.utils.singleton
    def get_version(self, project, group):
        version_title = self.testpalm_version_template.format(group=group.name)
        if self.unique_testpalm_version:
            return self.task.testpalm_client.create_unique_version(project, version_title)
        else:
            return self.task.testpalm_client.get_or_create_version(version_title, project)

    def create_runs(self, version, group_cases, suite_info, platforms):
        '''
        :param version: testpalm version id
        :param grouped_cases: {group:{component: list(list(test_case))}}
        :return: list(TestRuns) – created runs
        '''
        runs = []
        for component, component_cases in group_cases.iteritems():
            for part, cases in enumerate(component_cases, 1):
                for platform in platforms:
                    title = self.run_title_template.format(version=version, part=part, component=component,
                                                           platform=platform)
                    logging.debug("TRY CREATE RUN suite: {} cases: {}".format(suite_info.id, len(list(cases))))
                    if len(list(cases)) > 0:
                        run_id = self.task.testpalm_client.create_testrun(
                            version=version, cases=cases, project=suite_info.project, title=title,
                            properties=self.testrun_properties, current_environment=platform,
                            tags=suite_info.tags,
                        )
                        runs.append(self.task.testpalm_client.get_testrun(run_id, suite_info.project))
                        time.sleep(3)  # creating runs is hard, let testpalm rest for 3 seconds
                    else:
                        logging.debug("SKIP enpty run creation")
        return runs

    def process_everything(self, grouped_runs, platforms, main_ticket=None):
        runs = defaultdict(dict)
        tickets = {}
        for group, group_runs in grouped_runs.iteritems():
            testpalm_versions = {}  # {project: version_id}
            for group_cases, suite_info in group_runs:
                suite = self.task.testpalm_client.get_testsuite(suite_info.id, suite_info.project)
                version = self.get_version(suite_info.project, group)
                testpalm_versions[suite_info.project] = version
                runs[group][suite] = self.create_runs(version, group_cases, suite_info, platforms[suite_info.id])
            ticket = self.create_group_ticket(group, runs[group], testpalm_versions, main_ticket)
            tickets[group] = ticket
        return tickets, runs


class RegressionResult(object):
    def __init__(self, main_ticket,
                 manual_tickets,
                 manual_runs,
                 assessors_tickets,
                 assessors_runs):
        self.main_ticket = main_ticket
        self.manual_tickets = manual_tickets
        self.manual_runs = manual_runs
        self.assessors_tickets = assessors_tickets
        self.assessors_runs = assessors_runs


class BaseRegressionInit(sdk2.Task):

    assessors_testrun_properties_to_show = ''

    def is_assessors(self, case):
        raise NotImplementedError()

    def get_component(self, case):
        raise NotImplementedError()

    def create_main_ticket(self):
        raise NotImplementedError()

    @property
    def default_group(self):
        raise NotImplementedError()

    @property
    def whom_notify_about_asessors_start(self):
        raise NotImplementedError()

    # link to build that assessors/tolokers will see in yang
    @property
    def test_stend(self):
        raise NotImplementedError()

    @property
    def whom_notify_from_hitman(self):
        raise NotImplementedError()

    def get_group(self, component):
        return self.default_group

    @property
    def assessors_runs_creator(self):
        raise NotImplementedError()

    @property
    def manual_runs_creator(self):
        raise NotImplementedError()

    @property
    def requester_code(self):
        """
        Parameter for hitman process to check, that assessors test specified build.
        If not needed return None
        :return: str
        """
        raise NotImplementedError()

    @property
    def affected_version(self):
        """
        Used in hitman https://st.yandex-team.ru/ASSESSORTEST-1665
        :return: str
        """
        return None

    class Context(sdk2.Context):
        main_ticket = None
        manual_tickets = {}
        manual_runs = {}
        assessors_tickets = {}
        assessors_runs = {}
        monitor_assessors_task_id = None
        monitor_manuak_task_id = None

    class Parameters(sdk2.Parameters):
        test_suites = sdk2.parameters.JSON(
            'Testsuites and platforms',
            default={
                'testsuites': [
                    {
                        'project': '',
                        'id': '',
                        'assessors_platforms': [],
                        'manual_platforms': [],
                    }
                ]
            }
        )
        main_regression_ticket_assignee = sdk2.parameters.String('Regression assignee')
        with sdk2.parameters.Group('Assessors parameters') as assessors_params:
            @property
            def hitman_process_id(self):
                raise NotImplementedError()

            @property
            def assessors_quota(self):
                raise NotImplementedError()

            booking_id = sdk2.parameters.Integer(
                'Booking id', required=True,
                description='Id from <a href="https://booking.yandex-team.ru">booking</a>.')
            assessors_launch_comment = sdk2.parameters.String('Assessors launch comment', multiline=True)
            max_run_duration = sdk2.parameters.Integer('Testrun max duration', required=True, default=MAX_ASSESSOR_RUN)
            deadline = sdk2.parameters.Integer(
                'Deadline for assessors', default=24, description='In hours', required=True)
            start_hitman_jobs_automatically = sdk2.parameters.Bool(
                'Start hitman jobs automatically', default=False,
                description='If checked, assessors will get their tasks after testruns are created without any approve')

    def check_input(self):
        from marshmallow import ValidationError
        fail_reasons = []
        try:
            load_suites_info(self.test_suites)
        except ValidationError as e:
            logging.exception('Failed to parse JSON')
            fail_reasons.append(u'Невалидный JSON: ' + e.message)
        return fail_reasons

    def validate_input(self):
        fail_reasons = self.check_input()
        if fail_reasons:
            self.set_info(
                ''.join([u'Ошибка в параметрах таски:<ul>'] +
                        ['<li>{}</li>'.format(problem) for problem in fail_reasons] +
                        ['</ul>']), do_escape=False
            )
            raise TaskFailure(u'Ошибка во входных данных')

    def _slice_cases(self, cases, max_duration=None):
        sliced_cases, current_slice = [], []
        current_estimate = 0
        for case in cases:
            current_estimate += case.estimate
            if max_duration is not None and current_estimate > max_duration:
                current_estimate = case.estimate
                sliced_cases.append(current_slice)
                current_slice = []
            current_slice.append(case)
        if current_slice:
            sliced_cases.append(current_slice)
        return sliced_cases

    def _slice_cases_by_component(self, cases, max_duration=None):
        """
        :param cases list(TestCase)
        :param max_duration max run duration in ms
        :return: {TestingGroup: {component : [[TestCase, TestCase], [...]]}}
        """
        result = defaultdict(dict)
        sorted_cases = sorted(cases, key=self.get_component)
        for component, cases in itertools.groupby(sorted_cases, key=self.get_component):
            sliced_cases = self._slice_cases(cases, max_duration)
            result[self.get_group(component)][component] = sliced_cases
        return result

    def load_suite_cases(self, suite_info):
        return self.testpalm_client.get_testsuite_cases(
            suite_info.id, suite_info.project,
            include_fields=('isAutotest', 'stats', 'id', 'estimatedTime',
                            'name', 'attributes', 'properties', 'status', 'removed'))

    def get_cases(self, suites_info):
        assessors_cases = defaultdict(list)
        manual_cases = defaultdict(list)
        for suite_info in suites_info.testsuites:
            testcases = self.load_suite_cases(suite_info)
            for case in testcases:
                if self.is_assessors(case):
                    assessors_cases[suite_info].append(case)
                else:
                    manual_cases[suite_info].append(case)
        return assessors_cases, manual_cases

    def _group_cases(self, grouped_cases, max_duration=None):
        groups = defaultdict(list)
        for suite_info, testcases in grouped_cases.iteritems():
            grouped_cases = self._slice_cases_by_component(testcases, max_duration=max_duration)
            for group, group_cases in grouped_cases.iteritems():
                groups[group].append((group_cases, suite_info))
        return groups

    def group_cases(self, assessors_cases, manual_cases):
        grouped_assessors = self._group_cases(assessors_cases,
                                              max_duration=self.Parameters.max_run_duration * 60 * 1000)
        grouped_manual = self._group_cases(manual_cases)
        return grouped_assessors, grouped_manual

    def get_assessors_deadline(self, assessors_runs):
        return self.Parameters.deadline

    def get_assessors_versions_info(self, assessors_runs, assessors_tickets):
        versions = defaultdict(lambda: defaultdict(str))
        for group, group_runs in assessors_runs.iteritems():
            for component, component_runs in group_runs.iteritems():
                if component_runs:
                    versions[group][component_runs[0].project] = component_runs[0].version

        return [
            {
                "ticket": assessors_tickets[group],
                "group": group.name,
                "telegram": group.telegram,
                "version_id": version,
                "testpalm_project": project,
            }
            for group, group_projects in versions.iteritems()
            for project, version in group_projects.iteritems()
        ]

    def create_assessors_task(self, assessors_runs, assessors_tickets):
        child = BrowserStartHitmanProcesses(
            self,
            versions=self.get_assessors_versions_info(assessors_runs, assessors_tickets),
            hitman_process_id=self.Parameters.hitman_process_id,
            deadline=self.get_assessors_deadline(assessors_runs),
            requester_code=self.requester_code,
            notifications=[
                sdk2.Notification(
                    [Status.FAILURE, Status.SUCCESS, Status.NO_RES, Status.EXCEPTION, Status.TIMEOUT, Status.EXPIRED],
                    [self.whom_notify_about_asessors_start],
                    ctn.Transport.EMAIL
                )
            ] if self.whom_notify_about_asessors_start else [],
            requester=self.whom_notify_from_hitman,
            quota=self.Parameters.assessors_quota,
            test_stend=self.test_stend,
            special_condition=self.Parameters.assessors_launch_comment,
            affected_version=self.affected_version,
            booking_id=self.Parameters.booking_id,
            testrun_properties_to_show=self.assessors_testrun_properties_to_show,
            check_booking=False,
        )
        if self.Parameters.start_hitman_jobs_automatically:
            child.enqueue()
            self.set_info(
                u'Создана задача по запуску процессов в Хитмане: {}'.format(
                    html_link(get_task_link(child.id))
                ), do_escape=False)
        else:
            self.set_info(
                u'Создан черновик задачи по запуску процессов в Хитмане: {}\n'
                u'Запустите задачу, если все раны для асессоров созданы верно.'.format(
                    html_link(get_task_link(child.id))
                ), do_escape=False)

    def _reformat_runs(self, created_runs, created_issues=None):
        created_issues = created_issues if created_issues else {}
        return {
            group.name: [
                {
                    'suite_id': suite.id,
                    'project': suite.project,
                    'testruns': [run.id for run in runs],
                    'issue': created_issues.get(group)
                }
                for suite, runs in suites.iteritems()
            ]
            for group, suites in created_runs.iteritems()
        }

    def start_runs_monitoring(self, assessors_runs, manual_runs, manual_tickets, main_ticket):
        monitor_assessors = BrowserWaitTestpalmRuns(
            self, description=self.Parameters.description + u' асессорские раны',
            runs=self._reformat_runs(assessors_runs),
            show_estimate=True
        ).enqueue()
        self.Context.monitor_assessors_task_id = monitor_assessors.id
        monitor_manual = BrowserWaitTestpalmRuns(
            self, description=self.Parameters.description + u' ручные раны',
            runs=self._reformat_runs(manual_runs, manual_tickets),
            main_regression_issue=main_ticket,
        ).enqueue()
        self.Context.monitor_manual_task_id = monitor_manual.id

    def on_create(self):
        if not self.Parameters.main_regression_ticket_assignee:
            self.Parameters.main_regression_ticket_assignee = (
                self.author if not is_dev_sandbox()
                else 'robot-bro-qa-infra'  # author is always "guest" on dev sandbox and there is no such user on staff
            )

    @property
    def all_manual_oses(self):
        res = set()
        for suite in self.test_suites['testsuites']:
            res.update(suite['manual_platforms'])
        return res

    @property
    @sandbox.common.utils.singleton
    def test_suites(self):
        return self.Parameters.test_suites

    def on_execute(self):
        self.validate_input()
        suites_info = load_suites_info(self.test_suites)
        assessors_cases, manual_cases = self.get_cases(suites_info)
        assessors_groups, manual_groups = self.group_cases(assessors_cases, manual_cases)

        main_ticket = self.create_main_ticket()
        assessors_tickets, assessors_runs = self.assessors_runs_creator.process_everything(
            assessors_groups, {suite.id: suite.assessors_platforms for suite in suites_info.testsuites}, main_ticket)
        manual_tickets, manual_runs = self.manual_runs_creator.process_everything(
            manual_groups, {suite.id: suite.manual_platforms for suite in suites_info.testsuites}, main_ticket)
        self.set_info(
            (u'Главный тикет на регрессию: {}</br>'.format(
                html_link(urlparse.urljoin(self.st_base_url, main_ticket)))
             if main_ticket else '') +
            (u'Тикеты на ручное тестирование:</br>{}</br>'.format(
                '<br/>'.join('{}: {}'.format(group.name, html_link(urlparse.urljoin(self.st_base_url, ticket)))
                             for group, ticket in manual_tickets.iteritems()))
             if manual_tickets else '') +
            (u'Тикеты на тестирование асессорами:</br>{}</br>'.format(
                '<br/>'.join('{}: {}'.format(group.name, html_link(urlparse.urljoin(self.st_base_url, ticket)))
                             for group, ticket in assessors_tickets.iteritems()))
             if assessors_tickets else ''),
            do_escape=False)

        if assessors_runs:
            self.create_assessors_task(assessors_runs, assessors_tickets)
        self.start_runs_monitoring(assessors_runs, manual_runs, manual_tickets, main_ticket)
        res = RegressionResult(main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs)
        self.save_regression_result_in_context(res)
        return res

    def save_regression_result_in_context(self, result):
        self.Context.main_ticket = result.main_ticket
        self.Context.manual_tickets = {group.name: issue for group, issue in result.manual_tickets.iteritems()}
        self.Context.assessors_tickets = {group.name: issue for group, issue in result.assessors_tickets.iteritems()}
        self.Context.assessors_runs = self._reformat_runs(result.assessors_runs, result.assessors_tickets)
        self.Context.manual_runs = self._reformat_runs(result.manual_runs, result.manual_tickets)
        self.Context.assessors_versions = self.get_assessors_versions_info(
            result.assessors_runs, result.assessors_tickets)

    @sdk2.report(title=u"Асессорские раны")
    def assessors_runs(self):
        if not self.Context.monitor_assessors_task_id:
            return u'Нет информации про раны'

        task = sdk2.Task.find(id=self.Context.monitor_assessors_task_id, children=True).limit(1).first()
        return task.Context.report

    @sdk2.report(title=u"Ручные раны")
    def manual_runs(self):
        if not self.Context.monitor_manual_task_id:
            return u'Нет информации про раны'

        task = sdk2.Task.find(id=self.Context.monitor_manual_task_id, children=True).limit(1).first()
        return task.Context.report

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
            PipEnvironment('testpalm-api-client', version='4.0.2'),
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
            PipEnvironment('jsonschema==2.5.1'),
            PipEnvironment('marshmallow==3.0.0b5'),
        ]
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='browser-old-starship-regression')
            ],
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def st_base_url(self):
        return 'https://st.test.yandex-team.ru' if is_dev_sandbox() else 'https://st.yandex-team.ru'

    @property
    @sandbox.common.utils.singleton
    def startrek_client(self):
        from startrek_client import Startrek
        return Startrek(
            useragent=ROBOT_USERAGENT,
            token=sdk2.Vault.data('st-test-token') if is_dev_sandbox()
            else sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT),
            base_url=ST_TEST_BASE_API_URL if is_dev_sandbox() else ST_BASE_API_URL)

    @property
    @sandbox.common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        )

    @property
    @sandbox.common.utils.singleton
    def testpalm_client(self):
        from testpalm_api_client.client import TestPalmClient
        return TestpalmClientWrapper(TestPalmClient(oauth_token=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)))
