# -*- coding: utf-8 -*-
import logging

from requests import HTTPError

import sandbox
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt

from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport import (
    ROBOT_USERAGENT, ST_TEST_BASE_API_URL, ST_BASE_API_URL)
from sandbox.projects.browser.autotests.regression.conf import MAX_TESTCASE_STAT_ESTIMATE
from sandbox.projects.browser.autotests.testpalm_helper import TestpalmClientWrapper
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT, is_dev_sandbox

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


TIME_TO_WAIT = 20 * 60


class BrowserWaitTestpalmRuns(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('testpalm-api-client', version='4.0.2'),
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
        ]

        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='browser-regression-wait-runs')
            ],
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        runs = sdk2.parameters.JSON('Runs to monitor', default={
            'group_name': [
                {
                    'suite_id': '',
                    'project': '',
                    'testruns': [],
                    'issue': '',
                }
            ]
        })
        main_regression_issue = sdk2.parameters.String('Main regression issue', default=None,
                                                       description='To report runs statistics')
        show_estimate = sdk2.parameters.Bool('Show estimate time for runs', default=False)

    class Context(sdk2.Context):
        report = ''
        finished_runs = {}  # id: html
        pretty_suites = {}  # id: html

    def on_execute(self):
        try:
            report = ''
            unfinished_runs = False
            for group_name, group_runs in self.Parameters.runs.iteritems():
                report += u'Раны для группы {}:</br>'.format(group_name)
                for runs_info in group_runs:
                    suite_id = runs_info['suite_id']
                    project = runs_info['project']
                    runs = runs_info['testruns']
                    report += self.Context.pretty_suites.get(
                        suite_id,
                        u'Сьют {}:<ul>'.format(
                            self.testpalm_client.get_testsuite(suite_id, project).pretty_str())
                    )
                    for run_id in runs:
                        try:
                            if run_id in self.Context.finished_runs:
                                report += self.Context.finished_runs[run_id]
                            else:
                                run = self.testpalm_client.get_testrun(
                                    run_id, project,
                                    include_fields=('resolution', 'id', 'title', 'archived', 'statistics', 'estimate')
                                )
                                pretty_run = '<li>{}{}</li>'.format(
                                    run.pretty_str(print_case_statuses=True,
                                                   show_estimate=self.Parameters.show_estimate),
                                    u' ✓' if run.finished else '')
                                if run.finished:
                                    self.Context.finished_runs[run_id] = pretty_run
                                unfinished_runs |= not run.finished
                                report += pretty_run
                        except HTTPError as ex:
                            if ex.response.status_code == 404:
                                # must be archived run
                                continue
                            else:
                                raise

                    report += '</ul>'
            self.Context.report = report
        except HTTPError:
            message = 'Request to testpalm failed even after retries.'
            self.set_info('{} Will wait {} minutes and try again'.format(message, TIME_TO_WAIT / 60))
            logging.exception(message)
            unfinished_runs = True
        if unfinished_runs:
            raise sdk2.WaitTime(TIME_TO_WAIT)
        else:
            self.report_statistic_to_issues()

    def report_statistic_to_issues(self):
        def update_issue(issue_obj, cases_count, runs_duration):
            fields = {'verificationWorkTime': 'PT{}S'.format(runs_duration / 1000), }
            if not is_dev_sandbox():
                fields['numberOfRequests'] = cases_count
            self.startrek_client.issues.update(issue_obj, params={'notify': False}, **fields)

        overall_cases_count, overall_runs_duration = 0, 0
        for group_name, group_runs in self.Parameters.runs.iteritems():
            cases_count, runs_duration = 0, 0
            issue = None
            for runs_info in group_runs:
                issue = runs_info.get('issue')
                for run_id in runs_info['testruns']:
                    try:
                        run = self.testpalm_client.get_testrun(
                            run_id, runs_info['project'],
                            include_fields=('testGroups.testCases.duration', 'resolution'))
                    except HTTPError as ex:
                        if ex.response.status_code == 404:
                            # must be archived run
                            continue
                        else:
                            raise

                    runs_duration += sum(
                        min(case_info['duration'], MAX_TESTCASE_STAT_ESTIMATE)
                        for group in run.testGroups
                        for case_info in group['testCases']
                    )
                    cases_count += run.resolution['counter']['total']

            if issue and cases_count:
                update_issue(self.startrek_client.issues[issue], cases_count, runs_duration)
            overall_cases_count += cases_count
            overall_runs_duration += runs_duration
        if self.Parameters.main_regression_issue:
            update_issue(self.startrek_client.issues[self.Parameters.main_regression_issue],
                         overall_cases_count, overall_runs_duration)

    @sdk2.footer()
    def footer(self):
        return self.Context.report

    @property
    @sandbox.common.utils.singleton
    def testpalm_client(self):
        from testpalm_api_client.client import TestPalmClient
        return TestpalmClientWrapper(TestPalmClient(oauth_token=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)))

    @property
    @sandbox.common.utils.singleton
    def startrek_client(self):
        from startrek_client import Startrek
        return Startrek(
            useragent=ROBOT_USERAGENT,
            token=sdk2.Vault.data('st-test-token') if is_dev_sandbox()
            else sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT),
            base_url=ST_TEST_BASE_API_URL if is_dev_sandbox() else ST_BASE_API_URL)
