from datetime import datetime

import sandbox
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.regression import assessors
from sandbox.projects.browser.autotests.regression.assessors import ASSESSORS_BUCKET
from sandbox.projects.browser.autotests_qa_tools.common import split_list
from sandbox.sandboxsdk.environments import PipEnvironment


class BrowserClearAssessorsS3Bucket(sdk2.Task):
    name = 'BROWSER_CLEAR_ASSESSORS_S3_BUCKET'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment("boto3", use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        months = sdk2.parameters.Integer(
            'Months', description='Files and folders older than this number of months will be deleted.'
                                  'E.g. if now is april 2019 and month=1 then folder "2019-02" and all'
                                  ' previous will be deleted', required=True)

    def on_execute(self):
        now = datetime.now()
        common_prefixes = self.s3_client.list_objects(Bucket=ASSESSORS_BUCKET, Delimiter='/')['CommonPrefixes']
        date_prefix_format = assessors.S3_KEY_DATE_FORMAT + '/'
        prefixes = [
            prefix['Prefix'] for prefix in common_prefixes
            if (now - datetime.strptime(prefix['Prefix'], date_prefix_format)).days / 30 > self.Parameters.months
        ]

        all_objects_to_delete = [
            {'Key': k}
            for folder in prefixes
            for k in [
                obj['Key'] for obj in
                self.s3_client.list_objects(Bucket=ASSESSORS_BUCKET, Prefix=folder).get('Contents', [])
            ]
        ]

        for objects_pack_to_delete in split_list(all_objects_to_delete, 100):
            self.s3_client.delete_objects(Bucket=ASSESSORS_BUCKET, Delete={'Objects': objects_pack_to_delete})

    @property
    @sandbox.common.utils.singleton
    def s3_client(self):
        import boto3

        return assessors.make_s3_client(
            boto3,
            aws_access_key_id=sdk2.Vault.data('robot-bro-qa-infra-s3-key-id'),
            aws_secret_access_key=sdk2.Vault.data('robot-bro-qa-infra-s3-secret-key'),
        )
