# -*- coding: utf-8 -*-
import re
import time

from sandbox.common.errors import TaskFailure
from sandbox import common
from sandbox import sdk2
from sandbox.projects.browser.autotests.classes.ya_booking import get_booking_state
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression.conf import get_group_priority
from sandbox.projects.browser.autotests.regression.assessors.BrowserStartHitmanProcess.hitman_client import (
    HitmanClient, HITMAN_URL, JobStatus)
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment

WAIT_PROCESSES_PERIOD = 2  # how often check hitman processes status, in minutes
DEFAULT_ASESSORS_PRIORITY = 1


class BrowserStartHitmanProcesses(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        environments = [
            PipEnvironment('startrek_client', version='1.7.0', use_wheel=True),
            PipEnvironment('ya-booking-client==1.0.1'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        booking_id = sdk2.parameters.Integer(
            'Booking id', required=True,
            description='Id from <a href="https://booking.yandex-team.ru">booking</a>')
        check_booking = sdk2.parameters.Bool(
            'Check booking', default=True,
            description='Check if booking is actual before starting hitman task. '
                        'Uncheck only if you know, what ypu\'re doing'
        )
        versions = sdk2.parameters.JSON('Versions', default=[
            {'version_id': '',
             'testpalm_project': '',
             'ticket': '',
             'telegram': '',
             'group': ''}
        ])
        test_stend = sdk2.parameters.String('Test stend', description='Build to test', multiline=True, required=True)
        quota = sdk2.parameters.String('Quota', description='Which project quota to use')
        requester = sdk2.parameters.String(
            'Requester', description='This value will be passed as REQUESTER parameter to hitman process')
        requester_code = sdk2.parameters.String(
            'Verification code', description='Code to verify that assessor tested right version.'
                                             ' Revision of browser build for example.')
        hitman_process_id = sdk2.parameters.String('Hitman process', default='testing_browser_automatic')
        deadline = sdk2.parameters.Integer('Deadline for assessors', default=24, description='In hours')
        special_condition = sdk2.parameters.String(
            'Special condition', description='Use to tell assessors something other than usual instructions')
        affected_version = sdk2.parameters.String(
            'Affected versions', description='affected versions separated by comma')
        testrun_properties_to_show = sdk2.parameters.String(
            'Testrun properties to show', description='Testrun properties to show for asessors separated by comma',
            default="")
        advanced_hitman_process_properties = sdk2.parameters.JSON(
            'Advanced hitman process properties',
            default={}
        )

    class Context(sdk2.Context):
        jobs_info = {}

    @property
    @common.utils.singleton
    def hitman_client(self):
        return HitmanClient(HITMAN_URL, sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @property
    @common.utils.singleton
    def startrek_client(self):
        return self.clients.startrek

    @property
    @common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @decorators.retries(10, delay=2, backoff=2)
    def start_process(self, process_id, properties, token):
        return self.hitman_client.start_process(process_id, self.Parameters.requester or self.author,
                                                properties=properties, job_token=token)

    @decorators.retries(10, delay=2, backoff=2)
    def get_job_status(self, job_id):
        return self.hitman_client.get_job_status(job_id)

    def check_booking(self):
        booking_check_state = None
        if self.Parameters.booking_id and self.Parameters.check_booking:
            booking_check_state = get_booking_state(self.clients.ya_booking, self.Parameters.booking_id)
            if booking_check_state["problems"]:
                raise TaskFailure(u'Ошибка проверки брони: \n{}'.format(u'\n'.join(booking_check_state["problems"])))
            if booking_check_state["available_time"] < 60:
                raise TaskFailure(
                    u'Ошибка проверки брони: бронь booking_id={} просрочена'.format(self.Parameters.booking_id))
        return booking_check_state

    def on_execute(self):

        with self.memoize_stage.check_booking:
            self.check_booking()

        with self.memoize_stage.create_job_info:
            self.Context.jobs_info = [dict(process_params, status=None, id=None,
                                           priority=get_group_priority(process_params['group'],
                                                                       DEFAULT_ASESSORS_PRIORITY))
                                      for process_params in self.Parameters.versions]

        _groups = set([re.split(r'-', job_info['group'])[0] for job_info in self.Context.jobs_info])
        common_params = dict(self.Parameters.advanced_hitman_process_properties)
        common_params.update({
            'DEADLINE': self.Parameters.deadline,
            'TEST_STEND': self.Parameters.test_stend,
            'REQUESTER_CODE': self.Parameters.requester_code or '',
            'PARENT_NAMESPACE': self.Parameters.quota or '',
            'SPECIAL_CONDITION': self.Parameters.special_condition or '',
            'TASKS_SORT_REGEX_BY_VERSION_NAME': r'(\\d+\\.\\d+\\.\\d+).*({}).*'.format("|".join(_groups)),
            'SET_TICKET_AFFECTED_VERSIONS': self.Parameters.affected_version or '',
            'BOOKING_ID': self.Parameters.booking_id or '',  # empty string is used to pass 'null' to nirvana.
            'TESTRUN_PROPERTIES_TO_SHOW': self.Parameters.testrun_properties_to_show
        })

        last_asessors_priority = None
        for job_info in sorted(self.Context.jobs_info, key=lambda x: x["priority"]):
            status = job_info['status']
            job_id = job_info['id']
            version = job_info['version_id']
            project = job_info['testpalm_project']

            if not job_id:
                # see https://st.yandex-team.ru/BYIN-8567
                if last_asessors_priority is not None and last_asessors_priority != job_info['priority']:
                    self.set_info(u"Новый приоритет задачи={} Пауза".format(job_info['priority']))
                    time.sleep(95)
                    self.check_booking()

                last_asessors_priority = job_info['priority']
                token = '{}-{}-{}'.format(self.id, version, project)
                properties = dict(common_params)
                properties.update({
                    'VERSION': version,
                    'NAMESPACE': project,
                    'TICKET': job_info['ticket'],
                    'TELEGRAM': job_info.get('telegram', '')
                })

                job_id = self.start_process(
                    properties=properties,
                    token=token, process_id=self.Parameters.hitman_process_id)['id']
                status = JobStatus.NEW
                self.set_info(u'Запущена <a href="{link}">задача</a> для группы {group} (проект {project})'.format(
                    group=job_info['group'], project=project, link=self.hitman_client.get_job_url(job_id),
                    ), do_escape=False)
            else:
                if status not in JobStatus.Group.FINISHED:
                    status = self.get_job_status(job_id)['status']

            job_info.update({'status': status, 'id': job_id})

        if all(job_info['status'] in JobStatus.Group.FINISHED or job_info['status'] not in JobStatus.Group.ALL for
               job_info in self.Context.jobs_info):
            task_info = []
            for job_info in self.Context.jobs_info:
                ticket_id = job_info['ticket']

                issue = self.startrek_client.issues[ticket_id]
                issue.comments.create(
                    text=u'Отправка асессорам тестранов из проекта {project} завершилась со статусом (({url} {status}))'.format(
                        project=job_info['testpalm_project'], url=self.hitman_client.get_job_url(job_info['id']),
                        status=job_info['status']))
                task_info.append(
                    u'<a href="{link}">Задача</a> для группы {group} (проект {project}) завершилась со статусом {status}'.format(
                        link=self.hitman_client.get_job_url(job_info['id']), group=job_info['group'],
                        project=job_info['testpalm_project'], status=job_info['status']
                    ))
            self.set_info('</br>'.join(task_info), do_escape=False)
            if any(job_info['status'] not in JobStatus.Group.SUCCESS for job_info in self.Context.jobs_info):
                raise TaskFailure('Какая-то хитман-задача не завершилась успешно')
        else:
            raise sdk2.WaitTime(WAIT_PROCESSES_PERIOD * 60)
