import os
import tempfile
import shutil
from requests import HTTPError

from sandbox import sdk2, common
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.autotests.regression import assessors
from sandbox.projects.browser.autotests_qa_tools.common import (
    BRANDED_BUILDS_PROJECT, get_platform, html_link, build_number_tuple, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL,
    DEFAULT_DESKTOP_DISTRIBUTION_CHANNEL)
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment


ASSESSORS_ARTIFACTS = {
    'Win': {  # testpalm_field: (s3 name, teamcity artifact path)
        'build': ('FULL_{version}_{distribution_name}.exe', '{distribution_type}/{distribution_name}/Yandex.exe'),
        'lite_installer': (
            'LITE_{version}_{distribution_name}.exe', '{distribution_type}/{distribution_name}/lite/Yandex.exe'),
        'arc': ('browser-setup.arc', 'browser-setup.arc'),
        'mini_installer': ('mini_installer.exe', '{distribution_type}/{distribution_name}/update/mini_installer.exe'),
        'rss': ('update.rss', '{distribution_type}/{distribution_name}/update/update.rss'),
    },
    'Mac': {
        'build': ('FULL_{version}_{distribution_name}.dmg', '{distribution_type}/{distribution_name}/Yandex.dmg'),
        'rss': ('update.xml', '{distribution_type}/{distribution_name}/update.xml'),
    },
    'Linux': {
        'build': ('FULL_{version}_{distribution_name}.deb',
                  '{distribution_type}/{distribution_name}/yandex-browser-{distribution_channel}_{version}-1_amd64.deb'),
    }
}

ASSESSORS_ARTIFACTS_FAKE = {
    'Win': {
        'fake': ('FAKE_{version}_{distribution_name}.exe', '{distribution_type}/{distribution_name}/Yandex.exe'),
        'fake_mini_installer':
            ('fake_mini_installer.exe', '{distribution_type}/{distribution_name}/update/mini_installer.exe'),
        'fake_rss': ('fake_update.rss', '{distribution_type}/{distribution_name}/update/update.rss'),
    },
    'Mac': {
        'fake': ('FAKE_{version}_{distribution_name}.dmg', '{distribution_type}/{distribution_name}/Yandex.dmg'),
        'fake_rss': ('fake_update.xml', '{distribution_type}/{distribution_name}/update.xml'),
    }
}


class BrowserUploadBrandedBuildToS3(sdk2.Task):
    name = 'BROWSER_UPLOAD_BRANDED_BUILD_TO_S3'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
            PipEnvironment('boto3', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        distribution_type = sdk2.parameters.String('Distribution type', required=True, default='brands',
                                                   choices=[(_, _) for _ in ['brands', 'partners']])
        distribution_name = sdk2.parameters.String('Brand/Partner name', required=True, default='yandex')
        distribution_channel = sdk2.parameters.String(
            'Distribution channel, for linux only, see BYIN-14198',
            required=False, default=DEFAULT_DESKTOP_DISTRIBUTION_CHANNEL)
        assessors_artifacts = sdk2.parameters.JSON('Artifacts structure template for build upload',
                                                   required=True,
                                                   default=ASSESSORS_ARTIFACTS)
        assessors_artifacts_fake = sdk2.parameters.JSON('Artifacts structure template for fake_build upload',
                                                        required=True,
                                                        default=ASSESSORS_ARTIFACTS_FAKE)
        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build '
                        '(<a href="{base}/project.html?projectId={project}">'
                        '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
            required=True
        )
        fake_build_id = sdk2.parameters.Integer(
            'Fake browser build id',
            description='Id of fake browser teamcity build '
                        '(<a href="{base}/project.html?projectId={project}">'
                        '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
        )

    class Context(sdk2.Context):
        build_problems = []

    def download_from_teamcity(self, build, distribution_type, distribution_name, distribution_channel,
                               download_dir, s3_name_template, tc_path_template):
        version = build.number.split('/')[0]
        s3_file_name = s3_name_template.format(version=version, distribution_name=distribution_name)
        teamcity_path = tc_path_template.format(
            version=version, distribution_name=distribution_name, distribution_type=distribution_type,
            distribution_channel=distribution_channel)
        try:
            build.download_artifact(teamcity_path, os.path.join(download_dir, s3_file_name))
        except HTTPError as e:
            if e.response.status_code == 404:
                problem = 'Teamcity build #{} does not have artifact {}'.format(build.id, teamcity_path)
                self.Context.build_problems.append(problem)
                raise TaskFailure(problem)
            else:
                raise

        return s3_file_name

    def upload_files_to_s3(self, distribution_type, distribution_name, distribution_channel, task_id, build, fake_build=None):
        platform = get_platform(build)
        res = {}
        download_dir = tempfile.mkdtemp()
        s3_key_prefix = assessors.get_s3_key_prefix(task_id)
        try:
            for file_type, (s3_name_template, tc_path_template) in self.Parameters.assessors_artifacts[platform].iteritems():
                res[file_type] = self.download_from_teamcity(build, distribution_type, distribution_name,
                                                             distribution_channel, download_dir,
                                                             s3_name_template, tc_path_template)
            if fake_build:
                for file_type, (s3_name_template, tc_path_template) in self.Parameters.assessors_artifacts_fake.get(platform,
                                                                                                                    {}).iteritems():
                    res[file_type] = self.download_from_teamcity(fake_build, distribution_type, distribution_name,
                                                                 distribution_channel, download_dir,
                                                                 s3_name_template, tc_path_template)

            for file_to_upload in res.values():
                src = os.path.join(download_dir, file_to_upload)
                assessors.upload_file_to_s3(self.s3_client, src, s3_key_prefix, file_to_upload)
        finally:
            shutil.rmtree(download_dir)
        res = {
            file_type: assessors.get_s3_proxy_url(s3_key_prefix, filename)
            for file_type, filename in res.iteritems()
        }
        return res

    def on_save(self):
        self.Context.distribution_type = self.Parameters.distribution_type
        self.Context.distribution_name = self.Parameters.distribution_name
        self.Context.build_id = self.Parameters.build_id

    def on_execute(self):
        browser_build = self.teamcity_client.Build(id=self.Parameters.build_id)
        fake_build = (self.teamcity_client.Build(id=self.Parameters.fake_build_id)
                      if self.Parameters.fake_build_id else None)
        if fake_build:
            browser_build, fake_build = sorted([browser_build, fake_build], key=build_number_tuple)
        urls = self.upload_files_to_s3(self.Parameters.distribution_type, self.Parameters.distribution_name,
                                       self.Parameters.distribution_channel, self.id, browser_build, fake_build)
        self.set_info('<br/>'.join(html_link(url) for url in urls.values()), do_escape=False)
        self.Context.uploaded_files = urls

    @property
    @common.utils.singleton
    def s3_client(self):
        import boto3

        return assessors.make_s3_client(
            boto3,
            aws_access_key_id=sdk2.Vault.data('robot-bro-qa-infra-s3-key-id'),
            aws_secret_access_key=sdk2.Vault.data('robot-bro-qa-infra-s3-secret-key'),
        )

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        )
