from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.regression import assessors
from sandbox.projects.browser.autotests_qa_tools.common import html_link, ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2, common


class BrowserUploadTCArtifactToAssessorsS3(sdk2.Task):
    name = 'BROWSER_UPLOAD_TC_ARTIFACT_TO_ASSESSORS_S3'

    class Requirements(sdk2.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==5.0.0'),
            PipEnvironment('boto3', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        build_id = sdk2.parameters.Integer(
            'Browser build id',
            description='Id of browser teamcity build',
            required=True,
        )
        artifact_file_path = sdk2.parameters.String(
            'Build artifact file path',
            description='Path to the build artifact to be uploaded',
            required=True,
        )
        s3_filename = sdk2.parameters.String(
            'S3 filename',
            description='Name of the artifact file after uploading to S3',
            required=True,
        )

    class Context(sdk2.Context):
        uploaded_file_url = None

    def get_artifact(self, teamcity_build, artifact_path):
        path, filename = artifact_path.rsplit('/', 1)
        artifacts = [a for a in teamcity_build.artifacts(path) if a.name == filename]
        if len(artifacts) != 1:
            raise TaskFailure('Expected that teamcity build #{} has one artifact "{}" but found {}'
                              .format(teamcity_build.id, artifact_path, len(artifacts)))
        artifact = artifacts[0]
        if not artifact.is_file:
            raise TaskFailure('Expected that artifact "{}" from teamcity build #{} be a file'
                              .format(artifact_path, teamcity_build.id))
        return artifact

    def upload_file_to_s3(self, teamcity_build, artifact_path, s3_file_name):
        s3_key_prefix = assessors.get_s3_key_prefix(self.id)
        artifact = self.get_artifact(teamcity_build, artifact_path)
        with artifact.download() as artifact_stream:
            assessors.upload_fileobj_to_s3(s3_client=self.s3_client,
                                           fileobj=artifact_stream,
                                           s3_key_prefix=s3_key_prefix,
                                           filename=s3_file_name)
        return assessors.get_s3_proxy_url(s3_key_prefix, s3_file_name)

    def on_execute(self):
        build = self.teamcity_client.Build(id=self.Parameters.build_id)
        artifact_path = self.Parameters.artifact_file_path
        s3_file_name = self.Parameters.s3_filename
        url = self.upload_file_to_s3(build, artifact_path, s3_file_name)
        self.set_info(html_link(url), do_escape=False)
        self.Context.uploaded_file_url = url

    @property
    @common.utils.singleton
    def s3_client(self):
        import boto3
        return assessors.make_s3_client(
            boto3,
            aws_access_key_id=sdk2.Vault.data('robot-bro-qa-infra-s3-key-id'),
            aws_secret_access_key=sdk2.Vault.data('robot-bro-qa-infra-s3-secret-key'),
        )

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)
        )
