# -*- coding: utf-8 -*-
import datetime
import logging
import os
import urlparse
import yaml

import jinja2
from collections import defaultdict, OrderedDict
import itertools

import sandbox
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport import ROBOT_USERAGENT, ST_BASE_API_URL
from sandbox.projects.browser.autotests.classes.functionalities import (
    get_issue_functionalities, FunctionalitiesTree, should_run_case_in_regression,
    Functionality)
from sandbox.projects.browser.autotests.regression.dbro.DbroRegressionInit.test_suite_configs import TEST_SUITE_CONFIGS
from sandbox.projects.browser.autotests.regression.dbro.DiffTickets import ISSUES_QUERY
from sandbox.projects.browser.autotests.regression.dbro.DiffTickets.builds_diff_issues import (
    get_diff_tickets, get_issue_obj, get_report
)
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.autotests.BrowserAutotestRun import CONFIG_NAMES_TEMPLATES
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserAutotestRunRegression import BrowserAutotestRunRegression
from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.autotests.regression.assessors.BrowserUploadBrandedBuildToS3 import (
    BrowserUploadBrandedBuildToS3
)
from sandbox.projects.browser.autotests.regression.conf import (
    TestingGroup, REPORT_OVERHEAD, TSP_HOURS, REGRESSION_MAILLIST)
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds
from sandbox.projects.browser.autotests.BrowserAutotestTestpalmExport.const import TESTSUITES_WIN, TESTSUITES_MAC
from sandbox.projects.browser.autotests.regression.BaseRegressionInit import BaseRegressionInit, RunsCreator
from sandbox.projects.browser.autotests.regression.conf import get_group as get_group_from_config
from sandbox.projects.browser.autotests_qa_tools.common import (
    TESTS_BUILDS_PROJECT, BRANDED_BUILDS_PROJECT, build_number_tuple, is_dev_sandbox, html_link, html_cut,
    check_browser_build, check_tests_build, get_platform, get_browser_version_number, get_st_browser_version,
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, TEAMCITY_URL)

TEST_SUITE_CONFIGS_PATH = os.path.join(os.path.dirname(__file__), 'test_suite_configs')
ASSESSORS_FILES_ORDER = [
    'build',
    'lite_installer',
    'alice',
    'arc',
    'mini_installer',
    'rss',
    'fake',
    'fake_mini_installer',
    'fake_rss',
]


class DbroRegressionInit(BaseRegressionInit):
    class Context(BaseRegressionInit.Context):
        deadline = None
        assessors_links = {}
        diff_components = None

    class Parameters(BaseRegressionInit.Parameters):
        # this parameter is overridden below, so that it would be hidden in ui,
        # if test suites should be loaded from config file
        test_suites = None
        deadline = None  # assessors deadline is calculated by number of cases
        hitman_process_id = sdk2.parameters.String('Hitman process', default='testing_browser_automatic',
                                                   required=True)
        assessors_quota = sdk2.parameters.String('Assessors launch quota', default='brocase', required=True)
        start_hitman_jobs_automatically = BaseRegressionInit.Parameters.start_hitman_jobs_automatically(default=True)
        with sdk2.parameters.Group('DBRO specific parameters') as dbro_params:
            regression_type = sdk2.parameters.String(
                'Regression type', required=True, choices=[(_, _) for _ in TEST_SUITE_CONFIGS + ['custom']],
                ui=sdk2.parameters.String.UI('select')
            )
            with regression_type.value['custom']:
                test_suites_override = BaseRegressionInit.Parameters.test_suites()
            distribution_type = sdk2.parameters.String('Distribution type', required=True, default='brands',
                                                       choices=[(_, _) for _ in ['brands', 'partners']])
            distribution_name = sdk2.parameters.String('Brand/Partner name', required=True, default='yandex')
            autotests_pack = sdk2.parameters.String(
                'Autotests launching config', default='all_custom_{}.json',
                choices=[(_, _) for _ in CONFIG_NAMES_TEMPLATES + ['']],
                description='Leave empty ("Any") if no autotests needed'
            )
            check_tickets_from_diff = sdk2.parameters.Bool('Aggregators needed', default=False,
                                                           description='Whether to list issues from diff'
                                                                       ' to manual testing issues for resolve checks')
            old_build_id = sdk2.parameters.Integer('Previous build id',
                                                   description='Use to calculate diff tickets')
            build_id = sdk2.parameters.Integer(
                'Browser build id',
                description='Id of browser teamcity build '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
                required=True
            )
            fake_build_id = sdk2.parameters.Integer(
                'Fake browser build id',
                description='Id of fake browser teamcity build '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
            )
            browser_tests_build_id = sdk2.parameters.Integer(
                'Tests build id',
                description='Id of teamcity build with browser tests '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=TESTS_BUILDS_PROJECT),
            )
            configuration_ticket = sdk2.parameters.String('Configuration ticket')
            release_ticket = sdk2.parameters.String('Release ticket')
            regression_deadline = sdk2.parameters.Integer('Regression deadline', required=True,
                                                          description='Regression duration in hours')
            manual_launch_comment = sdk2.parameters.String('Manual launch comment', multiline=True,
                                                           description='Special parameters for manual runs')
            link_for_tolokers = sdk2.parameters.String('Test stend for tolokers', description='link to yandex disk')
            after_autotests_suites = sdk2.parameters.JSON(
                'After autotests suites',
                default={
                    'win': TESTSUITES_WIN,
                    'mac': TESTSUITES_MAC
                }
            )

    def on_enqueue(self):
        if self.Parameters.regression_type != 'custom' and self.Parameters.test_suites_override != {}:
            self.Parameters.test_suites_override = {}

    def on_execute(self):
        if not is_dev_sandbox():
            info = u'<font color="red">' \
                   u"<p>Вы используете устаревшую версию инструмента для создания регрессии. Данная версия инструмента отключена." \
                   u"<p>Для создания регрессии воспользуйтесь sandbox-таской <b>RUN_DISTRIBUTION_DESKTOP_BROWSER_REGRESSION</b>" \
                   u"<br><p>Отличия описаны <a href='https://nda.ya.ru/t/1LKZeQPW3VwDUP'>тут</a>" \
                   u"<p>Чат поддержки: <a href='https://t.me/joinchat/EdxsYxG7mSf7Q9LDnZ6e3g'>Browser QA tools support</a>" \
                   u"</font>"
            self.set_info(info, do_escape=False)
            raise TaskFailure()

        if not self.Context.deadline:
            self.Context.deadline = (datetime.datetime.now() + datetime.timedelta(
                hours=self.Parameters.regression_deadline)).date().isoformat()
        self.validate_input()
        browser_build, fake_build, tests_build = self.get_builds()
        self.wait_builds([build for build in [browser_build, fake_build, tests_build] if build])
        assessors_links = self.upload_files_to_s3(browser_build, fake_build)
        self.Context.assessors_links = assessors_links
        res = super(DbroRegressionInit, self).on_execute()
        self.update_tickets(res)
        self.launch_autotests(
            res.main_ticket, {group.name: ticket for group, ticket in res.manual_tickets.iteritems()})
        self.publish_diff_tickets(res.manual_tickets, res.main_ticket)

    @property
    @sandbox.common.utils.singleton
    def test_suites(self):
        if self.Parameters.regression_type != 'custom':
            with open(os.path.join(TEST_SUITE_CONFIGS_PATH, self.Parameters.regression_type)) as suites:
                return yaml.load(suites)
        else:
            return self.Parameters.test_suites_override

    # override base method to filter out cases, that are not suitable for build's platform
    def load_suite_cases(self, suite_info):
        cases = super(DbroRegressionInit, self).load_suite_cases(suite_info)
        ok_cases = []
        filtered_out = []
        for case in cases:
            if not any(os.startswith(self.platform) for os in case.oses):
                filtered_out.append((case, u'Кейс не для {} платформы'.format(self.platform)))
            elif (suite_info.filter_by_diff and
                  not should_run_case_in_regression(self.get_diff_functionalities(), case)):
                filtered_out.append((case, u'Функциональности кейса не затронуты в диффе'))
            else:
                ok_cases.append(case)
        if filtered_out:
            logging.info(u'Следующие кейсы не попадут в раны:')
            for case, reason in filtered_out:
                logging.info('{}: {}'.format(case.url, reason))
        return ok_cases

    def check_input(self):
        browser_build, fake_build, tests_build = self.get_builds()
        build_problems = super(DbroRegressionInit, self).check_input()
        build_problems.extend(check_browser_build(browser_build))
        if fake_build:
            build_problems.extend(check_browser_build(fake_build))
        if tests_build:
            build_problems.extend(check_tests_build(tests_build))
        if (self.platform != 'Linux' and self.Parameters.autotests_pack) and (not fake_build or not tests_build):
            build_problems.append(u'Для автотестов требуется фейковая сборка браузера и сборка с тестами')
        if self.Parameters.check_tickets_from_diff and not self.Parameters.old_build_id:
            build_problems.extend(u'Требуется проверка тикетов из диффа тестированием,'
                                  u' но не указана предыдущая сборка для подсчета диффа')
        return build_problems

    def is_assessors(self, case):
        return 'Asessors' in case.mapped_attributes.get('Checklist', [])

    def get_component(self, case):
        return case.mapped_attributes['Component'][0]

    def get_group(self, component):
        return get_group_from_config(component, self.default_group)

    @property
    def whom_notify_about_asessors_start(self):
        return REGRESSION_MAILLIST

    @property
    def whom_notify_from_hitman(self):
        return REGRESSION_MAILLIST

    @property
    def test_stend(self):
        return self.Context.assessors_links['build'] + (
            u'\nСсылка на сборку в Яндекс.Диск: {}'.format(self.Parameters.link_for_tolokers)
            if self.Parameters.link_for_tolokers else ''
        )

    default_group = TestingGroup('UnknownComponents', 'https://t.me/joinchat/ENYckkOPDdt41iN5GfRd4A', [], ['edragun'])
    default_hitman_process = 'testing_browser_automatic'

    @property
    def requester_code(self):
        browser_build, fake_build, tests_builds = self.get_builds()
        return browser_build.last_changes[0].version

    @property
    def affected_version(self):
        return self.st_version

    @decorators.retries(5, delay=1, backoff=2)
    def get_issue_obj(self, issue):
        return self.startrek_client.issues[issue]

    @sandbox.common.utils.singleton
    @decorators.retries(5, delay=1, backoff=2)
    def create_startrek_version_if_not_exists(self, st_queue, st_version):
        if st_version not in [_v.name for _v in self.startrek_client.queues[st_queue].versions]:
            self.startrek_client.versions.create(name=st_version, queue=st_queue)
        return st_version

    def get_assessors_deadline(self, assessors_runs):
        return 24 if 'master' not in self.Parameters.regression_type.lower() else 48

    @sandbox.common.utils.singleton
    def assessors_runs_properties(self):
        res = OrderedDict()
        res['Brand/PartnerID'] = self.Parameters.distribution_name

        res.update(
            (assessors_file, self.Context.assessors_links[assessors_file])
            for assessors_file in ASSESSORS_FILES_ORDER
            if assessors_file in self.Context.assessors_links
        )
        return res

    @property
    @sandbox.common.utils.singleton
    def assessors_runs_creator(self):
        return RunsCreator(
            tickets_queue='BROASESSORS', task=self,
            ticket_jinja_template_package='sandbox.projects.browser.autotests.regression.dbro.DbroRegressionInit',
            ticket_jinja_template_name='assessors_ticket.jinja',
            ticket_summary_template=''.join(
                '[{}]'.format(v) for v in [
                    'Regression', self.version_number, self.platform, self.Parameters.regression_type,
                ]
            ) + u'[{group}] Прогон регрессионных тестов',
            ticket_jinja_template_args={'build': self.Context.assessors_links['build'],
                                        'assessors_launch_comment': self.Parameters.assessors_launch_comment},
            ticket_properties={'priority': 'critical',
                               'components': ['Regression', self.platform],
                               'deadline': self.Context.deadline,
                               'affectedVersions': [self.create_startrek_version_if_not_exists(
                                   'BROASESSORS',
                                   self.st_version)],
                               },
            default_group=self.default_group,
            testrun_properties=self.assessors_runs_properties(),
            testpalm_version_template='_'.join(
                [self.version_number, self.platform, self.Parameters.regression_type]
            ) + '{group}',
            run_title_template='[{version}][{platform}][{component}] Part {part}',
            unique_testpalm_version=True)

    @property
    @sandbox.common.utils.singleton
    def manual_runs_creator(self):
        return RunsCreator(
            tickets_queue='BROWSER', task=self,
            ticket_jinja_template_package='projects.browser.autotests.regression.dbro.DbroRegressionInit',
            ticket_jinja_template_name='manual_ticket.jinja',
            ticket_summary_template=''.join(
                '[{}]'.format(v) for v in [
                    'Regression', self.version_number, self.platform, self.Parameters.regression_type
                ]
            ) + u'[{group}] Регрессия',
            ticket_jinja_template_args={'launch_comment': self.Parameters.manual_launch_comment},
            ticket_properties={'priority': 'critical',
                               'components': ['Regression', self.platform],
                               'deadline': self.Context.deadline,
                               'affectedVersions': [self.create_startrek_version_if_not_exists(
                                   'BROWSER',
                                   self.st_version)],
                               'tags': ['main'],
                               },
            default_group=self.default_group,
            testpalm_version_template='_'.join([
                self.version_number, self.Parameters.regression_type,
                self.platform, 'Manual',
            ]),
            run_title_template=''.join('[{}]'.format(v) for v in [
                'Regression', self.version_number,
                self.Parameters.regression_type, ]) + '[{platform}][{component}] Manual run',
            testrun_properties={'Brand/PartnerID': self.Parameters.distribution_name},
            unique_testpalm_version=False)

    def create_main_ticket(self):
        browser_build, fake_build, tests_build = self.get_builds()
        env = jinja2.Environment(
            loader=jinja2.PackageLoader('projects.browser.autotests.regression.dbro.DbroRegressionInit'))
        template = env.get_template('main_ticket.jinja')
        old_build = self.Parameters.old_build_id
        description = template.render(
            brand=self.Parameters.distribution_name,
            oses=self.all_manual_oses, release_ticket=self.Parameters.release_ticket,
            configuration_ticket=self.Parameters.configuration_ticket, browser_build=browser_build.web_url,
            fake_build=fake_build.web_url if fake_build else None,
            tests_build=tests_build.web_url if tests_build else None,
            previous_build=self.teamcity_client.Build(id=old_build).web_url if old_build else None,
            launch_comment=self.Parameters.manual_launch_comment)
        issue_dict = {
            'description': description,
            'assignee': self.Parameters.main_regression_ticket_assignee,
            'queue': 'BROWSER',
            'summary': u'[{version}][{platform}][{type}] Регрессия'.format(
                version=self.version_number, platform=self.platform, type=self.Parameters.regression_type
            ),
            'deadline': self.Context.deadline,
            'affectedVersions': [self.create_startrek_version_if_not_exists(
                'BROWSER',
                self.st_version)],
            'priority': 'critical',
            'components': ['Regression', self.platform],
        }
        return self.startrek_client.issues.create(**issue_dict)['key']

    @property
    @sandbox.common.utils.singleton
    def version_number(self):
        browser_build, fake_build, tests_builds = self.get_builds()
        return get_browser_version_number(browser_build)

    @property
    @sandbox.common.utils.singleton
    def platform(self):
        browser_build, fake_build, tests_builds = self.get_builds()
        return get_platform(browser_build)

    @property
    @sandbox.common.utils.singleton
    def st_version(self):
        browser_build, fake_build, tests_builds = self.get_builds()
        return get_st_browser_version(browser_build)

    @sandbox.common.utils.singleton
    def get_builds(self):
        browser_build = self.teamcity_client.Build(id=self.Parameters.build_id)
        fake_build = (self.teamcity_client.Build(id=self.Parameters.fake_build_id)
                      if self.Parameters.fake_build_id else None)
        tests_build = (self.teamcity_client.Build(id=self.Parameters.browser_tests_build_id)
                       if self.Parameters.browser_tests_build_id else None)
        if fake_build and browser_build.state == 'finished' and fake_build.state == 'finished':
            browser_build, fake_build = sorted([browser_build, fake_build], key=build_number_tuple)
        return browser_build, fake_build, tests_build

    def get_diff_functionalities(self):
        if not self.Context.diff_functionalities:
            new_issues, maybe_new_issues, old_issues, issues_by_query = self.get_diff()

            functionalities = set()
            for issue_group in [new_issues, maybe_new_issues, old_issues]:
                for issue in issue_group:
                    functionalities.update(get_issue_functionalities(issue))

            self.Context.diff_functionalities = [
                str(functionality)
                for functionality in functionalities
            ]
            self.set_info(u'Дифф затрагивает следующие функциональности: {}'.format(
                ', '.join(sorted(self.Context.diff_functionalities))))
        return FunctionalitiesTree(Functionality.from_str(f) for f in self.Context.diff_functionalities)

    def get_diff(self):
        if self.Parameters.old_build_id:
            if not self.Context.diff_issues:
                query = ISSUES_QUERY.format(version=self.st_version)
                new_issues, maybe_new_issues, old_issues, issues_by_query, report = get_diff_tickets(
                    project='stardust', repo='browser', old_build_id=self.Parameters.old_build_id,
                    new_build_id=self.Parameters.build_id, teamcity_client=self.teamcity_client,
                    bb_client=self.bitbucket_client, startrek_client=self.prod_startrek_client, query=query)

                self.Context.diff_issues = (
                    [issue.key for issue in new_issues],
                    [issue.key for issue in maybe_new_issues],
                    [issue.key for issue in old_issues],
                    [issue.key for issue in issues_by_query],
                )

            return (
                [get_issue_obj(self.prod_startrek_client, issue) for issue in issues]
                for issues in self.Context.diff_issues
            )
        raise TaskFailure(u'Невозможно посчитать дифф, не указана предыдущая сборка')

    def publish_diff_tickets(self, manual_tickets, main_ticket):
        if self.Parameters.check_tickets_from_diff:
            query = ISSUES_QUERY.format(version=self.st_version)
            new_issues, maybe_new_issues, old_issues, issues_by_query = self.get_diff()
            old_build_url = self.teamcity_client.Build(id=self.Parameters.old_build_id).web_url
            new_build_url = self.teamcity_client.Build(id=self.Parameters.build_id).web_url,

            group_issues = defaultdict(
                lambda: {
                    'new_issues': list(),
                    'maybe_new_issues': list(),
                    'old_issues': list(),
                    'issues_by_query': list(),
                    'testingStoryPoints': 0,
                }
            )
            for key, issues in {'new_issues': new_issues, 'maybe_new_issues': maybe_new_issues,
                                'old_issues': old_issues, 'issues_by_query': issues_by_query}.iteritems():
                for issue in issues:
                    if hasattr(issue, 'status') and issue.status.key in ['tested', 'closed']:
                        continue
                    components = [component.display for component in getattr(issue, 'components', [])]
                    if not components:
                        groups = [self.default_group]
                    else:
                        groups = {self.get_group(component) for component in components}
                        if len(groups) > 1:
                            groups.discard(self.default_group)
                    logging.debug('issue %s going to groups %s', issue.key, {group.name for group in groups})
                    for group in groups:
                        group_issues[group][key].append(issue)
                        group_issues[group]['testingStoryPoints'] += getattr(issue, 'testingStoryPoints', 0) or 0

            env = jinja2.Environment(
                loader=jinja2.PackageLoader(
                    'sandbox.projects.browser.autotests.regression.dbro.DbroRegressionInit', 'templates'),
            )
            template = env.get_template('builds_diff_issues_startrek.jinja')
            for group, diff_issues in group_issues.iteritems():
                group_ticket = manual_tickets.get(group)
                if not group_ticket:
                    group_ticket = self.manual_runs_creator.create_group_ticket(group, {}, {}, main_ticket)
                    self.set_info(u'Создан тикет для проверки диффа группой {}: {}'.format(
                        group.name, html_link(urlparse.urljoin(self.st_base_url, group_ticket))
                    ), do_escape=False)
                group_ticket_obj = self.get_issue_obj(group_ticket)
                diff_report = template.render(
                    old_build_url=old_build_url,
                    new_build_url=new_build_url,
                    maybe_new_issues=group_by_priority(diff_issues['maybe_new_issues']),
                    new_issues=group_by_priority(diff_issues['new_issues']),
                    old_issues=group_by_priority(diff_issues['old_issues']),
                    issues_by_query=group_by_priority(diff_issues['issues_by_query']),
                    query=query
                ).encode('utf-8')
                fields = {'description': u'{}\n==== Тикеты для проверки\n{}'.format(
                    group_ticket_obj.description, diff_report)}
                if not is_dev_sandbox():
                    fields['testingStoryPoints'] = (group_ticket_obj.testingStoryPoints or 0) + diff_issues[
                        'testingStoryPoints']

                self.startrek_client.issues.update(group_ticket_obj, **fields)
            if main_ticket:
                report = get_report(old_build_url, new_build_url, new_issues, maybe_new_issues,
                                    old_issues, issues_by_query, query=query)

                main_ticket_obj = self.startrek_client.issues[main_ticket]
                totaltsp = sum(diff_issues['testingStoryPoints'] for group, diff_issues in group_issues.iteritems())
                fields = {
                    'description': u'{}\n=== Дифф по тикетам\n<#{}#>'.format(
                        main_ticket_obj.description, html_cut(u'Дифф по тикетам', report))}
                if not is_dev_sandbox():
                    fields['testingStoryPoints'] = (main_ticket_obj.testingStoryPoints or 0) + totaltsp
                self.startrek_client.issues.update(main_ticket_obj, **fields)

    def update_tickets(self, regression_result):
        for group, assessors_ticket in regression_result.assessors_tickets.iteritems():
            estimate_ms = self.count_assessors_time(regression_result.assessors_runs[group])
            self.startrek_client.issues.update(
                self.startrek_client.issues[assessors_ticket],
                qaEstimation='PT{}S'.format(estimate_ms / 1000)
            )
        total_tsp = 0
        for group, manual_ticket in regression_result.manual_tickets.iteritems():
            issue = self.startrek_client.issues[manual_ticket]
            estimate_ms = sum(run.fixed_average_duration
                              for suite, suite_runs in regression_result.manual_runs[group].iteritems()
                              for run in suite_runs)
            time_to_test_ms = estimate_ms * REPORT_OVERHEAD
            tsp = round(time_to_test_ms / 1000.0 / 60 / 60 / TSP_HOURS, 1)
            total_tsp += tsp
            fields = {}
            if not is_dev_sandbox():
                fields['testingStoryPoints'] = tsp
            self.startrek_client.issues.update(issue, **fields)
            for suite, suite_runs in regression_result.manual_runs[group].iteritems():
                for run in suite_runs:
                    issue.remotelinks.create('relates', 'testrun/{}/{}'.format(run.project, run.id),
                                             'ru.yandex.testpalm')
        if not is_dev_sandbox():
            self.startrek_client.issues.update(
                self.startrek_client.issues[regression_result.main_ticket],
                testingStoryPoints=total_tsp
            )

    def count_assessors_time(self, runs):
        all_time = 0
        for suite, suite_runs in runs.iteritems():
            # Assessors cases are run on different environments just to check same case with different assessors
            # Here we count time only from one of environments
            # https://st.yandex-team.ru/BYIN-7264
            environment = suite_runs[0].currentEnvironment['title']
            all_time += sum(run.estimate_with_default for run in
                            filter(lambda r: r.currentEnvironment['title'] == environment, suite_runs))
        return all_time

    def upload_files_to_s3(self, build, fake_build=None):
        child = self.find(BrowserUploadBrandedBuildToS3).first()
        if child:
            if child.status != Status.SUCCESS:
                raise TaskFailure('Failed to upload builds to s3: {}'.format(
                    ', '.join(child.Context.build_problems or [])
                ))
            return child.Context.uploaded_files
        raise sdk2.WaitTask(
            BrowserUploadBrandedBuildToS3(
                self,
                distribution_type=self.Parameters.distribution_type,
                distribution_name=self.Parameters.distribution_name,
                build_id=build.id,
                fake_build_id=fake_build.id if fake_build else None
            ).enqueue(),
            list(Status.Group.FINISH + Status.Group.BREAK),
            True,
        )

    def launch_autotests(self, main_ticket, group_tickets):
        browser_build, fake_build, tests_build = self.get_builds()
        if self.platform != 'Linux' and self.Parameters.autotests_pack:
            child = BrowserAutotestRunRegression(
                self,
                description='Regression autotests',
                config_file=self.Parameters.autotests_pack.format(self.platform.lower()),
                build_id=browser_build.id,
                fake_build_id=fake_build.id if fake_build else None,
                browser_tests_build_id=self.Parameters.browser_tests_build_id,
                regression_issue=main_ticket,
                group_issues=group_tickets,
                after_autotests_suites=self.Parameters.after_autotests_suites
            )
            if not is_dev_sandbox():
                child.enqueue()
            self.set_info(u'Запуск автотестов: {}'.format(html_link(get_task_link(child.id))), do_escape=False)

    def wait_builds(self, builds):
        if any(b.state != 'finished' for b in builds):
            raise sdk2.WaitTask(
                BrowserWaitTeamcityBuilds(
                    self,
                    description='Wait browser regression builds',
                    mode='WAIT_GIVEN',
                    builds=' '.join(str(build.id) for build in builds),
                    oauth_vault=ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
                ).enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )
        else:
            if any(b.status != 'SUCCESS' for b in builds):
                raise TaskFailure('Одна из сборок упала')

    @property
    @sandbox.common.utils.singleton
    def bitbucket_client(self):
        return bitbucket.BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token',
                                   sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @property
    @sandbox.common.utils.singleton
    def prod_startrek_client(self):
        from startrek_client import Startrek
        return Startrek(
            useragent=ROBOT_USERAGENT,
            token=sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT),
            base_url=ST_BASE_API_URL
        )


def group_by_priority(tickets):
    def get_priority(ticket):
        if hasattr(ticket, 'priority'):
            return ticket.priority.display
        else:
            return 'Unknown'

    tickets = sorted(tickets, key=get_priority)
    return OrderedDict((k, list(v)) for k, v in itertools.groupby(tickets, get_priority))
