import collections
from collections import defaultdict
import jinja2
import logging
import re

import sandbox
from sandbox.projects.browser.autotests.regression.conf import get_project_by_component

logger = logging.getLogger(__name__)

BUILD_ID_RE = re.compile(r'buildId=(?P<id>\d+)')
API_BUILD_ID_RE = re.compile(r'/(?P<id>\d+):id/')
ISSUES_RE = re.compile('[A-Z]+-[0-9]+')


@sandbox.common.utils.singleton
def get_priority_order(st_client, priority_id):
    return st_client.priorities[priority_id].order


def natsort_key(issue_obj, _numbers_regex=re.compile('([0-9]+)')):
    """ Returns key for natural sorting. """
    return [int(chunk) if chunk.isdigit() else chunk
            for chunk in re.split(_numbers_regex, issue_obj.key)]


def sort_issues(issue_objs, st_client):
    nat_sorted = sorted(issue_objs, key=natsort_key)
    return sorted(nat_sorted, reverse=True,
                  key=lambda issue: get_priority_order(st_client, issue.priority.id)
                  if hasattr(issue, 'priority') else 6)


def log_issues_and_commits(project, repo, bb, begin_commit, end_commit):
    commits = bb.get_commits(
        project, repo,
        since=begin_commit, until=end_commit, limit=400)
    issues = collections.defaultdict(set)
    for commit in commits:
        for issue in ISSUES_RE.findall(commit['message']):
            issues[issue].add(commit['id'])

    return issues


def build_revision(build):
    revisions = build.revisions
    assert len(revisions) == 1
    return revisions[0].version


def get_issue_obj(st_client, issue):
    import startrek_client.exceptions
    try:
        return st_client.issues[issue]
    except startrek_client.exceptions.Forbidden:
        return collections.namedtuple('IssueStub', ['key'])(issue)
    except startrek_client.exceptions.NotFound:
        return collections.namedtuple('IssueStub', ['key'])(issue)


def get_report(old_build_url, new_build_url, new_issue_objs, maybe_new_issue_objs, old_issue_objs,
               issues_by_query_objs, query=None):
    template = jinja2.Environment(
        loader=jinja2.PackageLoader(
            'projects.browser.autotests.regression.dbro.DiffTickets', 'templates'),
    ).get_template('builds_diff_issues.html')
    report = template.render(
        old_build_url=old_build_url,
        new_build_url=new_build_url,
        old_issues=old_issue_objs,
        new_issues=new_issue_objs,
        maybe_new_issues=maybe_new_issue_objs,
        issues_by_query=issues_by_query_objs,
        query=query,
    ).encode('utf-8')
    return report


def get_components(new_issues, maybe_new_issues, old_issues, issues_by_query):
    components_by_project = defaultdict(set)

    for issues in (new_issues, maybe_new_issues, old_issues, issues_by_query):
        for issue in issues:
            logging.debug('issues %s, components: %s', issue.key, getattr(issue, 'components', []))
            for component in getattr(issue, 'components', []):
                component = component.display

                if component in ['Installer+Updater', 'Browser-Update']:
                    components_by_project[get_project_by_component('Updater')].add('Updater')
                    components_by_project[get_project_by_component('Installer')].add('Installer')
                else:
                    components_by_project[get_project_by_component(component)].add(component)
    return components_by_project


def get_diff_tickets(project, repo, teamcity_client, bb_client, startrek_client,
                     old_build_id=None, new_build_id=None, old_commit=None, new_commit=None,
                     query=None):
    old_build = teamcity_client.Build(id=old_build_id) if old_build_id else None
    new_build = teamcity_client.Build(id=new_build_id) if new_build_id else None
    if not old_commit:
        old_commit = build_revision(old_build)
        new_commit = build_revision(new_build)
    common_commit = bb_client.merge_base(
        project, repo,
        (old_commit, new_commit, 'refs/heads/master'),
        octopus=True,
    )[0]

    logger.info('Common commit for master, %s and %s is %s',
                old_commit, new_commit, common_commit)

    old_build_issues = log_issues_and_commits(
        project, repo, bb_client, common_commit, old_commit)
    new_build_issues = log_issues_and_commits(
        project, repo, bb_client, common_commit, new_commit)

    new_issues = set()
    maybe_new_issues = set()
    old_issues = set()
    for issue in set(old_build_issues) | set(new_build_issues):
        old_commits = old_build_issues[issue]
        new_commits = new_build_issues[issue]

        logger.info("%s's commits in old build: %s", issue, old_commits)
        logger.info("%s's commits in new build: %s", issue, new_commits)

        if old_commits == new_commits:
            logger.info('%s has same commits in both builds', issue)
        if new_commits - old_commits:
            logger.info('%s has extra commits in new build', issue)
            if not old_commits:
                new_issues.add(issue)
            else:
                maybe_new_issues.add(issue)
        if old_commits - new_commits:
            logger.info('%s has extra commits in old build', issue)
            old_issues.add(issue)
    issues_by_query_objs = startrek_client.issues.find(query) if query else []
    issues_by_query_objs = sort_issues([issue for issue in issues_by_query_objs
                                        if issue.key not in set(new_issues) | set(maybe_new_issues) | set(old_issues)],
                                       startrek_client)

    new_issue_objs = sort_issues([get_issue_obj(startrek_client, issue) for issue in new_issues], startrek_client)
    maybe_new_issue_objs = sort_issues([get_issue_obj(startrek_client, issue) for issue in maybe_new_issues],
                                       startrek_client)
    old_issues_objs = sort_issues([get_issue_obj(startrek_client, issue) for issue in old_issues], startrek_client)

    report = get_report(old_build.web_url if old_build else None, new_build.web_url if new_build else None,
                        new_issue_objs=new_issue_objs,
                        maybe_new_issue_objs=maybe_new_issue_objs, old_issue_objs=old_issues_objs,
                        issues_by_query_objs=issues_by_query_objs, query=query)
    return new_issue_objs, maybe_new_issue_objs, old_issues_objs, issues_by_query_objs, report
