# -*- coding: utf-8 -*-
import jinja2

import sandbox
from sandbox import sdk2
from sandbox.projects.browser.autotests.regression.conf import TestingGroup
from sandbox.projects.browser.autotests.regression.BaseRegressionInit import BaseRegressionInit, RunsCreator


class CbroRegressionInit(BaseRegressionInit):
    class Parameters(BaseRegressionInit.Parameters):
        hitman_process_id = sdk2.parameters.String('Hitman process', required=True)
        assessors_quota = sdk2.parameters.String('Assessors launch quota')
        with sdk2.parameters.Group('CBRO specific parameters') as cbro_params:
            builds = sdk2.parameters.List('Builds', required=True)
            version = sdk2.parameters.String('Version', required=True)
            support_chat = sdk2.parameters.String('Assessors support chat', required=True)
            manual_ticket_queue = sdk2.parameters.String('Manual ticket queue', required=True)
            assessors_ticket_queue = sdk2.parameters.String('Assessors ticket queue', required=True)
            main_ticket_queue = sdk2.parameters.String('Main ticket queue')
            release_configuration = sdk2.parameters.String('Release configuration', multiline=True)
            common_regression_ticket = sdk2.parameters.String('Common regression ticket')
            manual_launch_comment = sdk2.parameters.String('Manual launch comment', multiline=True)
            grouping_field = sdk2.parameters.String(
                'Testpalm grouping filed', required=True, default='Functionality',
                description='Testcases with the same value in this field will be placed in same testruns')

    def check_input(self):
        return []

    def get_component(self, case):
        return case.mapped_attributes[self.Parameters.grouping_field][0]

    def is_assessors(self, case):
        return 'Asessors' in case.mapped_attributes.get('Checklist', [])

    @property
    def whom_notify_about_asessors_start(self):
        return None

    @property
    def whom_notify_from_hitman(self):
        return self.author

    @property
    def test_stend(self):
        return '\n'.join(self.Parameters.builds)

    @property
    @sandbox.common.utils.singleton
    def default_group(self):
        return TestingGroup('MBRO', self.Parameters.support_chat, [], [self.Parameters.main_regression_ticket_assignee])

    requester_code = None  # No need to check that assessors test right version

    def create_main_ticket(self):
        if self.Parameters.main_ticket_queue:
            env = jinja2.Environment(
                loader=jinja2.PackageLoader('projects.browser.autotests.regression.mbro.CbroRegressionInit'))
            template = env.get_template('main_ticket.jinja')
            description = template.render(
                oses=self.all_manual_oses,
                release_configuration=self.Parameters.release_configuration,
                builds=self.Parameters.builds,
                common_regression_ticket=self.Parameters.common_regression_ticket,
                launch_comment=self.Parameters.manual_launch_comment)
            issue_dict = {
                'description': description,
                'assignee': self.Parameters.main_regression_ticket_assignee,
                'queue': self.Parameters.main_ticket_queue,
                'summary': u'[{version}] Регрессия'.format(version=self.Parameters.version),
            }
            return self.startrek_client.issues.create(**issue_dict)['key']
        return None

    @property
    @sandbox.common.utils.singleton
    def assessors_runs_creator(self):
        return RunsCreator(
            tickets_queue=self.Parameters.assessors_ticket_queue,
            task=self,
            ticket_jinja_template_package='sandbox.projects.browser.autotests.regression.mbro.CbroRegressionInit',
            ticket_jinja_template_name='assessors_ticket.jinja',
            ticket_summary_template='[Асессоры] Regression Testing релиза {}'.format(self.Parameters.version),
            ticket_jinja_template_args={'version': self.Parameters.version, 'builds': self.Parameters.builds,
                                        'assessors_launch_comment': self.Parameters.assessors_launch_comment},
            ticket_properties={'components': [u'Регрессия', 'Core', 'asessors'],
                               'tags': ['CBRO_Testing']},
            default_group=self.default_group,
            testrun_properties={},
            testpalm_version_template='{}_assessors'.format(self.Parameters.version),
            run_title_template='[{} Regression testing]'.format(self.Parameters.version) +
                               '[{component}] Assessors.  Part {part}',
            unique_testpalm_version=True)

    @property
    @sandbox.common.utils.singleton
    def manual_runs_creator(self):
        return RunsCreator(
            tickets_queue=self.Parameters.manual_ticket_queue,
            task=self,
            ticket_jinja_template_package='projects.browser.autotests.regression.mbro.CbroRegressionInit',
            ticket_jinja_template_name='manual_ticket.jinja',
            ticket_summary_template='Ручное тестирование {}'.format(self.Parameters.version),
            ticket_jinja_template_args={'builds': self.Parameters.builds,
                                        'manual_launch_comment': self.Parameters.manual_launch_comment},
            ticket_properties={'components': [u'Регрессия', 'Core'],
                               'tags': ['CBRO_Testing']},
            default_group=self.default_group,
            testpalm_version_template='{}_manual'.format(self.Parameters.version),
            run_title_template='[{} Regression testing]'.format(self.Parameters.version) + '[{component}] Manual',
            testrun_properties={},
            unique_testpalm_version=False)

    def link_issues(self, regression_result):
        if regression_result.manual_tickets and regression_result.assessors_tickets:
            manual_ticket = regression_result.manual_tickets.values()[0]
            assessor_ticket = regression_result.assessors_tickets.values()[0]
            issue = self.startrek_client.issues[manual_ticket]
            issue.add_link('relates', assessor_ticket)

    def on_execute(self):
        res = super(CbroRegressionInit, self).on_execute()
        self.link_issues(res)
