# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.utils import get_task_link, singleton
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import BaseBrowserRegression, RegressionResult
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import html_link, DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT


class BaseMobileBrowserRegression(BaseBrowserRegression):

    STATE_RESOURCE_PATH = "regression_state_data"
    STATE_DATA_FILE_NAME = "regression_state.json"
    EXPECTED_CONSISTENT_BUILDS = [
        "browser_build", "tests_build"
    ]

    @property
    def regression_manager_class(self):
        raise NotImplementedError()

    @property
    def continue_regression_task(self):
        raise NotImplementedError()

    @property
    def requester_code(self):
        return self.Parameters.requester_code or None

    class Context(BaseBrowserRegression.Context):
        deadline = None
        assessors_links = {}
        task_id = None
        autotests_timeout = DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT

    class Parameters(BaseBrowserRegression.Parameters):

        test_suites = None
        regression_type = None

        with sdk2.parameters.RadioGroup("Diff type") as diff_type:
            diff_type.values.functionalities_diff = diff_type.Value("Functionalities diff")
            diff_type.values.component_diff = diff_type.Value("Component diff", default=True)
            diff_type.values.disabled = diff_type.Value("Diff disabled")

        with sdk2.parameters.Group('Assessors parameters') as assessors_params:

            hitman_process_id = sdk2.parameters.String('Hitman process', required=True)
            assessors_quota = sdk2.parameters.String('Assessors launch quota')
            requester_code = sdk2.parameters.String('Requester_code for asessors', default='', do_not_copy=True)
            start_hitman_jobs_automatically = BaseBrowserRegression.Parameters.start_hitman_jobs_automatically(
                default=False)

        with sdk2.parameters.Group('Mobile specific parameters') as mobile_params:
            builds = sdk2.parameters.List('Builds', required=True)
            version = sdk2.parameters.String('Version', required=True)
            release_configuration = sdk2.parameters.String('Release configuration', multiline=True)
            common_regression_ticket = sdk2.parameters.String('Common regression ticket')
            manual_launch_comment = sdk2.parameters.String('Manual launch comment', multiline=True)

    def on_enqueue(self):
        with self.memoize_stage.add_params_from_config:
            if self.Parameters.regression_type != 'custom':
                self.Parameters.test_suites_override = {}

    def on_execute(self):
        self.Context.task_id = self.id
        self.validate_input()
        builds = self.get_builds()
        self.wait_builds([build for build in builds.values() if build])

        with self.memoize_stage.check_booking:
            booking_check_state = self.check_booking()
            self.calculate_autotests_timeout(booking_check_state)

        manager = self.regression_manager_class(
            regression_config=self.regression_config,
            task_parameters=self.Parameters,
            task_context=self.Context,
            oauth_vault=self.oauth_vault)

        automated_info = manager.get_automated_info()
        if automated_info:
            self.set_info(u"Регрессия содержит автоматизированные тестовые сценарии {} из {}.\n"
                          u"Будут запущены автотесты, результаты будут обработаны дочерней таской.".format(
                              automated_info["summary"]["automated_count"],
                              automated_info["summary"]["total_count"]))
            autotests_task, finish_task = self.launch_autotests_and_continue_regression_tasks(automated_info)
            self.schedule_statistics_task(finish_task)
        else:
            self.set_info(u"В регрессии нет автоматизированных кейсов. Автотесты запущены не будут.\n"
                          u"Регрессия будет полностью создана в один шаг.")
            self.schedule_statistics_task(self)

        regression_summary = manager.create_regression()
        if regression_summary["info_message"]:
            self.set_info(regression_summary["info_message"],
                          do_escape=False)
        main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs = regression_summary["runs_and_tickets"]

        if assessors_runs:
            assessors_tasks = [self.create_assessors_task(assessors_runs, assessors_tickets)]
            self.enqueue_asessor_tasks(assessors_tasks)
        self.start_runs_monitoring(assessors_runs, manual_runs, manual_tickets, main_ticket)
        res = RegressionResult(main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs)
        self.create_regression_state_resource()
        return res

    def check_input(self):
        build_problems = super(BaseMobileBrowserRegression, self).check_input()
        builds = self.get_builds()
        browser_build, old_build, tests_build = builds.get("browser_build"), builds.get("old_build"), builds.get("tests_build")
        if (self.Parameters.diff_type != "disabled" and (not old_build or not browser_build)):
            build_problems.append(u'Включено использование диффа для формирования скоупа регрессии,'
                                  u' но не указана предыдущая сборка для подсчета диффа')
        if (self.Parameters.enable_autotests and not tests_build):
            build_problems.append(u'Включен флаг использования автотестов (enable_autotests),'
                                  u' для чего требуется browser_tests_build_id')
        return build_problems

    @property
    def whom_notify_about_asessors_start(self):
        return None

    @property
    def whom_notify_from_hitman(self):
        return self.author

    @property
    def test_stend(self):
        return '\n'.join(self.Parameters.builds)

    def get_assessors_deadline(self, assessors_runs):
        return self.regression_config['deadline']

    def launch_autotests_and_continue_regression_tasks(self, automated_info):

        # launch autotestst
        autotests_task = RunBrowserAutotests(
            self,
            description=u"'Regression autotests for {}".format(self.Parameters.description),
            launch_config=automated_info['launch_config'],
            browser_tests_build_id=self.Parameters.browser_tests_build_id,
            tested_application=automated_info['tested_application'],
            build_extra_args=automated_info['build_extra_args'],
            number_of_shards=automated_info['number_of_shards'],
            timeout=self.Context.autotests_timeout,
            kill_timeout=(self.Context.autotests_timeout + self.min_leeway_time_for_autotests) * 60,
            branch=automated_info['browser_branch'],
            commit=automated_info['browser_commit'],
            use_test_bitbucket=False,
            generate_allure_reports=True).enqueue()
        self.set_info(
            u'Запуск автотестов: {}'.format(html_link(get_task_link(autotests_task.id))),
            do_escape=False)
        self.Context.autotests_task_id = autotests_task.id
        # launch continue regression task
        params = {_p[0]: _p[1] for _p in self.Parameters}
        finish_task = self.continue_regression_task(
            self,
            run_regression_task=self,
            autotests_task=autotests_task,
            **params
        ).enqueue()
        self.set_info(
            u'Обработка результатов после автотестов: {}'.format(html_link(get_task_link(finish_task.id))),
            do_escape=False)

        return autotests_task, finish_task

    @singleton
    def get_builds(self):
        result = {}
        if self.Parameters.build_id:
            result["browser_build"] = self.teamcity_client.Build(id=self.Parameters.build_id)
        if self.Parameters.fake_build_id:
            result["fake_build"] = self.teamcity_client.Build(id=self.Parameters.fake_build_id)
        if self.Parameters.browser_tests_build_id:
            result["tests_build"] = self.teamcity_client.Build(id=self.Parameters.browser_tests_build_id)
        if self.Parameters.old_build_id:
            result["old_build"] = self.clients.teamcity.Build(id=self.Parameters.old_build_id)
        return result
