# -*- coding: utf-8 -*-
import sandbox
from sandbox import sdk2
from sandbox.sdk2.environments import PipEnvironment
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
import sandbox.common.types.client as ctc
from sandbox.common.types import notification as ctn
from sandbox.common.types.task import Status
from sandbox.common.types import resource as ctr
from sandbox.projects.browser.autotests.classes.autotests_result import BtrReport
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests_qa_tools.common import html_link, is_dev_sandbox, ROBOT_BRO_QA_INFRA_TOKEN_VAULT


STATE_DATA_FILE_NAME = "regression_state.json"
STATISTICS_RESOURCE_PATH = "regression_statistics_data"

TABLE_PATH = "//home/browser/infra/qa/autotests/prod"
TABLE_PATH_TEST = "//home/browser/infra/qa/autotests/test"
TABLE_ATTRIBUTES = {
    "unique_keys": True,
    "schema": {
        "$value": [
            {
                "type": "int32",
                "name": "task_id",
                "required": True
            },
            {
                "type": "datetime",
                "name": "date",
                "required": True
            },
            {
                "type": "datetime",
                "name": "tests_build_queued_date",
                "required": False
            },
            {
                "type": "string",
                "name": "case_id",
                "required": True
            },
            {
                "type": "string",
                "name": "test_name",
                "required": True
            },
            {
                "type": "string",
                "name": "binary",
                "required": True
            },
            {
                "type": "string",
                "name": "platform",
                "required": True
            },
            {
                "type": "string",
                "name": "status",
                "required": True
            },
            {
                "type": "int32",
                "name": "tests_build_id",
                "required": True
            },
            {
                "type": "int32",
                "name": "browser_build_id",
                "required": False
            },
            {
                "type": "int32",
                "name": "fake_build_id",
                "required": False
            },
            {
                "type": "string",
                "name": "tests_build_branch",
                "required": True
            },
            {
                "type": "string",
                "name": "tests_build_commit",
                "required": True
            },
            {
                "type": "string",
                "name": "runner_branch",
                "required": True
            },
            {
                "type": "string",
                "name": "runner_commit",
                "required": True
            },
            {
                "type": "string",
                "name": "launch_type",
                "required": False
            },
            {
                "type": "string",
                "name": "tested_application",
                "required": False
            },
            {
                "type": "int32",
                "name": "elapsed_time_ms",
                "required": False
            }
        ],
        '$attributes': {
            'strict': True,
        }
    }
}


class CollectAutotestsStatistics(sdk2.Task):
    class Parameters(sdk2.Parameters):
        autotests_task = sdk2.parameters.Task("autotests_task",
                                              description="Autotests task with report")
        launch_type = sdk2.parameters.String(
            'launch_type', required=True,
            choices=[(_, _) for _ in ['Nightly', 'Regression', 'Pull-request', 'Custom']],
            default='Custom')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('yandex-yt', version='0.10.3'),
            PipEnvironment('teamcity-client==3.0.0')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    def wait_autotests_task(self):
        autotests_task = self.Parameters.autotests_task
        with self.memoize_stage.run_regression_task:
            raise sdk2.WaitTask(autotests_task,
                                [Status.Group.FINISH, Status.Group.BREAK])
        if autotests_task.status != Status.SUCCESS:
            self.set_info(u'Родительская таска {} не успешна статус={}'.format(
                html_link(get_task_link(autotests_task.id)),
                autotests_task.status),
                do_escape=False)

        return autotests_task

    def get_autotests_report(self, autotests_task):
        from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import AutotestsReportResource
        autotests_result_resource = AutotestsReportResource.find(
            task=autotests_task,
            state=ctr.State.READY).first()
        if autotests_result_resource is None:
            self.set_info(u"Отчет автотестов не найден")
            raise TaskFailure()
        else:
            return BtrReport(str(sdk2.ResourceData(autotests_result_resource).path),
                             self.clients)

    def upload_to_yt(self, autotests_stat):
        import yt.yson as yson
        import yt.wrapper as yt
        table_path = '{}/{}'.format(TABLE_PATH if not is_dev_sandbox() else TABLE_PATH_TEST,
                                    self.Parameters.autotests_task.created.strftime("%Y-%m"))
        table = yt.TablePath(table_path, append=True)
        if not yt.exists(table_path, client=self.clients.yt):
            yt.create("table", table_path, attributes=yson.json_to_yson(TABLE_ATTRIBUTES),
                      client=self.clients.yt)
        yt.write_table(table, autotests_stat,
                       format=yt.JsonFormat(attributes={"encode_utf8": True}), client=self.clients.yt)
        self.set_info(u'Загружено в таблицу <a href="https://yt.yandex-team.ru/hahn/navigation?path={}">{}</a>'.format(
            table_path, table_path), do_escape=False)

    def on_save(self):
        if not any([n.recipients == ['browser-autotests-bots'] for n in self.Parameters.notifications]):
            self.Parameters.notifications = list(self.Parameters.notifications) + [
                sdk2.Notification(
                    [Status.FAILURE, Status.Group.BREAK],
                    ['browser-autotests-bots'],
                    ctn.Transport.EMAIL
                )
            ]

    def on_execute(self):
        autotests_task = self.wait_autotests_task()
        autotests_report = self.get_autotests_report(autotests_task)
        autotests_stat = autotests_report.get_yt_statistics(
            common_fields=dict(
                launch_type=self.Parameters.launch_type
            )
        )
        self.upload_to_yt(autotests_stat)
