# -*- coding: utf-8 -*-
import os
import json
import time

import sandbox
from sandbox import sdk2
from sandbox.sdk2.environments import PipEnvironment
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
import sandbox.common.types.client as ctc
from sandbox.common.types import notification as ctn
from sandbox.common.types.task import Status
from sandbox.common.types import resource as ctr
from sandbox.projects.browser.autotests_qa_tools.common import html_link, is_dev_sandbox, ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.projects.browser.autotests_qa_tools.classes.regression_report import BaseRegressionReport, get_report
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import YaClients


STATE_DATA_FILE_NAME = "regression_state.json"
STATISTICS_RESOURCE_PATH = "regression_statistics_data"

TABLE_ATTRIBUTES = {
    "unique_keys": True,
    "schema": {
        "$value": [
            {
                "type": "string",
                "name": "uuid",
                "required": True,
            },
            {
                "type": "int32",
                "name": "task_id",
                "required": True,
            },
            {
                "type": "string",
                "name": "regression_type"
            },

            {
                "type": "string",
                "name": "testcase_id"
            },

            {
                "type": "string",
                "name": "executor"
            },
            {
                "type": "string",
                "name": "testrun_environment"
            },
            {
                "type": "string",
                "name": "regression_config"
            },
            {
                "type": "boolean",
                "name": "is_automated"
            },
            {
                "type": "string",
                "name": "group_name"
            },

            {
                "type": "double",
                "name": "estimate"
            },

            {
                "type": "datetime",
                "name": "date"
            },
            {
                "type": "string",
                "name": "component"
            },

            {
                "type": "string",
                "name": "regression_main_ticket"
            },

            {
                "type": "int16",
                "name": "succeeded_in_autotests"
            },
            {
                "type": "int16",
                "name": "sent_to_work"
            },
            {
                "type": "string",
                "name": "autotests_status"
            },
        ],
        '$attributes': {
            'strict': True,
        }
    }
}


class BrowserRegressionStatistics(sdk2.Resource):
    share = True


class CollectRegressionStatistics(sdk2.Task):
    class Parameters(sdk2.Parameters):
        regression_task = sdk2.parameters.Task("regression task",
                                               description="Run or Continue BrowserRegression task")
        upload_to_yt = sdk2.parameters.Bool("upload to yt", default=True)
        summary_notification_type = sdk2.parameters.String('Summary notification type', default='')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [PipEnvironment('yandex-yt', version='0.10.3')]

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    def wait_regression_task(self):

        regression_task = self.Parameters.regression_task
        with self.memoize_stage.run_regression_task:
            raise sdk2.WaitTask(regression_task,
                                [Status.Group.FINISH, Status.Group.BREAK])
        if regression_task.status != Status.SUCCESS:

            if self.Parameters.summary_notification_type:
                report = get_report(self.Parameters.summary_notification_type, default=BaseRegressionReport,
                                    params=(None, None, self.clients))
                report.regression_failed_notification(regression_task.id,
                                                      regression_task.status,
                                                      regression_task.Parameters.description)
            self.set_info(u'Родительская таска {} не успешна статус={}'.format(
                html_link(get_task_link(regression_task.id)),
                regression_task.status),
                do_escape=False)
            raise TaskFailure()
        return regression_task

    def get_regression_state(self, task):
        from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import RegressionState
        state_resource = RegressionState.find(task=task,
                                              state=ctr.State.READY).first()
        if not state_resource:
            raise RuntimeError(u"Не найден ресурс состояния регрессии")

        file_path = os.path.join(str(sdk2.ResourceData(state_resource).path))
        with open(os.path.join(file_path, STATE_DATA_FILE_NAME), "r") as _f:
            state_data = json.load(_f)
        return state_data

    def save_result(self, jobs_stat, summary):
        data_path = str(self.path(STATISTICS_RESOURCE_PATH))
        os.mkdir(data_path)
        resource = BrowserRegressionStatistics(self, "Browser regression statistics",
                                               str(data_path))
        with open(os.path.join(data_path, 'jobs_stat.json'), 'w') as _f:
            json.dump(jobs_stat, _f)
        with open(os.path.join(data_path, 'summary.json'), 'w') as _f:
            json.dump(summary, _f)
        sdk2.ResourceData(resource).ready()

    def upload_to_yt(self, jobs_stat, regression_task):
        def _prepare_jobs_for_upload(statistic_json):
            problems = []
            all_jobs = {job['uuid']: dict(
                job.items() + [
                    ('succeeded_in_autotests', statistic_json['autotests_success'].count(job['uuid'])),
                    ("sent_to_work", statistic_json['sent_to_work'].count(job['uuid'])),
                    ("autotests_status", statistic_json['autotests_status'].get(job['uuid']))])
                for job in statistic_json['total_jobs']}
            for job in all_jobs.values():
                job['estimate'] = float(job['estimate'])
                for k, v in job.iteritems():
                    if isinstance(v, unicode):
                        job[k] = v.encode('utf-8')

            for job_uuid in statistic_json['sent_to_work'] + statistic_json['autotests_success'] + statistic_json['autotests_status'].keys():
                if job_uuid not in all_jobs:
                    problems.append('unexpected job {}'.format(job_uuid))
            return all_jobs.values(), problems

        import yt.yson as yson
        import yt.wrapper as yt
        table_path = '//home/browser/infra/qa/regressions/{}/{}'.format('test' if is_dev_sandbox() else 'prod',
                                                                        regression_task.created.strftime("%Y-%m"))
        table = yt.TablePath(table_path, append=True)
        if not yt.exists(table_path, client=self.clients.yt):
            yt.create("table", table_path, attributes=yson.json_to_yson(TABLE_ATTRIBUTES),
                      client=self.clients.yt)
        jobs, problems = _prepare_jobs_for_upload(jobs_stat)
        yt.write_table(table, jobs,
                       format=yt.JsonFormat(attributes={"encode_utf8": True}), client=self.clients.yt)
        if problems:
            self.set_info(u'Обнаружена неконсистентность в данных:</br>{}'.format(u'</br>'.join(problems)),
                          do_escape=False)
            raise TaskFailure(u'Обнаружена неконсистентность в данных')

    def on_save(self):
        if not any([n.recipients == ['browser-autotests-bots'] for n in self.Parameters.notifications]):
            self.Parameters.notifications = list(self.Parameters.notifications) + [
                sdk2.Notification(
                    [Status.FAILURE, Status.Group.BREAK],
                    ['browser-autotests-bots'],
                    ctn.Transport.EMAIL
                )
            ]

    def on_execute(self):
        regression_task = self.wait_regression_task()
        regression_state = self.get_regression_state(regression_task)
        _task_info = {
            'task_id': regression_task.id,
            'date': int(time.mktime(regression_task.updated.timetuple()))
        }
        report = get_report(self.Parameters.summary_notification_type,
                            (_task_info, regression_state, self.clients),
                            default=BaseRegressionReport)

        problems = []
        summary = report.summary_report
        summary_massage = report.summary_text
        self.set_info(summary_massage)
        try:
            report.regression_summary_notification()
        except Exception as e:
            problems.append(e)

        try:
            # отчет по всей статистике. Отправить в хранилище
            jobs_stat = report.jobs_stat
            self.save_result(jobs_stat, summary)

            if self.Parameters.upload_to_yt:
                self.upload_to_yt(jobs_stat, regression_task)
        except Exception as e:
            problems.append(e)

        if problems:
            raise problems[0]
