# -*- coding: utf-8 -*-
import sandbox
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.autotests.classes.testing_builds import ReleaseKits, Builder
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT, build_number_tuple
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds

INFO_NAMES = {
    "browser": u"Браузер (Browser build id)",
    "fake_browser": u"Фейк браузер (Fake browser build id)",
    "autotests": u"Сборка для автотестов (Tests build id)"
}


class PrepareBrowserRegressionBuilds(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
        ]

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    class Context(sdk2.Context):
        teamcity_builds = None

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group(u'Основное') as main_params:
            browser_branch = sdk2.parameters.String(
                'Browser branch',
                description=u'Ветка браузера для сборки',
                required=True)
            browser_commit = sdk2.parameters.String(
                'Browser commit',
                description=u'Коммит (пустой = latest)',
                default='')
            build_type = sdk2.parameters.String(
                'Build type',
                required=True,
                choices=[(_, _) for _ in [kit.name for kit in ReleaseKits]],
                description=u'Тип главной сборки',
                ui=sdk2.parameters.String.UI('select')
            )
            add_autotests_build = sdk2.parameters.Bool(
                'Add autotests build',
                default=True,
                description=u'Собрать билд с автотестами'
            )
        with sdk2.parameters.Group(u'Дополнительно') as adv_params:
            teamcity_build_parameters = sdk2.parameters.JSON(
                'Teamcity build аdvanced parameters',
                description=u'Дополнительные параметры для целевой сборки (опционально)'
            )
            teamcity_build_tags = sdk2.parameters.List(
                'Teamcity build tags',
                description=u'Теги для целевой сборки (опционально)'
            )

    def on_break(self, prev_status, status):
        if status == Status.STOPPED:
            self.stop_teamcity_builds()

    def stop_teamcity_builds(self):
        if self.Context.teamcity_builds:
            BrowserStopTeamcityBuilds(
                None,
                description='Stop builds for task {}'.format(self.id),
                teamcity_build_ids=self.Context.teamcity_builds.values(),
                cancel_comment='Task sandbox-{} was stopped'.format(self.id),
                owner=self.Parameters.owner,
                oauth_vault=ROBOT_BRO_QA_INFRA_TOKEN_VAULT,
            ).enqueue()

    def wait_builds(self):
        if any(b.state != 'finished' for b in self.triggering_builds.values()):
            raise sdk2.WaitTime(60 * 10)

    @property
    def triggering_builds(self):
        if self.Context.teamcity_builds is None:
            return {}
        return {_k: self.clients.teamcity.Build(id=_id) for _k, _id in self.Context.teamcity_builds.iteritems()}

    def on_execute(self):

        with self.memoize_stage.run_builds:
            builder = Builder(self.clients)
            builds = builder.build_kit(
                kit_name=self.Parameters.build_type,
                branch=self.Parameters.browser_branch,
                commit=self.Parameters.browser_commit,
                params=self.Parameters.teamcity_build_parameters,
                tags=self.Parameters.teamcity_build_tags,
                build_autotests=self.Parameters.add_autotests_build
            )

            self.Context.teamcity_builds = {key: build.id for key, build in builds.iteritems()}
            if not self.Context.teamcity_builds:
                self.set_info(u"Задание на сборку пусто. Ничего не собрано.")
                raise TaskFailure()
            else:
                if all(_b in self.Context.teamcity_builds for _b in ["browser", "fake_browser"]):
                    message = u"(!) Запланированы сборки browser и fake_browser. " \
                              u"Версия итоговой сборки доступна только по её завершению, фейк и браузер могут поменяться местами." \
                              u"Для получения окончательного корректного результата дождитесь завершения всех сборок"
                    message += u"<br>Запланированы сборки:"
                    for build in self.triggering_builds.values():
                        message += u' <a href="{}">#{}</a>'.format(build.web_url, build.id)
                    self.set_info(message, do_escape=False)
                else:
                    self.info_result(u"Запланированы сборки:")

        self.wait_builds()
        problem_builds = {key: build for key, build in self.triggering_builds.iteritems() if build.status != 'SUCCESS'}
        if problem_builds:
            message = u"Упали сборки:"
            for key, build in problem_builds.iteritems():
                message += u'<br> * {}: <a href="{}">{}</a>'.format(key, build.web_url, build.id)
            self.set_info(message, do_escape=False)
            raise TaskFailure()
        else:
            if all(_b in self.Context.teamcity_builds for _b in ["browser", "fake_browser"]):
                browser, fake_browser = sorted([self.triggering_builds["browser"], self.triggering_builds["fake_browser"]], key=build_number_tuple)
                self.Context.teamcity_builds["browser"] = browser.id
                self.Context.teamcity_builds["fake_browser"] = fake_browser.id
                self.info_result(u"Все сборки завершены успешно:")
            else:
                self.set_info(u"Все сборки завершены успешно")

    def info_result(self, title=None):
        message = u"<br>{}".format(title) if title else u""
        for key, build in self.triggering_builds.iteritems():
            message += u'<br> * {}: <a href="{}">{}</a>'.format(INFO_NAMES.get(key, key),
                                                                build.web_url,
                                                                build.id)
        self.set_info(message, do_escape=False)
