# -*- coding: utf-8 -*-
import datetime
import json
import os
import time
import zipfile

import sandbox
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.browser.autotests.classes.btr_log_parser import BtrReport
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.classes.btr_log_parser import RESULT_FILENAME, LAUNCH_INFO_FILENAME
from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT, DEFAULT_KILL_TIMEOUT,
    DEFAULT_MAX_FLAKY_RETRIES, INPUT_PATH, RESULT_DIR)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import (
    AutotestsDump, AutotestsAllureReports, AutotestsReportResource, AutotestsScreenshotsDiff,
    AutotestsTestResultsArchives)
from sandbox.projects.browser.common import LXC_RESOURCE_ID
from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript


def _is_nonempty_directory(path):
    if not os.path.isdir(path):
        return False
    return bool(os.listdir(path))


def print_allure_reports_info(task, allure_reports_dict):
    if allure_reports_dict:
        message = u'Доступны аllure-отчеты (!не все изоляты имеют allure, полный отчет доступен по ссылке выше): <ul>'
        for platform, reports in allure_reports_dict.iteritems():
            message += u'<li> {}: <ul>'.format(platform)
            for report in reports:
                message += u'<li> <a href="{}">{}</a></li>'.format(report[1], report[0])
            message += u'</ul></li>'
        message += u'</ul>'
    else:
        message = u'Allure-отчетов нет'
    task.set_info(message, do_escape=False)


class RunBrowserAutotests(RunBrowserScript):
    class Requirements(RunBrowserScript.Requirements):
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(RunBrowserScript.Context):
        script_config = None
        report_url = None
        allure_reports = {}

    class Parameters(RunBrowserScript.Parameters):
        kill_timeout = DEFAULT_KILL_TIMEOUT
        launch_config = sdk2.parameters.JSON(
            'Launch config',
            description='Launch config for yandex.autotests.binary.run_tests script'
        )
        browser_tests_build_id = sdk2.parameters.Integer(
            'Tests build id',
            description='Id of teamcity build with browser tests Browser_Tests_BuildTests'
        )
        build_extra_args = sdk2.parameters.JSON(
            'Build extra args',
            default={}
        )
        tested_application = sdk2.parameters.String(
            'Tested application: browser or searchapp', default='browser')
        timeout = sdk2.parameters.Integer(
            'Timeout',
            description='Autotests timeout in minutes',
            default=DEFAULT_REGRESSION_AUTOTESTS_TIMEOUT
        )
        bulk_retries_number = sdk2.parameters.Integer(
            'Number of retries allowed for each isolate',
            default=2
        )
        max_flaky_retries = sdk2.parameters.Integer(
            'Number of retries flaky tests',
            default=DEFAULT_MAX_FLAKY_RETRIES
        )
        number_of_shards = sdk2.parameters.Integer(
            'Number of shards',
            default=1
        )
        ignore_blacklists = sdk2.parameters.Bool(
            'Ignore blacklists', default=False)
        generate_allure_reports = sdk2.parameters.Bool(
            'Generate allure reports', default=False)
        yav_token_vault = sdk2.parameters.String(
            'Vault item with yav token', default='robot-bro-qa-infra-token')
        pixel_diff_pr_source_branch = sdk2.parameters.String(
            'ID of PR source branch to generate pixel diff',
            required=False
        )
        _container = sdk2.parameters.Container('LXC container', default=LXC_RESOURCE_ID, required=False)

    @property
    @sandbox.common.utils.singleton
    def bitbucket_browser_sparse_checkout_paths(self):
        return [
            '.gclient_default', 'requirements.txt', 'src/.DEPS.snapshot', 'src/build', 'src/testing',
            'src/tools/pixel_tests'
        ]

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    def script_extra_env(self):
        env = super(RunBrowserAutotests, self).script_extra_env()
        env.update({
            'YAV_TOKEN': sdk2.Vault.data(self.Parameters.yav_token_vault),
        })
        return env

    @property
    def bb(self):
        return self.clients.bitbucket

    def prepare_input_json(self):
        result = {}
        for platform, cases in self.Parameters.launch_config.iteritems():
            _data = {
                "builds": [
                    {
                        "build_id": self.Parameters.browser_tests_build_id,
                        "testcases": cases
                    }
                ]
            }
            if self.Parameters.build_extra_args:
                _data["builds"][0].update(self.Parameters.build_extra_args)

            result[platform] = _data
        self.Context.script_config = result
        return result

    @property
    @sandbox.common.utils.singleton
    def input_path(self):
        input_path = str(self.path(INPUT_PATH))
        with open(input_path, 'w') as _f:
            json.dump(self.prepare_input_json(), _f)
        return input_path

    @property
    @sandbox.common.utils.singleton
    def out_dir(self):
        data_path = str(self.path(RESULT_DIR))
        os.mkdir(data_path)
        return data_path

    @property
    @sandbox.common.utils.singleton
    def pixel_diff_dir(self):
        return str(self.path('pixel_diff'))

    @property
    @sandbox.common.utils.singleton
    def artifacts_dir(self):
        return str(self.path('test_results'))

    @property
    @sandbox.common.utils.singleton
    def allure_reports_dir(self):
        return str(self.path('allure_reports'))

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', 'yandex.autotests.binary.run_tests', self.input_path]
        cmd += ['--tests-results', str(os.path.join(self.out_dir, RESULT_FILENAME))]
        cmd += ['--bulk-retries-number', str(self.Parameters.bulk_retries_number)]
        cmd += ['--max_flaky_retries', str(self.Parameters.max_flaky_retries)]
        cmd += ['--number-of-shards', str(self.Parameters.number_of_shards)]
        if self.Parameters.ignore_blacklists:
            cmd += ['--ignore-blacklists']
        timeout = int(time.mktime(datetime.datetime.now().timetuple())) + self.Parameters.timeout * 60
        cmd += ['--timeout', str(timeout)]
        cmd += ['--pixel-diff-path', self.pixel_diff_dir]
        if self.Parameters.pixel_diff_pr_source_branch:
            cmd += ['--pixel-diff-pr-source-branch', str(self.Parameters.pixel_diff_pr_source_branch)]
        cmd += ['--test-artifacts-dir', self.artifacts_dir]
        cmd += ['--allure-reports-dir', self.allure_reports_dir if self.Parameters.generate_allure_reports else '']
        cmd += ['--tested-application', self.Parameters.tested_application]
        return cmd

    def on_execute(self):
        super(RunBrowserAutotests, self).on_execute()
        self.save_report_resource()
        self.set_info(u'Отчет автотестов: <a href="{}">{}</a>'.format(
            self.Context.report_url, self.Context.report_url), do_escape=False)
        if _is_nonempty_directory(self.pixel_diff_dir):
            self.publish_pixel_diff()
        if _is_nonempty_directory(self.artifacts_dir):
            self.publish_tests_results_archives()
        if self.Parameters.generate_allure_reports and _is_nonempty_directory(self.allure_reports_dir):
            self.publish_allure_reports()

    def save_report_resource(self):
        # save launch info
        try:
            params = {_p[0]: _p[1] for _p in self.Parameters if _p[0] != "_container"}
            launch_info = {
                "date": int(time.mktime(datetime.datetime.now().timetuple())),
                "parameters": params,
                "task_id": self.id
            }
            with open(os.path.join(self.out_dir, LAUNCH_INFO_FILENAME), "w") as _f:
                json.dump(launch_info, _f, indent=4)
        except Exception as _e:
            self.set_info(u"Не удалось сохранить launch_info в отчет: {}".format(_e.message))

        resource = AutotestsReportResource(self, "Autotests result", self.out_dir)
        sdk2.ResourceData(resource).ready()
        self.Context.report_url = resource.http_proxy
        try:
            report = BtrReport(self.out_dir, self.clients)
            self.set_info(report.html_summary, do_escape=False)

            # tests_dump resource
            dump_path = str(self.path('tests_dump'))
            os.makedirs(dump_path)
            report.dump_tests(dump_path)
            dump_resource = AutotestsDump(self, 'Autotests dump', dump_path)
            sdk2.ResourceData(dump_resource).ready()
        except Exception as e:
            self.set_info(u"Не удалось получить резюме и дамп результатов: {}".format(e.message))

    def publish_pixel_diff(self):
        resource = AutotestsScreenshotsDiff(self, "Schreenshots diff", self.pixel_diff_dir)
        sdk2.ResourceData(resource).ready()

    def publish_tests_results_archives(self):
        resource = AutotestsTestResultsArchives(self, "Test results archives", self.artifacts_dir)
        sdk2.ResourceData(resource).ready()

    def publish_allure_reports(self):
        resource_result_path = str(self.path('unpacked_allure_reports'))
        source = self.allure_reports_dir
        allure_reports_names = {}
        for platform_name in [_dir for _dir in os.listdir(source) if os.path.isdir(os.path.join(source, _dir))]:

            for archive_name in [_file for _file in os.listdir(os.path.join(source, platform_name)) if
                                 _file.endswith('.zip')]:
                unpack_path = os.path.join(resource_result_path, platform_name, archive_name.replace('.zip', ''))
                if not os.path.exists(unpack_path):
                    os.makedirs(unpack_path)
                with zipfile.ZipFile(os.path.join(source, platform_name, archive_name), 'r') as archive:
                    archive.extractall(unpack_path)
                allure_reports_names.setdefault(platform_name, []).append(os.path.basename(unpack_path))

        if allure_reports_names:
            resource = AutotestsAllureReports(self, "Аllure reports", resource_result_path)
            sdk2.ResourceData(resource).ready()
            for platform, reports in allure_reports_names.iteritems():
                for report in reports:
                    url = '{}/{}/{}/index.html'.format(resource.http_proxy, platform, report)
                    self.Context.allure_reports.setdefault(platform, []).append([report, url])

        print_allure_reports_info(self, self.Context.allure_reports)
