# -*- coding: utf-8 -*-
import datetime
import json
import os
import time

import sandbox
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.allure_parser import AllureReport
from sandbox.projects.browser.autotests.BrowserAutotestRun import BrowserAutotestRun
from sandbox.projects.browser.autotests.classes.autotests_bundle import PYTHON_TESTS_PLATFORM_PREFIX, get_latest_autotests_bundle_by_browser_build
from sandbox.projects.browser.autotests.classes.autotests_result import allure_2_btr, merge_btr_reports
from sandbox.projects.browser.autotests.classes.btr_log_parser import BtrReport
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.classes.btr_log_parser import RESULT_FILENAME, LAUNCH_INFO_FILENAME
from sandbox.projects.browser.autotests.regression_tasks.RunBrowserAutotests import RunBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import (
    ROBOT_BRO_QA_INFRA_TOKEN_VAULT, html_link, extract_zip_resource, INPUT_PATH, RESULT_DIR)
from sandbox.projects.browser.autotests_qa_tools.sb_common.resources import (
    AutotestsDump, AutotestsAllureData, AutotestsReportResource)
from sandbox.sandboxsdk.environments import PipEnvironment


class RunDesktopBrowserAutotests(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 150
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
            PipEnvironment('jsonschema==2.5.1'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        script_config = None
        report_url = None
        binary_autotests_task = None
        python_autotests_task = None
        report_url = None
        problems = []

    class Parameters(RunBrowserAutotests.Parameters):
        pass

    @property
    @sandbox.common.utils.singleton
    def bitbucket_browser_sparse_checkout_paths(self):
        return ['.gclient_default', 'requirements.txt', 'src/.DEPS.snapshot', 'src/build', 'src/testing']

    @property
    @sandbox.common.utils.singleton
    def oauth_vault(self):
        return sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT)

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(self.oauth_vault)

    @property
    def bb(self):
        return self.clients.bitbucket

    def prepare_input_json(self):
        result = {}
        for platform, cases in self.Parameters.launch_config.iteritems():
            _data = {
                "builds": [
                    {
                        "build_id": self.Parameters.browser_tests_build_id,
                        "testcases": cases
                    }
                ]
            }
            if self.Parameters.build_extra_args:
                _data["builds"][0].update(self.Parameters.build_extra_args)

            result[platform] = _data
        self.Context.script_config = result
        return result

    @property
    @sandbox.common.utils.singleton
    def separated_launch_confid(self):
        result = {
            "binary": {},
            "python": {}
        }
        for _platform in self.Parameters.launch_config:
            if _platform.startswith(PYTHON_TESTS_PLATFORM_PREFIX):
                result["python"][_platform.replace(PYTHON_TESTS_PLATFORM_PREFIX, "")] = self.Parameters.launch_config[_platform]
            else:
                result["binary"][_platform] = self.Parameters.launch_config[_platform]
        return result

    @property
    @sandbox.common.utils.singleton
    def input_path(self):
        input_path = str(self.path(INPUT_PATH))
        with open(input_path, 'w') as _f:
            json.dump(self.prepare_input_json(), _f)
        return input_path

    @property
    @sandbox.common.utils.singleton
    def out_dir(self):
        data_path = str(self.path(RESULT_DIR))
        os.mkdir(data_path)
        return data_path

    def launch_binary_tests(self, launch_conf):

        params = {_p[0]: _p[1] for _p in self.Parameters}
        params['launch_config'] = launch_conf
        params['timeout'] = params['timeout'] - 10
        # до изолирования питон-тестов
        params['build_extra_args'] = {}
        params['kill_timeout'] = (self.Parameters.timeout + 30) * 60

        binary_autotests_task = RunBrowserAutotests(
            self,
            tags=self.Parameters.tags,
            **params
        ).enqueue()
        self.set_info(
            u'Запуск бинарных автотестов: {}'.format(html_link(get_task_link(binary_autotests_task.id))),
            do_escape=False)
        self.Context.binary_autotests_task = binary_autotests_task.id

    def launch_python_tests(self, launch_conf):

        branded_build = self.Parameters.build_extra_args.get('python_tests-extra-args', {}).get('build_id')
        fake_build = self.Parameters.build_extra_args.get('python_tests-extra-args', {}).get('fake_build_id')
        if not branded_build:
            self.Context.problems.append(u"Требуется build_id")
            return None
        bundle = get_latest_autotests_bundle_by_browser_build(
            branded_browser_build_id=branded_build,
            browser_tests_build_id=None,
            ya_clients=self.clients
        )
        if not bundle:
            self.Context.problems.append(u"для branded_build {} не найден autotests_bundle. Питон-тесты не запущены")
            return None
        all_cases = []
        for _platform, cases in launch_conf.iteritems():
            for _case in cases:
                all_cases.append((_case, _platform))
        launch_config = bundle.get_launch_config_by_cases(all_cases)
        if not launch_config:
            self.Context.problems.append(u"Не сформирован конфиг питон-автотестов. Питон-тесты не запущены")
            return None
        python_autotests_task = BrowserAutotestRun(
            self,
            description=self.Parameters.description,
            tags=self.Parameters.tags,
            browser_version='custom',
            config_file='custom',
            config=launch_config,
            sleep_time=10,
            wait_builds_time=self.Parameters.timeout - 10,
            framework_commit=bundle.bundle_commit,
            build_id=branded_build,
            fake_build_id=fake_build,
            browser_tests_build_id=None
        ).enqueue()
        self.set_info(
            u'Запуск питон-автотестов: {}'.format(html_link(get_task_link(python_autotests_task.id))),
            do_escape=False)
        self.Context.python_autotests_task = python_autotests_task.id

    def wait_autotests_tasks(self):

        binary_autotests_task = sdk2.Task.find(id=self.Context.binary_autotests_task, children=True).limit(1).first() if self.Context.binary_autotests_task else None
        python_autotests_task = sdk2.Task.find(id=self.Context.python_autotests_task, children=True).limit(1).first() if self.Context.python_autotests_task else None

        if binary_autotests_task and binary_autotests_task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
            raise sdk2.WaitTime(60 * 10)
        if python_autotests_task and python_autotests_task.status not in list(Status.Group.FINISH + Status.Group.BREAK):
            raise sdk2.WaitTime(60 * 10)

        return binary_autotests_task, python_autotests_task

    def get_autotests_results_path(self, autotests_task, result_resource_type):

        if autotests_task.status != Status.SUCCESS:
            self.Context.problems.append("autotests_task={} не успешна".format(autotests_task.id))
        autotests_result_resource = result_resource_type.find(
            task=autotests_task,
            state=ctr.State.READY).first()
        if autotests_result_resource is None:
            self.Context.problems.append(u"Отчет автотестов autotests_task={} не найден".format(autotests_task.id))
            return None
        else:
            return str(sdk2.ResourceData(autotests_result_resource).path)

    def get_results(self, binary_autotests_task, python_autotests_task):

        result = {}
        if binary_autotests_task:
            report_path = self.get_autotests_results_path(binary_autotests_task, AutotestsReportResource)
            if report_path:
                with open(os.path.join(report_path, RESULT_FILENAME), "r") as _f:
                    result = json.load(_f)

        if python_autotests_task:
            report_path = self.get_autotests_results_path(python_autotests_task, AutotestsAllureData)
            if report_path:
                allure = AllureReport(extract_zip_resource(self, report_path), self.clients)
                fake_btr = allure_2_btr(allure, self.Parameters.browser_tests_build_id)
                result = merge_btr_reports(result, fake_btr)

        with open(os.path.join(self.out_dir, RESULT_FILENAME), "w") as _res:
            json.dump(result, _res, indent=4)

    def on_execute(self):

        with self.memoize_stage.run_test_tasks:
            if self.separated_launch_confid["binary"]:
                self.launch_binary_tests(self.separated_launch_confid["binary"])
            if self.separated_launch_confid["python"]:
                self.launch_python_tests(self.separated_launch_confid["python"])

        binary_autotests_task, python_autotests_task = self.wait_autotests_tasks()
        if binary_autotests_task is None and python_autotests_task is None:
            self.Context.problems.append(u"Не запущено никаких тестов")

        self.save_report_resource(binary_autotests_task, python_autotests_task)
        self.set_info(u'Отчет автотестов: <a href="{}">{}</a>'.format(self.Context.report_url, self.Context.report_url), do_escape=False)

        if self.Context.problems:
            message = u""
            for _problem in self.Context.problems:
                message += u"<li>{}</li>".format(_problem)
            message = u"Имеются проблемы <ul>{}</ul>".format(message)
            self.set_info(message, do_escape=False)
            raise TaskFailure(u"Что то пошло не так, см. сообщение о проблемах")

    def save_report_resource(self, binary_autotests_task, python_autotests_task):
        result = {}
        if binary_autotests_task:
            report_path = self.get_autotests_results_path(binary_autotests_task, AutotestsReportResource)
            if report_path:
                with open(os.path.join(report_path, RESULT_FILENAME), "r") as _f:
                    result = json.load(_f)

        if python_autotests_task:
            report_path = self.get_autotests_results_path(python_autotests_task, AutotestsAllureData)
            if report_path:
                allure_path = extract_zip_resource(self, report_path)
                fake_btr = allure_2_btr(AllureReport(allure_path, self.clients),
                                        self.Parameters.browser_tests_build_id)
                result = merge_btr_reports(result, fake_btr)

        with open(os.path.join(self.out_dir, RESULT_FILENAME), "w") as _res:
            json.dump(result, _res, indent=4)

        # save launch info
        try:
            params = {_p[0]: _p[1] for _p in self.Parameters if _p[0] != "_container"}
            launch_info = {
                "date": int(time.mktime(datetime.datetime.now().timetuple())),
                "parameters": params,
                "task_id": self.id
            }
            with open(os.path.join(self.out_dir, LAUNCH_INFO_FILENAME), "w") as _f:
                json.dump(launch_info, _f, indent=4)
        except Exception as _e:
            self.set_info(u"Не удалось сохранить launch_info в отчет: {}".format(_e.message))

        resource = AutotestsReportResource(self, "Autotests result", self.out_dir)
        sdk2.ResourceData(resource).ready()
        self.Context.report_url = resource.http_proxy

        try:
            report = BtrReport(self.out_dir, self.clients)
            self.set_info(report.html_summary, do_escape=False)
            # tests_dump resource
            dump_path = str(self.path('tests_dump'))
            os.makedirs(dump_path)
            report.dump_tests(dump_path)
            dump_resource = AutotestsDump(self, 'Autotests dump', dump_path)
            sdk2.ResourceData(dump_resource).ready()
        except Exception as e:
            self.set_info(u"Не удалось получить резюме и дамп результатов: {}".format(e.message))
