# -*- coding: utf-8 -*-
import datetime

from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.BrowserCalculateFunctionalitiesProfit import BrowserCalculateFunctionalitiesProfit
from sandbox.projects.browser.autotests.regression_tasks.BaseDesktopBrowserRegression import BaseDesktopBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import RegressionResult
from sandbox.projects.browser.autotests.regression_tasks.ContinueDesktopBrowserRegression import ContinueDesktopBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.RunDesktopBrowserAutotests import RunDesktopBrowserAutotests
from sandbox.projects.browser.autotests_qa_tools.common import html_link


class RunDesktopBrowserRegression(BaseDesktopBrowserRegression):

    MIN_LEEWAY_TIME_FOR_AUTOTESTS = 30
    continue_regression_task = ContinueDesktopBrowserRegression

    def on_execute(self):

        super(RunDesktopBrowserRegression, self).on_execute()
        if not self.Context.deadline:
            self.Context.deadline = (datetime.datetime.now() + datetime.timedelta(
                hours=self.Parameters.regression_deadline)).date().isoformat()
        builds = self.get_builds()
        with self.memoize_stage.calculate_profit:
            if self.Parameters.diff_type != "component_diff":
                BrowserCalculateFunctionalitiesProfit(
                    self,
                    regression_task=self.id,
                    description=self.Parameters.description,
                    mode='GET_INFO_FROM_TASK',
                ).enqueue()
            else:
                self.set_info(
                    u"Регрессия запущена с компонентным диффом, таска BROWSER_CALCULATE_FUNCTIONALITIES_PROFIT запущена не будет")

        self.wait_builds([build for build in builds.values() if build])
        with self.memoize_stage.check_booking:
            booking_check_state = self.check_booking()
            self.calculate_autotests_timeout(booking_check_state)

        assessors_links = self.upload_files_to_s3(
            builds.get("browser_build"),
            builds.get("fake_build", None),
            distributions=[{"distribution_type": self.Parameters.distribution_type,
                            "distribution_name": self.Parameters.distribution_name,
                            "distribution_channel": self.Parameters.distribution_channel}]
        )
        self.Context.assessors_links = assessors_links

        manager = self.regression_manager_class(
            regression_config=self.regression_config,
            task_parameters=self.Parameters,
            task_context=self.Context,
            oauth_vault=self.oauth_vault)

        if self.Parameters.diff_type == "functionalities_diff":
            diff_functionalities = manager.diff_functionalities
            self.set_info(u'Дифф затрагивает следующие функциональности: {}'.format(
                ', '.join(sorted(diff_functionalities))))
        elif self.Parameters.diff_type == "component_diff":
            diff_components = manager.diff_components
            self.set_info(u'Дифф затрагивает следующие компоненты: {}'.format(
                ', '.join(sorted(diff_components))))

        automated_info = manager.get_automated_info()
        if automated_info:
            self.set_info(u"Регрессия содержит автоматизированные тестовые сценарии {} из {}.\n"
                          u"Будут запущены автотесты, результаты будут обработаны дочерней таской.".format(
                              automated_info["summary"]["automated_count"],
                              automated_info["summary"]["total_count"]))
            autotests_task, finish_task = self.launch_autotests_and_continue_regression_tasks(automated_info)
            self.schedule_statistics_task(finish_task)
        else:
            self.set_info(u"В регрессии нет автоматизированных кейсов. Автотесты запущены не будут.\n"
                          u"Регрессия будет полностью создана в один шаг.")
            self.schedule_statistics_task(self)

        regression_summary = manager.create_regression()

        if regression_summary["info_message"]:
            self.set_info(regression_summary["info_message"],
                          do_escape=False)
        main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs = regression_summary["runs_and_tickets"]

        if assessors_runs:
            assessors_tasks = [self.create_assessors_task(assessors_runs, assessors_tickets)]
            self.enqueue_asessor_tasks(assessors_tasks)
        self.start_runs_monitoring(assessors_runs, manual_runs, manual_tickets, main_ticket)

        res = RegressionResult(main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs)
        self.update_tickets(res)

        manager.publish_diff_tickets()
        self.create_regression_state_resource()

    def launch_autotests_and_continue_regression_tasks(self, automated_info):

        # launch autotestst
        autotests_task = RunDesktopBrowserAutotests(
            self,
            description=u"'Regression autotests for {}".format(self.Parameters.description),
            launch_config=automated_info['launch_config'],
            browser_tests_build_id=self.Parameters.browser_tests_build_id,
            tested_application=automated_info['tested_application'],
            build_extra_args=automated_info['build_extra_args'],
            number_of_shards=automated_info['number_of_shards'],
            timeout=self.Context.autotests_timeout,
            kill_timeout=(self.Context.autotests_timeout + self.min_leeway_time_for_autotests) * 60,
            branch=automated_info['browser_branch'],
            commit=automated_info['browser_commit'],
            use_test_bitbucket=False).enqueue()
        self.set_info(
            u'Запуск автотестов: {}'.format(html_link(get_task_link(autotests_task.id))),
            do_escape=False)
        self.Context.autotests_task_id = autotests_task.id
        # launch continue regression task
        params = {_p[0]: _p[1] for _p in self.Parameters}
        finish_task = self.continue_regression_task(
            self,
            run_regression_task=self,
            autotests_task=autotests_task,
            **params
        ).enqueue()
        self.set_info(
            u'Обработка результатов после автотестов: {}'.format(html_link(get_task_link(finish_task.id))),
            do_escape=False)

        return autotests_task, finish_task
