# -*- coding: utf-8 -*-
import datetime

import sandbox
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.autotests.regression.assessors.BrowserUploadBrandedBuildToS3 import (
    BrowserUploadBrandedBuildToS3)
from sandbox.projects.browser.autotests.classes.regression_manager.distribution_desktop import DistributionDBroRegressionManager
from sandbox.projects.browser.autotests.classes.frozendict import FrozenDict
from sandbox.projects.browser.autotests.regression_tasks.configs import RegressionConfigs
from sandbox.projects.browser.autotests.regression_tasks.BaseDesktopBrowserRegression import BaseDesktopBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import BaseBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import RegressionResult
from sandbox.projects.browser.autotests_qa_tools.common import BRANDED_BUILDS_PROJECT, check_browser_build, TEAMCITY_URL


CONFIGS_MODULE = RegressionConfigs.distribution_dbro.value
ASSESSORS_ARTIFACTS = {
    'Win': {
        'build': ('FULL_{version}_{distribution_name}.exe', '{distribution_type}/{distribution_name}/Yandex.exe')
    },
    'Mac': {
        'build': ('FULL_{version}_{distribution_name}.dmg', '{distribution_type}/{distribution_name}/Yandex.dmg')
    },
    'Linux': {
        'build': ('FULL_{version}_{distribution_name}.deb',
                  '{distribution_type}/{distribution_name}/yandex-browser-beta_{version}-1_amd64.deb')
    }
}


class RunDistributionDesktopBrowserRegression(BaseDesktopBrowserRegression):

    STATE_RESOURCE_PATH = "regression_state_data"
    STATE_DATA_FILE_NAME = "regression_state.json"

    regression_manager_class = DistributionDBroRegressionManager
    configs_module = CONFIGS_MODULE

    def upload_files_to_s3(self, *args, **kwargs):
        raise NotImplementedError()

    @property
    def test_stend(self):
        raise NotImplementedError()

    def check_input(self):
        builds = self.get_builds()
        build_problems = super(BaseDesktopBrowserRegression, self).check_input()
        if not (self.Parameters.browser_build_alice or self.Parameters.browser_build_noalice):
            build_problems.append(u"Необходим минимум один билд browser_build_alice или browser_build_noalice")
        if self.Parameters.browser_build_alice == self.Parameters.browser_build_noalice:
            build_problems.append(u"Одинаковые browser_build_alice и browser_build_noalice не допускаются")
        if self.Parameters.browser_build_alice:
            build_problems.extend(check_browser_build(builds.get("browser_build_alice")))
        if self.Parameters.browser_build_noalice:
            build_problems.extend(check_browser_build(builds.get("browser_build_noalice")))
        return build_problems

    class Context(BaseDesktopBrowserRegression.Context):
        deadline = None
        assessors_links = {}

    class Parameters(BaseBrowserRegression.Parameters):
        # these parameters are overridden below, so that they would be hidden in ui,
        # if they should be loaded from config file
        test_suites = None
        deadline = None
        build_id = None
        old_build_id = None
        fake_build_id = None
        browser_tests_build_id = None
        enable_autotests = None
        diff_type = None

        regression_type = sdk2.parameters.String(
            'Regression type', required=True, choices=[(_, _) for _ in CONFIGS_MODULE.TEST_SUITE_CONFIGS + ['custom']],
            ui=sdk2.parameters.String.UI('select')
        )
        with regression_type.value['custom']:
            test_suites_override = BaseBrowserRegression.Parameters.test_suites()

        hitman_process_id = sdk2.parameters.String('Hitman process', default='testing_browser_automatic',
                                                   required=True)
        assessors_quota = sdk2.parameters.String('Assessors launch quota', default='brocase', required=True)
        start_hitman_jobs_automatically = BaseBrowserRegression.Parameters.start_hitman_jobs_automatically(
            default=False)
        with sdk2.parameters.Group('Distribution DBRO specific parameters') as distribution_dbro_params:
            support_chat = sdk2.parameters.String('Assessors support chat', required=True)
            browser_build_alice = sdk2.parameters.Integer(
                'Browser with alice build id',
                description='Id of fake browser with alice teamcity build '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
            )
            browser_build_noalice = sdk2.parameters.Integer(
                'Browser without alice build id',
                description='Id of browser without alice teamcity build '
                            '(<a href="{base}/project.html?projectId={project}">'
                            '{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT)
            )
            configuration_ticket = sdk2.parameters.String('Configuration ticket')
            release_ticket = sdk2.parameters.String('Release ticket')
            regression_deadline = sdk2.parameters.Integer('Regression deadline', required=True,
                                                          description='Regression duration in hours')
            manual_launch_comment = sdk2.parameters.String('Manual launch comment', multiline=True,
                                                           description='Special parameters for manual runs')

    def on_enqueue(self):
        with self.memoize_stage.add_params_from_config:
            if self.Parameters.regression_type != 'custom':
                self.Parameters.test_suites_override = {}

    @sandbox.common.utils.singleton
    def get_builds(self):
        res = {}
        if self.Parameters.browser_build_alice:
            res["browser_build_alice"] = self.clients.teamcity.Build(id=self.Parameters.browser_build_alice)
        if self.Parameters.browser_build_noalice:
            res["browser_build_noalice"] = self.clients.teamcity.Build(id=self.Parameters.browser_build_noalice)

        # additional reference to compatibility
        res["browser_build"] = res.get("browser_build_alice", None) or res.get("browser_build_noalice")
        return res

    def get_builds_consistency_problems(self):
        consistency_problems = []
        builds = self.get_builds()
        browser_build_alice, browser_build_noalice = builds.get("browser_build_alice"), builds.get("browser_build_noalice")
        if browser_build_alice and browser_build_noalice:
            browser_build_alice_commit = self.clients.get_build_commit(browser_build_alice)
            browser_build_noalice_commit = self.clients.get_build_commit(browser_build_noalice)
            if browser_build_alice_commit is None or browser_build_noalice_commit is None:
                consistency_problems.append(
                    u'Не удалось определить commit для сборок <a href="{}">#{}</a> и <a href="{}">#{}</a>.'
                    u' Регрессия может быть не корректной'.format(
                        browser_build_alice.web_url, browser_build_alice.id,
                        browser_build_noalice.web_url, browser_build_noalice.id))
            elif browser_build_alice_commit != browser_build_noalice_commit:
                consistency_problems.append(
                    u'Сборки <a href="{}">#{}</a> и <a href="{}">#{}</a> собраны на разных коммитах.'
                    u' Регрессия может быть не корректной'.format(
                        browser_build_alice.web_url, browser_build_alice.id,
                        browser_build_noalice.web_url, browser_build_noalice.id))
        return consistency_problems

    def upload_distribution_files_to_s3(self, builds, test_suites):
        alice = set()
        if builds.get("browser_build_alice", None):
            alice = set(
                FrozenDict(distribution_type=_s.distribution_type,
                           distribution_name=_s.distribution_name,
                           build=builds.get("browser_build_alice").id) for _s in test_suites if _s.build_type == "alice")
        noalice = set()
        if builds.get("browser_build_noalice", None):
            noalice = set(
                FrozenDict(distribution_type=_s.distribution_type,
                           distribution_name=_s.distribution_name,
                           build=builds.get("browser_build_noalice").id) for _s in test_suites if _s.build_type == "noalice")

        with self.memoize_stage.launch_upload_tasks:
            tasks = [
                BrowserUploadBrandedBuildToS3(
                    self,
                    distribution_type=item['distribution_type'],
                    distribution_name=item['distribution_name'],
                    build_id=item["build"],
                    fake_build_id=None,
                    assessors_artifacts=ASSESSORS_ARTIFACTS,
                    assessors_artifacts_fake=ASSESSORS_ARTIFACTS
                ).enqueue() for item in list(alice) + list(noalice)
            ]
            raise sdk2.WaitTask(
                tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True
            )
        children = self.find(BrowserUploadBrandedBuildToS3)
        result = {}
        if children:
            if any(child.status != Status.SUCCESS for child in children):
                raise TaskFailure('Failed to upload builds to s3')

            result = dict()
            alice_build_id = builds.get("browser_build_alice").id if "browser_build_alice" in builds else None
            for child in children:
                result.setdefault(
                    "alice" if child.Context.build_id == alice_build_id else "noalice",
                    {}).setdefault(
                        child.Context.distribution_type,
                        {})[child.Context.distribution_name] = child.Context.uploaded_files
        return result

    def create_distribution_assessors_tasks(self, assessors_runs, assessors_tickets):
        _assessors_runs = {}
        for project, project_suites in assessors_runs.iteritems():
            for suite, runs in project_suites.iteritems():
                for run in runs:
                    _assessors_runs.setdefault(
                        run.get_propetry("build"), {}).setdefault(
                        project, {}).setdefault(suite, []).append(run)
        tasks = []
        for build, build_runs in _assessors_runs.iteritems():
            tasks.append(self.create_assessors_task(assessors_runs=build_runs,
                                                    assessors_tickets=assessors_tickets,
                                                    test_stend=build))
        self.enqueue_asessor_tasks(tasks)

    def on_execute(self):

        self.validate_input()
        if not self.Context.deadline:
            self.Context.deadline = (datetime.datetime.now() + datetime.timedelta(
                hours=self.Parameters.regression_deadline)).date().isoformat()

        builds = self.get_builds()
        self.wait_builds([build for build in builds.values() if build])
        with self.memoize_stage.check_booking:
            self.check_booking()

        test_suites = self.regression_manager_class.load_suites_info(self.regression_config)

        self.Context.assessors_links = self.upload_distribution_files_to_s3(builds, test_suites)

        manager = self.regression_manager_class(
            regression_config=self.regression_config,
            task_parameters=self.Parameters,
            task_context=self.Context,
            oauth_vault=self.oauth_vault)
        regression_summary = manager.create_regression()
        if regression_summary["info_message"]:
            self.set_info(regression_summary["info_message"],
                          do_escape=False)
        main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs = regression_summary["runs_and_tickets"]

        if assessors_runs:
            self.create_distribution_assessors_tasks(assessors_runs, assessors_tickets)
        self.start_runs_monitoring(assessors_runs, manual_runs, manual_tickets, main_ticket)

        res = RegressionResult(main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs)
        self.update_tickets(res)
        self.create_regression_state_resource()
        self.schedule_statistics_task(self)
