# -*- coding: utf-8 -*-
import datetime

import sandbox
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.autotests.regression.assessors.BrowserUploadBrandedBuildToS3 import (
    BrowserUploadBrandedBuildToS3)
from sandbox.projects.browser.autotests.classes.regression_manager.webntp_desktop import WebntpDBroRegressionManager
from sandbox.projects.browser.autotests.regression_tasks.configs import RegressionConfigs
from sandbox.projects.browser.autotests.regression_tasks.BaseDesktopBrowserRegression import BaseDesktopBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import BaseBrowserRegression
from sandbox.projects.browser.autotests.regression_tasks.BaseBrowserRegression import RegressionResult
from sandbox.projects.browser.autotests_qa_tools.common import (
    BRANDED_BUILDS_PROJECT, WEBNTP_BUILDS_PROJECT, check_build, get_platform, TEAMCITY_URL, get_st_browser_version)

CONFIGS_MODULE = RegressionConfigs.webntp_dbro.value
BROWSER_BUILDS_PARAM_SCHEMA = {
    "$schema": "http://json-schema.org/draft-04/schema#",
    "type": "array",
    "minItems": 1,
    "uniqueItems": True,
    "items": {
        "type": "object",
        "additionalProperties": False,
        "required": [
            "build_id",
            "distribution_type",
            "distribution_name"
        ],
        "properties": {
            "build_id": {
                "anyOf": [
                    {"type": "string", "pattern": r"^\d+$"},
                    {"type": "integer", "minimum": 1}
                ]
            },
            "distribution_type": {
                "type": "string",
                "minLength": 2
            },
            "distribution_name": {
                "type": "string",
                "minLength": 2
            }
        }
    }
}

ASSESSORS_ARTIFACTS = {
    'Win': {
        'build': ('FULL_{version}_{distribution_name}.exe', '{distribution_type}/{distribution_name}/Yandex.exe')
    },
    'Mac': {
        'build': ('FULL_{version}_{distribution_name}.dmg', '{distribution_type}/{distribution_name}/Yandex.dmg')
    },
    'Linux': {
        'build': ('FULL_{version}_{distribution_name}.deb',
                  '{distribution_type}/{distribution_name}/yandex-browser-beta_{version}-1_amd64.deb')
    }
}


class RunWebntpBrowserRegression(BaseDesktopBrowserRegression):

    STATE_RESOURCE_PATH = "regression_state_data"
    STATE_DATA_FILE_NAME = "regression_state.json"

    regression_manager_class = WebntpDBroRegressionManager
    configs_module = CONFIGS_MODULE

    class Context(BaseDesktopBrowserRegression.Context):

        correct_browser_builds = None

    def upload_files_to_s3(self, *args, **kwargs):
        raise NotImplementedError()

    @property
    def test_stend(self):
        raise NotImplementedError()

    @property
    def affected_version(self):
        raise NotImplementedError()

    def get_builds_consistency_problems(self):
        return []

    def check_input(self):
        build_problems = super(BaseDesktopBrowserRegression, self).check_input()
        from jsonschema import validate, ValidationError
        try:
            validate(self.Parameters.browser_builds, BROWSER_BUILDS_PARAM_SCHEMA)
        except ValidationError as e:
            build_problems.append(u'Невалидный JSON в параметре browser_builds: ' + e.message)
            return build_problems

        # save correct browsers_builds values 2 context
        self.convert_browsers_builds_parameter()
        # get builds
        try:
            ntp_builds = self.get_ntp_builds()
            browsers_builds = self.get_browsers_builds()
        except Exception as e:
            build_problems.append(u'Не удается получить информацию о сборках: ' + e.message)
            return build_problems

        # build types
        for _build in ntp_builds.values():
            build_problems.extend(check_build(_build, WEBNTP_BUILDS_PROJECT))
        for _build in browsers_builds.values():
            build_problems.extend(check_build(_build, BRANDED_BUILDS_PROJECT))

        # browser builds platforms
        # see https://st.yandex-team.ru/BYIN-11498#5fc6557cf482976350ec96ca
        _platforms = {_b.id: get_platform(_b) for _b in browsers_builds.values()}
        if len(set(_platforms.values())) != 1 or _platforms.values()[0] is None:
            _error = u"Сборки browser_builds имеют не поддерживаемые или различные платформы: "
            for _id, _platform in _platforms.iteritems():
                _error += u" {}:{}".format(_id, _platform)
            build_problems.append(_error)

        return build_problems

    class Parameters(BaseDesktopBrowserRegression.Parameters):
        # these parameters are overridden below, so that they would be hidden in ui,
        # if they should be loaded from config file
        distribution_type = None
        distribution_name = None
        skip_builds_consistency_check = None
        build_id = None
        old_build_id = None
        fake_build_id = None
        browser_tests_build_id = None
        enable_autotests = None

        regression_type = sdk2.parameters.String(
            'Regression type', required=True, choices=[(_, _) for _ in CONFIGS_MODULE.TEST_SUITE_CONFIGS + ['custom']],
            ui=sdk2.parameters.String.UI('select')
        )
        with regression_type.value['custom']:
            test_suites_override = BaseBrowserRegression.Parameters.test_suites()

        start_hitman_jobs_automatically = BaseDesktopBrowserRegression.Parameters.start_hitman_jobs_automatically(
            default=True)
        with sdk2.parameters.Group('WebNTP specific parameters') as distribution_dbro_params:

            external_ntp_build_id = sdk2.parameters.Integer(
                'ExternalNtp build id',
                description=u'Тестируемая сборка ExternalNtp '
                            u'(<a href="{base}/project.html?projectId={project}">'
                            u'{project}</a>)'.format(base=TEAMCITY_URL, project=WEBNTP_BUILDS_PROJECT),
            )
            previous_external_ntp_build_id = sdk2.parameters.Integer(
                'Previous ExternalNtp build id',
                description=u'Предыдущая сборка ExternalNtp '
                            u'(<a href="{base}/project.html?projectId={project}">'
                            u'{project}</a>)'.format(base=TEAMCITY_URL, project=WEBNTP_BUILDS_PROJECT)
            )
            browser_builds = sdk2.parameters.JSON(
                # do_not_copy=True,
                'Browsers',
                description=u'Сборки для для тестирования (минимум одна) '
                            u'(<a href="{base}/project.html?projectId={project}">'
                            u'{project}</a>)'.format(base=TEAMCITY_URL, project=BRANDED_BUILDS_PROJECT),
                default=[
                    {
                        "build_id": 0,
                        "distribution_type": "",
                        "distribution_name": ""
                    },
                    {
                        "build_id": 0,
                        "distribution_type": "",
                        "distribution_name": ""
                    },
                    {
                        "build_id": 0,
                        "distribution_type": "",
                        "distribution_name": ""
                    }])
            asessor_version_title = sdk2.parameters.String(
                'Asessor version title',
                default="",
                description=u'Если не пуст добавится в заголовок версии и асессорского рана вместо типа регрессии',
            )

    def on_enqueue(self):
        with self.memoize_stage.add_params_from_config:
            if self.Parameters.regression_type != 'custom':
                self.Parameters.test_suites_override = {}

    @sandbox.common.utils.singleton
    def get_builds(self):
        return {}

    @sandbox.common.utils.singleton
    def get_ntp_builds(self):
        res = {}
        if self.Parameters.external_ntp_build_id:
            res["external_ntp_build_id"] = self.clients.teamcity.Build(id=self.Parameters.external_ntp_build_id)
        if self.Parameters.previous_external_ntp_build_id:
            res["previous_external_ntp_build_id"] = self.clients.teamcity.Build(id=self.Parameters.previous_external_ntp_build_id)
        return res

    @sandbox.common.utils.singleton
    def get_browsers_builds(self):
        res = {}
        for item in self.Context.correct_browser_builds:
            res[item["build_id"]] = self.clients.teamcity.Build(id=item["build_id"])
        return res

    def upload_browsers_files_to_s3(self):

        with self.memoize_stage.launch_upload_tasks:
            tasks = [
                BrowserUploadBrandedBuildToS3(
                    self,
                    distribution_type=item['distribution_type'],
                    distribution_name=item['distribution_name'],
                    build_id=item["build_id"],
                    fake_build_id=None,
                    assessors_artifacts=ASSESSORS_ARTIFACTS,
                    assessors_artifacts_fake=ASSESSORS_ARTIFACTS
                ).enqueue() for item in self.Context.correct_browser_builds
            ]
            raise sdk2.WaitTask(
                tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True
            )
        children = self.find(BrowserUploadBrandedBuildToS3)
        result = {}
        if children:
            if any(child.status != Status.SUCCESS for child in children):
                raise TaskFailure('Failed to upload builds to s3')

            result = dict()
            for child in children:
                result.setdefault(
                    child.Context.distribution_type,
                    {}).setdefault(
                        child.Context.distribution_name, {})[child.Context.build_id] = child.Context.uploaded_files
            return result
        return result

    @sandbox.common.utils.singleton
    def _find_build(self, s3_build_href):

        def _find_build_id(s3_build_href):
            for distribution_names in self.Context.assessors_links.values():
                for builds in distribution_names.values():
                    for build_id, build_value in builds.iteritems():
                        if build_value["build"] == s3_build_href:
                            return build_id
            return None

        build_id = _find_build_id(s3_build_href)
        if build_id is None or build_id not in self.get_browsers_builds():
            raise RuntimeError(u"Не удалось найти сборку {}".format(s3_build_href))
        return self.get_browsers_builds()[build_id]

    @sandbox.common.utils.singleton
    def get_requester_code(self, s3_build_href):
        return self._find_build(s3_build_href).last_changes[0].version

    @sandbox.common.utils.singleton
    def get_affected_version(self, s3_build_href):
        return get_st_browser_version(self._find_build(s3_build_href))

    def create_webntp_assessors_tasks(self, assessors_runs, assessors_tickets):
        _assessors_runs = {}
        for project, project_suites in assessors_runs.iteritems():
            for suite, runs in project_suites.iteritems():
                for run in runs:
                    _assessors_runs.setdefault(
                        run.get_propetry("build"), {}).setdefault(
                        project, {}).setdefault(suite, []).append(run)
        tasks = []
        for build, build_runs in _assessors_runs.iteritems():
            tasks.append(self.create_assessors_task(assessors_runs=build_runs,
                                                    assessors_tickets=assessors_tickets,
                                                    test_stend=build,
                                                    requester_code=self.get_requester_code(build),
                                                    affected_version=self.get_affected_version(build)))
        self.enqueue_asessor_tasks(tasks)

    @sandbox.common.utils.singleton
    def convert_browsers_builds_parameter(self):
        self.Context.correct_browser_builds = []
        for item in self.Parameters.browser_builds:
            self.Context.correct_browser_builds.append(
                {
                    "build_id": int(item["build_id"]),
                    "distribution_type": item["distribution_type"],
                    "distribution_name": item["distribution_name"]
                }
            )

    def on_execute(self):

        self.validate_input()
        if not self.Context.deadline:
            self.Context.deadline = (datetime.datetime.now() + datetime.timedelta(
                hours=self.Parameters.regression_deadline)).date().isoformat()

        # builds = self.get_builds()
        self.wait_builds([build for build in self.get_ntp_builds().values() if build] +
                         [build for build in self.get_browsers_builds().values() if build])

        with self.memoize_stage.check_booking:
            self.check_booking()

        self.Context.assessors_links = self.upload_browsers_files_to_s3()

        manager = self.regression_manager_class(
            regression_config=self.regression_config,
            task_parameters=self.Parameters,
            task_context=self.Context,
            oauth_vault=self.oauth_vault)
        regression_summary = manager.create_regression()
        if regression_summary["info_message"]:
            self.set_info(regression_summary["info_message"],
                          do_escape=False)
        main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs = regression_summary["runs_and_tickets"]

        if assessors_runs:
            self.create_webntp_assessors_tasks(assessors_runs, assessors_tickets)
        self.start_runs_monitoring(assessors_runs, manual_runs, manual_tickets, main_ticket)

        res = RegressionResult(main_ticket, manual_tickets, manual_runs, assessors_tickets, assessors_runs)
        self.update_tickets(res)
        manager.publish_diff_tickets()
        self.create_regression_state_resource()
        self.schedule_statistics_task(self)
