# -*- coding: utf-8 -*-

from copy import deepcopy

import sandbox
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import notification as ctn
import sandbox.common.types.client as ctc
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.autotests.classes.testing_builds import AutotestsBuilds, Builder
from sandbox.projects.browser.autotests.classes.ya_clients import YaClients
from sandbox.projects.browser.autotests.regression_tasks.RunAndroidBrowserRegression import RunAndroidBrowserRegression
from sandbox.projects.browser.autotests_qa_tools.common import ROBOT_BRO_QA_INFRA_TOKEN_VAULT
from sandbox.sandboxsdk.environments import PipEnvironment


COMMON_REGRESSION_PARAMS = {
    'main_regression_ticket_assignee': 'nik-isaev',
    'old_build_id': 0,
    'booking_id': 0,
    'check_booking': False,
    'max_run_duration': 60,
    'deadline': 24,
    'start_hitman_jobs_automatically': False,
    'enable_autotests': True,
    'diff_type': 'disabled'
}

LAUNCH_CONFIGS = {
    'android_browser_alpha': {
        'regression_params': {
            'description': '{build_id} alpha',
            'version': '{build_id} alpha',
            'regression_type': 'Android_Bro_Alpha.yaml',
            'hitman_process_id': 'testing_mbrowser_android_auto'
        },
        'required_builds': {
            'browser_tests_build_id': AutotestsBuilds.android.value
        },
        'regression_type': RunAndroidBrowserRegression
    },
    'android_browser_beta': {
        'regression_params': {
            'description': '{build_id} beta',
            'version': '{build_id} beta',
            'regression_type': 'Android_Bro_Beta.yaml',
            'hitman_process_id': 'testing_mbrowser_android_auto'
        },
        'required_builds': {
            'browser_tests_build_id': AutotestsBuilds.android.value
        },
        'regression_type': RunAndroidBrowserRegression
    },
    'android_browser_stable': {
        'regression_params': {
            'description': '{build_id} stable',
            'version': '{build_id} stable',
            'regression_type': 'Android_Bro_rc_autotests.yaml',
            'hitman_process_id': 'testing_mbrowser_android_auto'
        },
        'required_builds': {
            'browser_tests_build_id': AutotestsBuilds.android.value
        },
        'regression_type': RunAndroidBrowserRegression
    }
}


class RunBuildRegression(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        cores = 1
        client_tags = ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==3.0.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        notifications = (
            sdk2.Notification(
                [Status.Group.FINISH, Status.Group.BREAK],
                ['browser-autotests-bots'],
                ctn.Transport.EMAIL
            )
        )
        teamcity_build_id = sdk2.parameters.Integer('Teamcity build id')
        application_build_id = sdk2.parameters.Integer('Application teamcity build id')
        application_type = sdk2.parameters.String('Application type')
        chennel = sdk2.parameters.String('Distribution chennel')
        regression_assignee = sdk2.parameters.String('Regression assignee')

    class Context(sdk2.Context):
        required_builds = {}

    @property
    @sandbox.common.utils.singleton
    def clients(self):
        return YaClients(sdk2.Vault.data(ROBOT_BRO_QA_INFRA_TOKEN_VAULT))

    @property
    @sandbox.common.utils.singleton
    def launch_conf(self):
        conf_name = "{}_{}".format(self.Parameters.application_type, self.Parameters.chennel)
        if conf_name not in LAUNCH_CONFIGS:
            self.set_info(u'Ошибка. Неизвестная регрессия "{}"'.format(conf_name))
            raise TaskFailure()

        conf = LAUNCH_CONFIGS[conf_name]
        regression_params = deepcopy(COMMON_REGRESSION_PARAMS)
        regression_params.update(conf.get('regression_params', {}))
        conf['regression_params'] = regression_params

        # update version and description
        for key, value in conf.get('regression_params').iteritems():
            if isinstance(value, str) and '{build_id}' in value:
                conf['regression_params'][key] = value.format(build_id=self.Parameters.application_build_id)

        conf['regression_params']['build_id'] = self.Parameters.application_build_id
        conf['regression_params']['builds'] = [
            self.clients.teamcity.Build(id=self.Parameters.application_build_id).web_url
        ]
        if self.Parameters.regression_assignee:
            conf['regression_params']['main_regression_ticket_assignee'] = self.Parameters.regression_assignee

        conf['regression_params']['tags'] = [self.Parameters.application_type, self.Parameters.chennel]
        conf['regression_params']['summary_notification_type'] = conf_name
        return conf

    @property
    @sandbox.common.utils.singleton
    def browser_build_revision(self):
        return self.clients.get_build_revision(self.Parameters.application_build_id)

    def on_execute(self):
        # enqueue required builds
        builder = Builder(self.clients)

        for name, _build_template in self.launch_conf['required_builds'].iteritems():
            _build = builder.run_build_template(
                template=_build_template,
                branch=self.browser_build_revision[0],
                commit=self.browser_build_revision[1],
                params={},
                tags=[]
            )
            self.Context.required_builds[name] = _build.id
            self.set_info(u'Запущена teamcity-сборка: {}: <a href="{}">#{}</a>'.format(name, _build.web_url, _build.id),
                          do_escape=False)

        # enqueue regression
        regression_params = self.launch_conf['regression_params']
        for build_name, build__id in self.Context.required_builds.iteritems():
            regression_params[build_name] = build__id

        regression_task = self.launch_conf['regression_type'](
            self,
            **regression_params
        ).enqueue()
        self.set_info(
            u'Запуск регрессии: <a href={}>#{}</a>'.format(get_task_link(regression_task.id), regression_task.id),
            do_escape=False
        )
