# -*- coding: utf-8 -*-
import re
import json
import os

from sandbox import common
from sandbox.projects.browser.autotests_qa_tools.common import (
    FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO,
    BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO)

PRIORITIES = [
    'Blocker',
    'Critical',
    'Normal',
    'Low',
    'Minor',
]
TESTS_DUMP_FILENAME = '_tests_dump.json'


def get_owners(bitbucket_client, project, repo, commit, files):
    owners = bitbucket_client.get_owners(project, repo, commit, list(files))
    res = {_file: [] for _file in files}
    for _owner in owners:
        for _file in _owner["files"]:
            res[_file].append(_owner["owner"]["emailAddress"].split("@")[0])
    return res


class Test(object):

    '''
    allure-report/index.html#/xunit/dbb8f943cb4c80ad/24b1967c681c7675
    allure-report/index.html#/xunit/suite/uid
    '''
    tested_application = 'browser'

    def __init__(self, uid, data_path, tested_builds, report_url,
                 test_version, ya_resources):
        self.uid = uid
        self._data_path = data_path
        self.tested_builds = tested_builds
        self._base_report_url = report_url
        self.test_version = test_version
        self.ya_resources = ya_resources
        self._owners = []

    @common.utils.singleton_property
    def _test_data(self):
        with open(os.path.join(self._data_path,
                  '{}-testcase.json'.format(self.uid))) as f:
            return json.load(f)

    # TODO cache
    def get_label(self, label_name):
        result = None
        for item in self._test_data.get('labels', []):
            if item['name'] == label_name:
                return item['value']
        return result

    # TODO cache
    def get_attachment(self, attachment_name):
        result = None
        for item in self._test_data.get('attachments', []):
            if item['title'] == attachment_name:
                attach = item
                attach_path = os.path.join(self._data_path,
                                           attach['source'])
                if attach['type'] == 'application/json':
                    with open(attach_path) as f:
                        return json.load(f)
                elif attach['type'] == 'text/plain':
                    with open(attach_path) as f:
                        return f.read()
                else:
                    return attach_path
        return result

    @common.utils.singleton_property
    def full_name(self):
        return self._test_data.get('name', '').decode('string_escape').decode('utf-8')

    @common.utils.singleton_property
    def name(self):
        if self.test_type == 'binary':
            name_groups = re.findall(r'\[([^\[\]]+)\]', self.full_name)
            return name_groups[0].split(' ')[1]
        return self.full_name

    @common.utils.singleton_property
    def suite(self):
        return self.get_label('suite-name')

    @common.utils.singleton_property
    def test_type(self):
        if self.suite == 'tests_binary.test_binary':
            return 'binary'
        else:
            return 'python'

    @common.utils.singleton_property
    def filepath(self):
        if self.test_type == 'binary':
            filepath = self.get_label("source")
        else:
            filepath = '{}.py'.format('/'.join(self.suite.split('.')))
        return filepath

    @property
    def sources(self):
        return [self.filepath]

    @property
    def owners(self):
        return self._owners

    @owners.setter
    def owners(self, value):
        self._owners = value

    @common.utils.singleton_property
    def feature(self):
        return self.get_label('feature')

    @common.utils.singleton_property
    def component(self):
        #  TODO: TestPalm interaction
        return self.get_label('feature')

    @common.utils.singleton_property
    def duration(self):
        milliseconds = self._test_data.get('time', {}).get('duration')
        return milliseconds

    @common.utils.singleton_property
    def start_time(self):
        timestamp = self._test_data.get('time', {}).get('start', -1)
        return timestamp

    @common.utils.singleton_property
    def stop_time(self):
        timestamp = self._test_data.get('time', {}).get('stop', -1)
        return timestamp

    @common.utils.singleton_property
    def platform(self):
        result = re.search(r'\[([^\[\]]+bit)\]', self.full_name)
        if result:
            return result.groups()[0]
        else:
            return None

    @common.utils.singleton_property
    def testpalm_case(self):
        if not self.test_id:
            return None
        _case_id = re.split(r'-(\d+)$', self.test_id)
        return self.ya_resources.testpalm.get_case(_case_id[0], _case_id[1], include_fields=('id', 'attributes'))

    @common.utils.singleton_property
    def priority(self):
        case = self.testpalm_case
        result = case.priority if case and case.priority else ['Normal']
        return sorted(result,
                      key=self.priority_sort)[0]

    @staticmethod
    def priority_sort(value):
        if value in PRIORITIES:
            return PRIORITIES.index(value)
        else:
            return len(PRIORITIES)

    @common.utils.singleton_property
    def test_id(self):
        report_test_id = self._test_data.get('testId')
        if report_test_id:
            return report_test_id['name'].split('/')[-1]
        else:
            return None

    @common.utils.singleton_property
    def status(self):
        return self._test_data.get('status')

    @common.utils.singleton_property
    def is_passed(self):
        return bool(self.status == 'PASSED')

    @common.utils.singleton_property
    def is_failure(self):
        return self.status in ['FAILED', 'BROKEN']

    @common.utils.singleton_property
    def failure(self):
        return self._test_data.get('failure')

    @common.utils.singleton_property
    def failure_message(self):
        if not self.is_failure:
            return ''
        return self._test_data.get('failure', {}).get('message', '')

    @common.utils.singleton_property
    def test_log(self):
        if self.test_type == 'binary':
            log = str(self.get_attachment('Full text log'))
            try:
                return log.decode('cp1251')
            except Exception:
                return log.decode('utf-8')
        else:
            if not self.is_failure:
                return 'The test has passed!'
            else:
                return self.failure['stackTrace']

    @common.utils.singleton_property
    def test_environment_parameters(self):
        return self.get_attachment("Environment parameters") or {}

    # @ TODO убрать
    @common.utils.singleton_property
    def dump(self):
        return {
            "full_name": self.full_name,
            "message": self.failure['message'],
            "owners": self.owners,
            "story": self.get_label("story"),
            "failure": self.failure
        }


class AllureReport(object):

    tested_application = 'browser'

    def __init__(self, data_path, ya_resources):
        self._data_path = data_path
        self.ya_resources = ya_resources

    @common.utils.singleton_property
    def _features_data(self):
        with open(os.path.join(self._data_path, 'behaviors.json')) as f:
            return json.load(f)

    @common.utils.singleton_property
    def _xunit_data(self):
        with open(os.path.join(self._data_path, 'xunit.json')) as f:
            return json.load(f)

    @common.utils.singleton_property
    def _environment_data(self):
        with open(os.path.join(self._data_path, 'environment.json')) as f:
            return json.load(f)

    # TODO cache
    def get_environment_parameter(self, parameter_name):
        result = None
        for item in self._environment_data.get('parameter', []):
            if item['name'] == parameter_name:
                return item['value']
        return result

    @common.utils.singleton_property
    def tests(self):
        result = []
        for suite in self._xunit_data['testSuites']:
            for test in suite['testCases']:
                result.append(Test(test['uid'],
                                   self._data_path,
                                   self.tested_builds,
                                   self.report_url,
                                   self.get_environment_parameter(
                                       'Tests Version'),
                                   self.ya_resources))
        return result

    def get_case_tests(self, case_id, platform):
        return [test for test in self.tests if
                test.test_id == case_id and test.platform == platform]

    @common.utils.singleton_property
    def failures(self):
        return filter(lambda x: x.is_failure, self.tests)

    @common.utils.singleton_property
    def tested_builds(self):
        return {
            'yabrowser': self.get_environment_parameter('yabrowser'),
            'fakebuild': self.get_environment_parameter('fakebuild'),
            'binary_tests': self.get_environment_parameter('isolate_urls'),
        }

    @common.utils.singleton_property
    def sandbox_task(self):
        return self.get_environment_parameter('Sandbox task')

    @common.utils.singleton_property
    def report_url(self):
        return ''

    def dump_tests(self, target_path=None):

        target_path = target_path or self._data_path
        tests_dump = []
        for _t in self.tests:
            tests_dump.append(
                {
                    "name": _t.name,
                    "type": _t.test_type,
                    "platform": _t.platform,
                    "status": _t.status,
                    "test_id": _t.test_id,
                    "feature": _t.feature,
                    "failure_message": _t.failure_message,
                    "log": _t.test_log[-300:] if not _t.is_passed else ""
                })
        dump_file = os.path.join(target_path, TESTS_DUMP_FILENAME)
        if os.path.exists(dump_file):
            os.remove(dump_file)
        with open(dump_file, 'w') as _f:
            json.dump(
                {
                    "sandbox_task": self.sandbox_task,
                    "tests": tests_dump
                }, _f, indent=4)
        return dump_file

    def get_tests_owners(self, tests_list=None, branch="master"):
        if tests_list is None:
            tests_list = self.tests

        binary_tests = [test for test in tests_list if test.test_type == 'binary']
        python_tests = [test for test in tests_list if test.test_type == 'python']

        if python_tests:
            framework_files = set([
                test.filepath for test in python_tests
            ])
            framework_commit = self.ya_resources.bitbucket.get_latest_commit(
                FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO, branch)
            framework_owners = get_owners(
                self.ya_resources.bitbucket,
                FRAMEWORK_BITBUCKET_PROJECT, FRAMEWORK_BITBUCKET_REPO,
                framework_commit, framework_files)

            for item in python_tests:
                item.owners = framework_owners.get(item.filepath, [])

        if binary_tests:
            browser_files = set([
                test.filepath for test in binary_tests
            ])
            browser_commit = self.ya_resources.bitbucket.get_latest_commit(
                BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO, branch)
            browser_owners = get_owners(
                self.ya_resources.bitbucket,
                BROWSER_BITBUCKET_PROJECT, BROWSER_BITBUCKET_REPO,
                browser_commit, browser_files)

            for item in binary_tests:
                item.owners = browser_owners.get(item.filepath, [])
