# -*- coding: utf-8 -*-
import copy
import jinja2
import logging
import os
import re

from sandbox.projects.browser.autotests_qa_tools.classes.autoduty.issue import (
    BinaryTestsFailedIssue, PythonTestIssue, TICKET_JINJA_PACKAGE_PATH,
    PythonInstallUpdateTestIssue, PythonInstallUpdateTestIssueInfra, PythonLegacyTestIssue,
    AndroidTestsFailedIssue, IosTestsFailedIssue)
from sandbox.projects.browser.autotests_qa_tools.classes.allure_parser import AllureReport
from sandbox.projects.browser.autotests_qa_tools.classes.autotests_result import BtrReport
from sandbox.projects.browser.autotests_qa_tools.classes.ya_clients import YaClients


SUPPORTED_ISSUES = [
    BinaryTestsFailedIssue,
    PythonTestIssue,
    PythonInstallUpdateTestIssue,
    PythonInstallUpdateTestIssueInfra,
    PythonLegacyTestIssue,
    AndroidTestsFailedIssue,
    IosTestsFailedIssue,
]

NEW_ISSUES_ASSIGNEE = 'nik-isaev'
NEW_ISSUES_FOLLOWERS = ['nik-isaev']
DRAFT_ISSUE_TAG = 'fix_autotests_draft'
DRAFT_ISSUE_NEEDS_APPROVED_TAG = 'NOT_APPROVED'
DRAFT_SEPARATOR = u"----"
PR_ISSUE_QUEUE = "BYIN"
logger = logging.getLogger(__file__)


class FailuresAnalyzer(object):

    report_cls = BtrReport

    def __init__(self, oauth_token, task_link=None):
        self.clients = YaClients(oauth_token)
        self.task_link = task_link

    def get_failures(self, report_paths, cases_priority_filter=None):
        reports = [self.report_cls(_r, self.clients) for _r in report_paths]
        result = []
        for _r in reports:
            _r.get_tests_owners(_r.failures)

            report_failures = _r.failures
            if cases_priority_filter:
                report_failures = filter(lambda x: x.priority in cases_priority_filter, report_failures)
            result += report_failures

        return result

    def get_issues_templates(self, reports_paths, issue_types_names=None, cases_priority_filter=None, tests_names_filter=None):

        issue_types = filter(lambda x: x.__name__ in set(issue_types_names),
                             SUPPORTED_ISSUES)
        fail_tests = self.get_failures(reports_paths, cases_priority_filter)

        if tests_names_filter:
            fail_tests = filter(lambda x: x.name in tests_names_filter, fail_tests)

        issues = self.distribute_tests_by_issues(fail_tests, issue_types)
        return issues

    def create_main_ticket(self, assignee=NEW_ISSUES_ASSIGNEE):
        issue_dict = {
            'description': u"Завести тикеты на падающие автотесты",
            'assignee': assignee,
            'queue': PR_ISSUE_QUEUE,
            'summary': u"Завести тикеты на падающие автотесты",
            'affectedVersions': ['Dev'],
            'fixVersions': ['Dev'],
            'priority': 'critical',
            'components': ['Autotests']
        }
        issue_obj = self.clients.startrek.issues.create(**issue_dict)
        return issue_obj['key']

    def create_issues(self, issues, assignee=NEW_ISSUES_ASSIGNEE, parent=None):
        result = {}

        issue_description_head_template = jinja2.Environment(
            loader=jinja2.PackageLoader(TICKET_JINJA_PACKAGE_PATH,
                                        package_path='templates')).get_template("issue_draft_head.jinja")

        for issue in issues:
            issue_head = issue_description_head_template.render(
                assignee=issue.startrek_template['assignee'],
                followers=issue.startrek_template['followers'],
                task_link=self.task_link,
                draft_separator=DRAFT_SEPARATOR
            )

            issue_dict = copy.deepcopy(issue.startrek_template)
            if parent:
                issue_dict['parent'] = parent
            issue_dict['assignee'] = assignee
            issue_dict['followers'] = NEW_ISSUES_FOLLOWERS
            issue_dict['description'] = u'{}\n{}'.format(issue_head, issue_dict['description'])
            issue_dict.setdefault('tags', []).extend([DRAFT_ISSUE_NEEDS_APPROVED_TAG, DRAFT_ISSUE_TAG])
            issue_obj = self.clients.startrek.issues.create(**issue_dict)
            logger.info("TICKET CREATED: {}/{}".format(self.clients.st_base_url,
                                                       issue_obj['key']))

            # add attachments
            for _name, _data in issue.attachments.iteritems():
                attach_file_path = os.path.join("{}_{}".format(issue_obj.key, _name))
                with open(attach_file_path, "wb") as _f:
                    _f.write(_data)
                issue_obj.attachments.create(attach_file_path)

            result[issue_obj['key']] = {
                'assignee': issue.assignee,
                'type': issue.test_type,
                'followers': issue.followers,
                'tags': issue.tags,
                'blacklists_targets': list(issue.blacklists_targets),
                'tests': [[_t.test_id, _t.platform, _t.full_name] for _t in issue.tests]
            }
        return result

    @staticmethod
    def create_issue_template(test, issue_types):

        result = None
        for issue_type in issue_types:
            if issue_type.check_type(test):
                result = issue_type(test.ya_resources)
        return result

    @classmethod
    def distribute_tests_by_issues(cls, tests, issue_types):

        issues = []
        for test in tests:
            destination_issue = None
            for issue in issues:
                if issue.check_type(test) and issue.check_add_test(test):
                    destination_issue = issue
                    break
            if destination_issue is not None:
                destination_issue.add_test(test)
            else:
                new_issue = cls.create_issue_template(test, issue_types)
                if new_issue is not None:
                    new_issue.add_test(test)
                    issues.append(new_issue)

        issues.sort(key=lambda x: len(x.tests),
                    reverse=True)
        return issues

    def get_actual_issues_drafts(self, issues_data):

        return {issue_key: issue_data for issue_key, issue_data in issues_data.iteritems()
                if self.is_actual_issue_draft(issue_key)}

    def is_actual_issue_draft(self, issue_key):
        issue = self.clients.startrek.issues[issue_key]
        if not issue:
            return False

        return ((issue.status.key.lower() == 'open') and
                (DRAFT_ISSUE_TAG in issue.tags) and
                (DRAFT_ISSUE_NEEDS_APPROVED_TAG not in issue.tags))

    def send_issues_to_work(self, actual_issues):
        for issue_key, issue_data in actual_issues.iteritems():
            def _modifier(issue):
                # regex = re.compile('{}([\s\S]*){}'.format(DRAFT_SEPARATOR, DRAFT_SEPARATOR))
                regex = re.compile(r'^([\s\S]*?){}'.format(DRAFT_SEPARATOR))
                return {
                    'description': re.sub(regex, "", issue.description, 1),
                    'tags': issue_data['tags'],
                    'assignee': issue_data['assignee'],
                    'followers': issue_data['followers']
                }

            self.clients.safely_update_issue(
                issue_key,
                modifier=_modifier
            )


class AllureFailuresAnalyzer(FailuresAnalyzer):

    report_cls = AllureReport
